package com.applovin.sdk;

import android.content.Intent;

import java.util.Map;

/**
 * Service used for tracking various analytical events.
 */
public interface AppLovinEventService
{
    /**
     * Track an event without additional data.
     * <p>
     * Where applicable, it is suggested to use one of the predefined strings provided in {@link AppLovinEventTypes} for the event and parameter key.
     *
     * @param event A string representing the event to track.
     */
    void trackEvent(final String event);

    /**
     * Track an event with additional data.
     * <p>
     * Where applicable, it is suggested to use one of the predefined strings provided in {@link AppLovinEventTypes} for the event and parameter key.
     *
     * @param event  A string representing the event to track.
     * @param parameters A map containing key-value pairs further describing this event.
     */
    void trackEvent(final String event, final Map<String, ?> parameters);

    /**
     * Tracks an event with additional parameters and options.
     *
     * @param event      The name of the event to track (e.g., "page_view", "purchase").
     * @param parameters A map of parameter key-value pairs providing additional information about the event.
     * @param options    A map of additional options to be passed up with the event. Accepted options:
     *                   "dedupe_id" - A unique identifier used to uniquely identify and merge identical events occurring within a certain timeframe of each other—across both site-to-site and Axon Pixel integrations—so they’re only counted once.
     */
    void trackEvent(final String event, final Map<String, ?> parameters, final Map<String, Object> options);

    /**
     * Track an in app purchase.
     *
     * @param purchaseIntent Intent returned to you by Google Play.
     * @param parameters     A map containing key-value pairs further describing this event. You should provide, at a minimum, {@link AppLovinEventParameters#PRODUCT_IDENTIFIER}, {@link AppLovinEventParameters#REVENUE_AMOUNT} and {@link AppLovinEventParameters#REVENUE_CURRENCY}.
     */
    void trackInAppPurchase(final Intent purchaseIntent, final Map<String, String> parameters);

    /**
     * Track a checkout / standard purchase.
     *
     * @param transactionId An optional unique identifier for this transaction, as generated by you.
     * @param parameters    A map containing key-value pairs further describing this event. You should provide, at a minimum, {@link AppLovinEventParameters#PRODUCT_IDENTIFIER}, {@link AppLovinEventParameters#REVENUE_AMOUNT} and {@link AppLovinEventParameters#REVENUE_CURRENCY}.
     */
    void trackCheckout(final String transactionId, final Map<String, String> parameters);
}
