package com.applovin.adview;

import android.content.Context;
import android.view.ViewGroup;

import com.applovin.impl.sdk.CoreSdk;
import com.applovin.impl.sdk.Logger;
import com.applovin.impl.sdk.reward.IncentivizedAdController;
import com.applovin.sdk.AppLovinAd;
import com.applovin.sdk.AppLovinAdClickListener;
import com.applovin.sdk.AppLovinAdDisplayListener;
import com.applovin.sdk.AppLovinAdLoadListener;
import com.applovin.sdk.AppLovinAdRewardListener;
import com.applovin.sdk.AppLovinAdVideoPlaybackListener;
import com.applovin.sdk.AppLovinSdk;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.lifecycle.Lifecycle;

/**
 * Rewarded ads let you offer users in-app items—such as continued gameplay, virtual currency, or other rewards—in exchange for their engagement with ads.
 * Rewarded ads boost engagement because users receive a tangible benefit for their time.
 */
public class AppLovinIncentivizedInterstitial
{
    private final static String TAG = "AppLovinIncentivizedInterstitial";

    private final IncentivizedAdController controller;

    /**
     * Create a new instance of an {@link AppLovinIncentivizedInterstitial}.
     */
    public AppLovinIncentivizedInterstitial()
    {
        this( null, AppLovinSdk.getInstance( CoreSdk.getApplicationContext() ) );
    }

    /**
     * Create a new instance of an {@link AppLovinIncentivizedInterstitial} for a given zone identifier.
     *
     * @param zoneId Zone identifier to load ads for. May be null.
     */
    public AppLovinIncentivizedInterstitial(@Nullable final String zoneId)
    {
        this( zoneId, AppLovinSdk.getInstance( CoreSdk.getApplicationContext() ) );
    }

    /**
     * Set extra info to pass to the SDK.
     *
     * @param key   Parameter key. Must not be {@code null}.
     * @param value Parameter value. May be {@code null}.
     */
    public void setExtraInfo(@NonNull final String key, @Nullable final Object value)
    {
        // Check input
        if ( key == null ) throw new IllegalArgumentException( "No key specified" );

        controller.setExtraInfo( key, value );
    }

    /**
     * The zone identifier this incentivized ad was initialized with and is loading ads for, if any.
     */
    public String getZoneId()
    {
        return controller.getZoneId();
    }

    /**
     * Show a rewarded ad for the provided {@link AppLovinAd}.
     *
     * @param ad                    Ad to show. Must not be null.
     * @param adRewardListener      Reward listener to notify. Must not be null.
     * @param videoPlaybackListener Video playback listener to notify. Must not be null.
     * @param adDisplayListener     Ad display listener to notify, if desired. May be null.
     * @param adClickListener       Ad click listener to notify, if desired. May be null.
     */
    public void show(final AppLovinAd ad,
                     final AppLovinAdRewardListener adRewardListener,
                     final AppLovinAdVideoPlaybackListener videoPlaybackListener,
                     final AppLovinAdDisplayListener adDisplayListener,
                     final AppLovinAdClickListener adClickListener)
    {
        controller.show( ad, CoreSdk.getApplicationContext(), adRewardListener, videoPlaybackListener, adDisplayListener, adClickListener );
    }

    /**
     * Show a rewarded ad for the provided {@link AppLovinAd} in a container view.
     *
     * @param ad                    The ad to render into this incentivized ad. Must not be null.
     * @param containerView         ViewGroup used to show the ad in. Must not be null.
     * @param lifecycle             Lifecycle object to manage ad lifecycle events in container views. Must not be null.
     * @param adRewardListener      Reward listener to notify. Must not be null.
     * @param videoPlaybackListener Video playback listener to notify. Must not be null.
     * @param adDisplayListener     Ad display listener to notify, if desired. May be null.
     * @param adClickListener       Ad click listener to notify, if desired. May be null.
     */
    public void show(final AppLovinAd ad,
                     final ViewGroup containerView,
                     final Lifecycle lifecycle,
                     final AppLovinAdRewardListener adRewardListener,
                     final AppLovinAdVideoPlaybackListener videoPlaybackListener,
                     final AppLovinAdDisplayListener adDisplayListener,
                     final AppLovinAdClickListener adClickListener)
    {
        controller.show( ad, containerView, lifecycle, CoreSdk.getApplicationContext(), adRewardListener, videoPlaybackListener, adDisplayListener, adClickListener );
    }

    /**
     * @deprecated This API has been deprecated and will be removed in a future release. Please use {@link #AppLovinIncentivizedInterstitial()} instead.
     */
    @Deprecated
    public AppLovinIncentivizedInterstitial(final Context context)
    {
        this( AppLovinSdk.getInstance( context ) );
    }

    /**
     * @deprecated This API has been deprecated and will be removed in a future release. Please use {@link #AppLovinIncentivizedInterstitial()} instead.
     */
    @Deprecated
    public AppLovinIncentivizedInterstitial(final AppLovinSdk sdk)
    {
        this( null, sdk );
    }

    /**
     * @deprecated This API has been deprecated and will be removed in a future release. Please use {@link #AppLovinIncentivizedInterstitial(String)} instead.
     */
    @Deprecated
    public AppLovinIncentivizedInterstitial(final String zoneId, final AppLovinSdk sdk)
    {
        if ( sdk == null ) throw new IllegalArgumentException( "No sdk specified" );

        controller = new IncentivizedAdController( zoneId, sdk );
    }

    /**
     * @deprecated This API has been deprecated and will be removed in a future release. Please use {@link #AppLovinIncentivizedInterstitial()} instead.
     */
    @Deprecated
    public static AppLovinIncentivizedInterstitial create(final Context context)
    {
        return create( AppLovinSdk.getInstance( context ) );
    }

    /**
     * @deprecated This API has been deprecated and will be removed in a future release. Please use {@link #AppLovinIncentivizedInterstitial()} instead.
     */
    @Deprecated
    public static AppLovinIncentivizedInterstitial create(final AppLovinSdk sdk)
    {
        return create( null, sdk );
    }

    /**
     * @deprecated This API has been deprecated and will be removed in a future release. Please use {@link #AppLovinIncentivizedInterstitial(String)} instead.
     */
    @Deprecated
    public static AppLovinIncentivizedInterstitial create(final String zoneId, final AppLovinSdk sdk)
    {
        return new AppLovinIncentivizedInterstitial( zoneId, sdk );
    }


    /**
     * @deprecated This API has been deprecated and will be removed in a future release. Please use {@link #show(AppLovinAd, AppLovinAdRewardListener, AppLovinAdVideoPlaybackListener, AppLovinAdDisplayListener, AppLovinAdClickListener)} instead.
     */
    @Deprecated
    public void show(final Context context,
                     final AppLovinAdRewardListener adRewardListener,
                     final AppLovinAdVideoPlaybackListener videoPlaybackListener,
                     final AppLovinAdDisplayListener adDisplayListener)
    {
        show( context, adRewardListener, videoPlaybackListener, adDisplayListener, null );
    }

    /**
     * @deprecated This API has been deprecated and will be removed in a future release. Please use {@link #show(AppLovinAd, AppLovinAdRewardListener, AppLovinAdVideoPlaybackListener, AppLovinAdDisplayListener, AppLovinAdClickListener)} instead.
     */
    @Deprecated
    public void show(final Context context,
                     final AppLovinAdRewardListener adRewardListener,
                     final AppLovinAdVideoPlaybackListener videoPlaybackListener,
                     final AppLovinAdDisplayListener adDisplayListener,
                     final AppLovinAdClickListener adClickListener)
    {
        show( null, context, adRewardListener, videoPlaybackListener, adDisplayListener, adClickListener );
    }

    /**
     * @deprecated This API has been deprecated and will be removed in a future release. Please use {@link #show(AppLovinAd, AppLovinAdRewardListener, AppLovinAdVideoPlaybackListener, AppLovinAdDisplayListener, AppLovinAdClickListener)} instead.
     */
    @Deprecated
    public void show(final AppLovinAd ad,
                     final Context context,
                     final AppLovinAdRewardListener adRewardListener,
                     final AppLovinAdVideoPlaybackListener videoPlaybackListener,
                     final AppLovinAdDisplayListener adDisplayListener,
                     final AppLovinAdClickListener adClickListener)
    {
        controller.show( ad, context, adRewardListener, videoPlaybackListener, adDisplayListener, adClickListener );
    }

    /**
     * @deprecated This API has been deprecated and will be removed in a future release. Please use {@link #show(AppLovinAd, ViewGroup, Lifecycle, AppLovinAdRewardListener, AppLovinAdVideoPlaybackListener, AppLovinAdDisplayListener, AppLovinAdClickListener)} instead.
     */
    public void show(final AppLovinAd ad,
                     final ViewGroup containerView,
                     final Lifecycle lifecycle,
                     final Context context,
                     final AppLovinAdRewardListener adRewardListener,
                     final AppLovinAdVideoPlaybackListener videoPlaybackListener,
                     final AppLovinAdDisplayListener adDisplayListener,
                     final AppLovinAdClickListener adClickListener)
    {
        controller.show( ad, containerView, lifecycle, context, adRewardListener, videoPlaybackListener, adDisplayListener, adClickListener );
    }

    /**
     * @deprecated This API has been deprecated and will be removed in a future release.
     * Please use {@link com.applovin.sdk.AppLovinAdService#loadNextAdForZoneId(String, AppLovinAdLoadListener)} then
     * {@link #show(AppLovinAd, AppLovinAdRewardListener, AppLovinAdVideoPlaybackListener, AppLovinAdDisplayListener, AppLovinAdClickListener)} instead.
     */
    @Deprecated
    public void preload(final AppLovinAdLoadListener adLoadListener)
    {
        if ( adLoadListener == null )
        {
            Logger.userInfo( TAG, "AppLovinAdLoadListener was null when preloading incentivized interstitials; using a listener is highly recommended." );
        }
        controller.preload( adLoadListener );
    }

    /**
     * @deprecated This API has been deprecated and will be removed in a future release.
     */
    @Deprecated
    public boolean isAdReadyToDisplay()
    {
        return controller.isAdReady();
    }

    @Override
    public String toString()
    {
        return "AppLovinIncentivizedInterstitial{" +
                "zoneId='" + getZoneId() + "'" +
                '}';
    }
}
