package com.applovin.mediation.adapters;

import android.app.Activity;
import android.content.Context;

import com.applovin.mediation.MaxAdFormat;
import com.applovin.mediation.adapter.MaxAdViewAdapter;
import com.applovin.mediation.adapter.MaxAdapter;
import com.applovin.mediation.adapter.MaxAdapterError;
import com.applovin.mediation.adapter.MaxInterstitialAdapter;
import com.applovin.mediation.adapter.MaxRewardedAdapter;
import com.applovin.mediation.adapter.MaxSignalProvider;
import com.applovin.mediation.adapter.listeners.MaxAdViewAdapterListener;
import com.applovin.mediation.adapter.listeners.MaxInterstitialAdapterListener;
import com.applovin.mediation.adapter.listeners.MaxRewardedAdapterListener;
import com.applovin.mediation.adapter.listeners.MaxSignalCollectionListener;
import com.applovin.mediation.adapter.parameters.MaxAdapterInitializationParameters;
import com.applovin.mediation.adapter.parameters.MaxAdapterParameters;
import com.applovin.mediation.adapter.parameters.MaxAdapterResponseParameters;
import com.applovin.mediation.adapter.parameters.MaxAdapterSignalCollectionParameters;
import com.applovin.mediation.adapters.unityads.BuildConfig;
import com.applovin.sdk.AppLovinSdk;
import com.applovin.sdk.AppLovinSdkConfiguration;
import com.applovin.sdk.AppLovinSdkUtils;
import com.unity3d.ads.IUnityAdsInitializationListener;
import com.unity3d.ads.IUnityAdsListener;
import com.unity3d.ads.IUnityAdsLoadListener;
import com.unity3d.ads.UnityAds;
import com.unity3d.ads.UnityAdsLoadOptions;
import com.unity3d.ads.UnityAdsShowOptions;
import com.unity3d.ads.mediation.IUnityAdsExtendedListener;
import com.unity3d.ads.metadata.MediationMetaData;
import com.unity3d.ads.metadata.MetaData;
import com.unity3d.services.banners.BannerErrorCode;
import com.unity3d.services.banners.BannerErrorInfo;
import com.unity3d.services.banners.BannerView;
import com.unity3d.services.banners.UnityBannerSize;

import java.lang.reflect.Method;
import java.util.HashSet;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * This is a mediation adapterWrapper for the Unity Ads SDK
 */
public class UnityAdsMediationAdapter
        extends MediationAdapterBase
        implements MaxSignalProvider, MaxInterstitialAdapter, MaxRewardedAdapter, MaxAdViewAdapter
{
    private static final String KEY_GAME_ID                  = "game_id";
    private static final String KEY_SET_MEDIATION_IDENTIFIER = "set_mediation_identifier";

    private static final UnityAdsMediationAdapterRouter ROUTER;
    private static final AtomicBoolean                  INITIALIZED          = new AtomicBoolean();
    private static final Set<String>                    DISPLAYED_PLACEMENTS = new HashSet<>();
    private static       InitializationStatus           initializationStatus;

    private String     placementId;
    private String     biddingAdId;
    private BannerView bannerView;

    static
    {
        if ( AppLovinSdk.VERSION_CODE >= 90802 )
        {
            ROUTER = (UnityAdsMediationAdapterRouter) MediationAdapterRouter.getSharedInstance( UnityAdsMediationAdapterRouter.class );
        }
        else
        {
            ROUTER = new UnityAdsMediationAdapterRouter();
        }
    }

    // Explicit default constructor declaration
    public UnityAdsMediationAdapter(final AppLovinSdk sdk) { super( sdk ); }

    @Override
    public void initialize(final MaxAdapterInitializationParameters parameters, final Activity activity, final OnCompletionListener onCompletionListener)
    {
        updatePrivacyConsent( parameters, activity.getApplicationContext() );

        if ( INITIALIZED.compareAndSet( false, true ) )
        {
            final String gameId = parameters.getServerParameters().getString( KEY_GAME_ID, null );
            log( "Initializing UnityAds SDK with game id: " + gameId + "..." );
            initializationStatus = InitializationStatus.INITIALIZING;

            if ( parameters.getServerParameters().getBoolean( KEY_SET_MEDIATION_IDENTIFIER ) )
            {
                MediationMetaData mediationMetaData = new MediationMetaData( activity );
                mediationMetaData.setName( UnityAdsMediationAdapter.mediationTag() );
                mediationMetaData.setVersion( AppLovinSdk.VERSION );
                mediationMetaData.commit();
            }

            UnityAds.setDebugMode( parameters.isTesting() );
            UnityAds.addListener( ROUTER );

            UnityAds.initialize( activity.getApplicationContext(), gameId, parameters.isTesting(), true, new IUnityAdsInitializationListener()
            {
                @Override
                public void onInitializationComplete()
                {
                    log( "UnityAds SDK initialized" );
                    initializationStatus = InitializationStatus.INITIALIZED_SUCCESS;
                    onCompletionListener.onCompletion( InitializationStatus.INITIALIZED_SUCCESS, null );
                }

                @Override
                public void onInitializationFailed(final UnityAds.UnityAdsInitializationError error, final String message)
                {
                    log( "UnityAds SDK failed to initialize with error: " + message );
                    initializationStatus = InitializationStatus.INITIALIZED_FAILURE;
                    onCompletionListener.onCompletion( InitializationStatus.INITIALIZED_FAILURE, message );
                }
            } );
        }
        else
        {
            log( "UnityAds SDK already initialized" );
            onCompletionListener.onCompletion( initializationStatus, null );
        }
    }

    @Override
    public String getSdkVersion()
    {
        return UnityAds.getVersion();
    }

    @Override
    public String getAdapterVersion()
    {
        return BuildConfig.VERSION_NAME;
    }

    @Override
    public void onDestroy()
    {
        ROUTER.removeAdapter( this, placementId );

        if ( bannerView != null )
        {
            bannerView.destroy();
            bannerView = null;
        }
    }

    @Override
    public void collectSignal(final MaxAdapterSignalCollectionParameters parameters, final Activity activity, final MaxSignalCollectionListener callback)
    {
        log( "Collecting signal..." );

        String signal = UnityAds.getToken();
        callback.onSignalCollected( signal );
    }

    @Override
    public void loadInterstitialAd(final MaxAdapterResponseParameters parameters, final Activity activity, final MaxInterstitialAdapterListener listener)
    {
        placementId = parameters.getThirdPartyAdPlacementId();
        log( "Loading " + ( AppLovinSdkUtils.isValidString( parameters.getBidResponse() ) ? "bidding " : "" ) + "interstitial ad for placement \"" + placementId + "\"..." );

        updatePrivacyConsent( parameters, activity.getApplicationContext() );
        ROUTER.addInterstitialAdapter( this, listener, placementId );

        // Bidding ads need a random ID associated with each load and show
        if ( AppLovinSdkUtils.isValidString( parameters.getBidResponse() ) )
        {
            biddingAdId = UUID.randomUUID().toString();
        }

        // Note: Most load callbacks are also fired in onUnityAdsPlacementStateChanged() but not all, we need these callbacks to catch all load errors.
        UnityAds.load( placementId, createAdLoadOptions( parameters ), new IUnityAdsLoadListener()
        {
            @Override
            public void onUnityAdsAdLoaded(final String placementId)
            {
                log( "Interstitial placement \"" + placementId + "\" loaded" );
                listener.onInterstitialAdLoaded();
            }

            @Override
            public void onUnityAdsFailedToLoad(final String placementId)
            {
                MaxAdapterError loadError = getLoadError( placementId );
                log( "Interstitial placement \"" + placementId + "\" failed to load with error: " + loadError );
                listener.onInterstitialAdLoadFailed( loadError );
            }
        } );
    }

    @Override
    public void showInterstitialAd(final MaxAdapterResponseParameters parameters, final Activity activity, final MaxInterstitialAdapterListener listener)
    {
        log( "Showing interstitial ad for placement \"" + placementId + "\"..." );

        ROUTER.addShowingAdapter( this );

        if ( UnityAds.isReady( placementId ) )
        {
            UnityAds.show( activity, placementId, createAdShowOptions() );
        }
        else
        {
            log( "Interstitial ad not ready" );
            ROUTER.onAdDisplayFailed( placementId, MaxAdapterError.AD_NOT_READY );
        }
    }

    @Override
    public void loadRewardedAd(final MaxAdapterResponseParameters parameters, final Activity activity, final MaxRewardedAdapterListener listener)
    {
        placementId = parameters.getThirdPartyAdPlacementId();
        log( "Loading " + ( AppLovinSdkUtils.isValidString( parameters.getBidResponse() ) ? "bidding " : "" ) + "rewarded ad for placement \"" + placementId + "\"..." );

        updatePrivacyConsent( parameters, activity.getApplicationContext() );
        ROUTER.addRewardedAdapter( this, listener, placementId );

        // Bidding ads need a random ID associated with each load and show
        if ( AppLovinSdkUtils.isValidString( parameters.getBidResponse() ) )
        {
            biddingAdId = UUID.randomUUID().toString();
        }

        // Note: Most load callbacks are also fired in onUnityAdsPlacementStateChanged() but not all, we need these callbacks to catch all load errors.
        UnityAds.load( placementId, createAdLoadOptions( parameters ), new IUnityAdsLoadListener()
        {
            @Override
            public void onUnityAdsAdLoaded(final String placementId)
            {
                log( "Rewarded ad placement \"" + placementId + "\" loaded" );
                listener.onRewardedAdLoaded();
            }

            @Override
            public void onUnityAdsFailedToLoad(final String placementId)
            {
                MaxAdapterError loadError = getLoadError( placementId );
                log( "Rewarded ad placement \"" + placementId + "\" failed to load with error: " + loadError );
                listener.onRewardedAdLoadFailed( loadError );
            }
        } );
    }

    @Override
    public void showRewardedAd(final MaxAdapterResponseParameters parameters, final Activity activity, final MaxRewardedAdapterListener listener)
    {
        log( "Showing rewarded ad for placement \"" + placementId + "\"..." );

        ROUTER.addShowingAdapter( this );

        if ( UnityAds.isReady( placementId ) )
        {
            // Configure userReward from server.
            configureReward( parameters );

            UnityAds.show( activity, placementId, createAdShowOptions() );
        }
        else
        {
            log( "Rewarded ad not ready" );
            ROUTER.onAdDisplayFailed( placementId, MaxAdapterError.AD_NOT_READY );
        }
    }

    @Override
    public void loadAdViewAd(final MaxAdapterResponseParameters parameters, final MaxAdFormat adFormat, final Activity activity, final MaxAdViewAdapterListener listener)
    {
        String placementId = parameters.getThirdPartyAdPlacementId();
        log( "Loading banner ad for placement \"" + placementId + "\"..." );

        updatePrivacyConsent( parameters, activity.getApplicationContext() );

        bannerView = new BannerView( activity, placementId, toUnityBannerSize( adFormat ) );
        bannerView.setListener( new BannerView.IListener()
        {
            @Override
            public void onBannerLoaded(final BannerView bannerAdView)
            {
                log( "Banner ad loaded" );
                listener.onAdViewAdLoaded( bannerAdView );
            }

            @Override
            public void onBannerFailedToLoad(final BannerView bannerAdView, final BannerErrorInfo errorInfo)
            {
                log( "Banner ad failed to load" );
                listener.onAdViewAdLoadFailed( toMaxError( errorInfo ) );
            }

            @Override
            public void onBannerClick(final BannerView bannerAdView)
            {
                log( "Banner ad clicked" );
                listener.onAdViewAdClicked();
            }

            @Override
            public void onBannerLeftApplication(final BannerView bannerView)
            {
                log( "Banner ad left application" );
            }
        } );

        bannerView.load();
    }

    private UnityAdsLoadOptions createAdLoadOptions(final MaxAdapterResponseParameters parameters)
    {
        UnityAdsLoadOptions options = new UnityAdsLoadOptions();

        String bidResponse = parameters.getBidResponse();
        if ( AppLovinSdkUtils.isValidString( bidResponse ) && AppLovinSdkUtils.isValidString( biddingAdId ) )
        {
            options.setObjectId( biddingAdId );
            options.setAdMarkup( bidResponse );
        }

        return options;
    }

    private UnityAdsShowOptions createAdShowOptions()
    {
        UnityAdsShowOptions options = new UnityAdsShowOptions();
        if ( AppLovinSdkUtils.isValidString( biddingAdId ) )
        {
            options.setObjectId( biddingAdId );
        }

        return options;
    }

    private UnityBannerSize toUnityBannerSize(final MaxAdFormat adFormat)
    {
        if ( adFormat == MaxAdFormat.BANNER )
        {
            return new UnityBannerSize( 320, 50 );
        }
        else if ( adFormat == MaxAdFormat.LEADER )
        {
            return new UnityBannerSize( 728, 90 );
        }
        else
        {
            throw new IllegalArgumentException( "Unsupported ad format: " + adFormat );
        }
    }

    private MaxAdapterError getLoadError(final String placementId)
    {
        final UnityAds.PlacementState placementState = UnityAds.getPlacementState( placementId );

        // Current placement state is not available. SDK is not initialized or this placement has not been configured in Unity Ads admin tools.
        if ( placementState == UnityAds.PlacementState.NOT_AVAILABLE )
        {
            return new MaxAdapterError( MaxAdapterError.ERROR_CODE_INVALID_CONFIGURATION, "Invalid configuration", placementState.ordinal(), placementState.toString() );
        }
        // Placement is disabled. Placement can be enabled via Unity Ads admin tools.
        else if ( placementState == UnityAds.PlacementState.DISABLED )
        {
            return new MaxAdapterError( MaxAdapterError.ERROR_CODE_INVALID_CONFIGURATION, "Invalid configuration", placementState.ordinal(), placementState.toString() );
        }
        else if ( placementState == UnityAds.PlacementState.NO_FILL )
        {
            return new MaxAdapterError( MaxAdapterError.ERROR_CODE_NO_FILL, "No fill", placementState.ordinal(), placementState.toString() );
        }
        else if ( placementState == UnityAds.PlacementState.WAITING )
        {
            return new MaxAdapterError( MaxAdapterError.ERROR_CODE_AD_NOT_READY, "Ad not ready", placementState.ordinal(), placementState.toString() );
        }
        else
        {
            return MaxAdapterError.UNSPECIFIED;
        }
    }

    private static MaxAdapterError toMaxError(final BannerErrorInfo unityAdsBannerError)
    {
        final MaxAdapterError adapterError;

        if ( unityAdsBannerError.errorCode == BannerErrorCode.NO_FILL )
        {
            adapterError = MaxAdapterError.NO_FILL;
        }
        else if ( unityAdsBannerError.errorCode == BannerErrorCode.NATIVE_ERROR )
        {
            adapterError = MaxAdapterError.INTERNAL_ERROR;
        }
        else if ( unityAdsBannerError.errorCode == BannerErrorCode.WEBVIEW_ERROR )
        {
            adapterError = MaxAdapterError.WEBVIEW_ERROR;
        }
        else
        {
            adapterError = MaxAdapterError.UNSPECIFIED;
        }
        return new MaxAdapterError( adapterError.getErrorCode(), adapterError.getErrorMessage(), unityAdsBannerError.errorCode.ordinal(), unityAdsBannerError.errorMessage );
    }

    void updatePrivacyConsent(final MaxAdapterParameters parameters, final Context context)
    {
        MetaData privacyMetaData = new MetaData( context );

        if ( getWrappingSdk().getConfiguration().getConsentDialogState() == AppLovinSdkConfiguration.ConsentDialogState.APPLIES )
        {
            Boolean hasUserConsent = getPrivacySetting( "hasUserConsent", parameters );
            if ( hasUserConsent != null )
            {
                privacyMetaData.set( "gdpr.consent", hasUserConsent );
            }
        }

        if ( AppLovinSdk.VERSION_CODE >= 91100 )
        {
            Boolean isDoNotSell = getPrivacySetting( "isDoNotSell", parameters );
            if ( isDoNotSell != null ) // CCPA compliance - https://unityads.unity3d.com/help/legal/gdpr
            {
                privacyMetaData.set( "privacy.consent", !isDoNotSell ); // isDoNotSell means user has opted out and is equivalent to false.
            }
        }

        privacyMetaData.commit();
    }

    private Boolean getPrivacySetting(final String privacySetting, final MaxAdapterParameters parameters)
    {
        try
        {
            // Use reflection because compiled adapters have trouble fetching `boolean` from old SDKs and `Boolean` from new SDKs (above 9.14.0)
            Class<?> parametersClass = parameters.getClass();
            Method privacyMethod = parametersClass.getMethod( privacySetting );
            return (Boolean) privacyMethod.invoke( parameters );
        }
        catch ( Exception exception )
        {
            log( "Error getting privacy setting " + privacySetting + " with exception: ", exception );
            return ( AppLovinSdk.VERSION_CODE >= 9140000 ) ? null : false;
        }
    }

    private static class UnityAdsMediationAdapterRouter
            extends MediationAdapterRouter
            implements IUnityAdsListener, IUnityAdsExtendedListener
    {
        //TODO: marked for deletion, pending SDK change.
        void initialize(final MaxAdapterInitializationParameters parameters, final Activity activity, final MaxAdapter.OnCompletionListener onCompletionListener) { }

        @Override
        public void onUnityAdsReady(String placementId)
        {
            // Handled in {@link UnityAdsMediationAdapterRouter#onUnityAdsPlacementStateChanged(String, UnityAds.PlacementState, UnityAds.PlacementState)}
        }

        @Override
        public void onUnityAdsStart(String placementId)
        {
            onAdDisplayed( placementId );
            onRewardedAdVideoStarted( placementId );

            DISPLAYED_PLACEMENTS.add( placementId );
        }

        @Override
        public void onUnityAdsFinish(String placementId, UnityAds.FinishState result)
        {
            if ( result == UnityAds.FinishState.COMPLETED )
            {
                onRewardedAdVideoCompleted( placementId );
                onUserRewarded( placementId, getReward( placementId ) );
                onAdHidden( placementId );
            }
            else if ( result == UnityAds.FinishState.SKIPPED )
            {
                onRewardedAdVideoCompleted( placementId );

                if ( shouldAlwaysRewardUser( placementId ) )
                {
                    onUserRewarded( placementId, getReward( placementId ) );
                }

                onAdHidden( placementId );
            }
            else if ( result == UnityAds.FinishState.ERROR )
            {
                log( "UnityAds failed to finish ad for placement " + placementId );
                onAdDisplayFailed( placementId, MaxAdapterError.INTERNAL_ERROR );

                // This point may be reached if the ad fails to display OR if the ad is closed after experiencing an error (in which case we should fire onAdHidden).
                if ( DISPLAYED_PLACEMENTS.contains( placementId ) )
                {
                    onRewardedAdVideoCompleted( placementId );

                    if ( shouldAlwaysRewardUser( placementId ) )
                    {
                        onUserRewarded( placementId, getReward( placementId ) );
                    }

                    onAdHidden( placementId );
                }
            }

            DISPLAYED_PLACEMENTS.remove( placementId );
        }

        @Override
        public void onUnityAdsError(UnityAds.UnityAdsError error, String message)
        {
            // Note: there are a few things to note about this delegate method
            // 1) This delegate method is missing a placementIdentifier argument to pinpoint which ad load or show errored-out.
            // 2) UnityAdsError covers init, load, and show errors. MoPub treats them all as a load error.
            // 3) We could try to save the error, but unfortunately, this delegate method is called after some of the delegate methods implemented below.
            //    Could be a TBD feature to support sending UnityAdsError with the delegate.

            final String adapterError = error != null ? error.name() : "UNKNOWN";
            log( "UnityAds did error " + error.name() + " with message: " + message );
        }

        @Override
        public void onUnityAdsClick(String placementId)
        {
            onAdClicked( placementId );
        }

        @Override
        public void onUnityAdsPlacementStateChanged(String placementId, UnityAds.PlacementState oldState, UnityAds.PlacementState newState)
        {
            if ( newState == UnityAds.PlacementState.READY )
            {
                log( "Placement \"" + placementId + "\" is ready" );
                onAdLoaded( placementId );
            }
            else if ( newState == UnityAds.PlacementState.WAITING )
            {
                log( "Placement \"" + placementId + "\" is loading..." );
            }
            else if ( newState == UnityAds.PlacementState.NOT_AVAILABLE )
            {
                log( "Placement \"" + placementId + "\" is not available" );

                final MaxAdapterError error = new MaxAdapterError( MaxAdapterError.ERROR_CODE_INVALID_CONFIGURATION, "Invalid Configuration", newState.ordinal(), newState.toString() );
                onAdLoadFailed( placementId, error );
            }
            else if ( newState == UnityAds.PlacementState.DISABLED )
            {
                log( "Placement \"" + placementId + "\" is disabled - please check your Unity admin tools." );

                final MaxAdapterError error = new MaxAdapterError( MaxAdapterError.ERROR_CODE_INVALID_CONFIGURATION, "Invalid Configuration", newState.ordinal(), newState.toString() );
                onAdLoadFailed( placementId, error );
            }
            else if ( newState == UnityAds.PlacementState.NO_FILL )
            {
                log( "Placement \"" + placementId + "\" NO FILL'd" );
                onAdLoadFailed( placementId, MaxAdapterError.NO_FILL );
            }
        }
    }
}
