package com.cloud.hisavana.net.disklrucache.utils;

import android.text.TextUtils;
import android.util.Log;

import com.cloud.hisavana.net.HttpRequest;
import com.cloud.hisavana.net.disklrucache.ImageCacheURL;
import com.cloud.hisavana.net.disklrucache.impl.DiskCacheProvider;
import com.cloud.hisavana.net.disklrucache.impl.IDiskCache;
import com.cloud.hisavana.net.utils.StorageUtils;
import com.cloud.sdk.commonutil.util.CommonLogUtil;
import com.transsion.core.CoreUtil;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.Reader;
import java.io.StringWriter;
import java.nio.charset.Charset;

/**
 * 物料缓存工具类
 *
 * @author chenshijun
 */
public class DiskLruCacheUtil {
    public static final Charset US_ASCII = Charset.forName("US-ASCII");
    public static final Charset UTF_8 = Charset.forName("UTF-8");

    private static IDiskCache diskCache;

    private DiskLruCacheUtil() {
    }

    private static final char[] HEX_CHAR_ARRAY = "0123456789abcdef".toCharArray();
    /**
     * 32 bytes from sha-256 -> 64 hex chars.
     */
    private static final char[] SHA_256_CHARS = new char[64];

    /**
     * Returns the hex string of the given byte array representing a SHA256 hash.
     */
    public static String sha256BytesToHex(byte[] bytes) {
        synchronized (SHA_256_CHARS) {
            return bytesToHex(bytes, SHA_256_CHARS);
        }
    }

    /**
     * 将字节数组转换为十六进制字符串
     * http://stackoverflow.com/questions/9655181/convert-from-byte-array-to-hex-string-in-java
     */
    @SuppressWarnings("PMD.UseVarargs")
    private static String bytesToHex(byte[] bytes, char[] hexChars) {
        int v;
        for (int j = 0; j < bytes.length; j++) {
            v = bytes[j] & 0xFF;
            hexChars[j * 2] = HEX_CHAR_ARRAY[v >>> 4];
            hexChars[j * 2 + 1] = HEX_CHAR_ARRAY[v & 0x0F];
        }
        return new String(hexChars);
    }

    public static void closeQuietly(Closeable closeable) {
        if (closeable != null) {
            try {
                closeable.close();
            } catch (IOException ignored) {
                CommonLogUtil.netLog("closeQuietly --> " + Log.getStackTraceString(ignored));
            }
        }
    }

    public static String readFully(Reader reader) throws IOException {
        try {
            StringWriter writer = new StringWriter();
            char[] buffer = new char[1024];
            int count;
            while ((count = reader.read(buffer)) != -1) {
                writer.write(buffer, 0, count);
            }
            return writer.toString();
        } finally {
            reader.close();
        }
    }

    /**
     * Deletes the contents of {@code dir}. Throws an IOException if any file
     * could not be deleted, or if {@code dir} is not a readable directory.
     */
    public static void deleteContents(File dir) throws IOException {
        File[] files = dir.listFiles();
        if (files == null) {
            throw new IOException("not a readable directory: " + dir);
        }
        for (File file : files) {
            if (file.isDirectory()) {
                deleteContents(file);
            }
            if (!file.delete()) {
                throw new IOException("failed to delete file: " + file);
            }
        }
    }

    public static IDiskCache getDiskCache(boolean isOfflineMode) {
        if (CoreUtil.getContext() != null) {
            long maxSize;
            if (isOfflineMode) {
                maxSize = HttpRequest.INSTANCE.getOfflineCacheSize() <= 0 ? IDiskCache.Factory.DEFAULT_OFFLINE_CACHE_SIZE :
                        HttpRequest.INSTANCE.getOfflineCacheSize() * 1024 * 1024L;
            } else {
                maxSize = HttpRequest.INSTANCE.getCacheSize() <= 0 ? IDiskCache.Factory.DEFAULT_DISK_CACHE_SIZE :
                        HttpRequest.INSTANCE.getCacheSize() * 1024 * 1024L;
            }
            diskCache = new DiskCacheProvider().getDiskCache(maxSize, isOfflineMode);
        }
        return diskCache;
    }

    /**
     * 通过url生成缓存时的file路径
     */
    public static String getCacheFilePath(String url, boolean isOfflineAd) {
        if (TextUtils.isEmpty(url)) {
            return "";
        }
        //获取广告素材保存的路径
        File directory = isOfflineAd ? StorageUtils.getOfflineCacheDirectory(CoreUtil.getContext(), true)
                : StorageUtils.getCacheDirectory(CoreUtil.getContext());
        if (directory == null) {
            return "";
        }
        //根据url获取素材的文件名
        ImageCacheURL imageURL = new ImageCacheURL(url);
        //命名规则".0"参考代码：DiskLruCache内部类Entry的构造方法
        String fileName = SafeKeyUtils.getSafeKey(imageURL) + ".0";
        return directory + File.separator + fileName;
    }

}
