package com.cloud.hisavana.net;

import static com.cloud.hisavana.net.constants.HttpCode.FROM_DISK_FAIL;
import static com.cloud.hisavana.net.constants.HttpCode.HTTP_CODE_DIVIDER;
import static com.cloud.hisavana.net.constants.HttpCode.HTTP_IMAGE_CACHE;

import android.os.Looper;
import android.text.TextUtils;
import android.util.Log;

import com.cloud.hisavana.net.constants.HttpCode;
import com.cloud.hisavana.net.disklrucache.ImageCacheURL;
import com.cloud.hisavana.net.disklrucache.impl.IDiskCache;
import com.cloud.hisavana.net.disklrucache.utils.DiskLruCacheUtil;
import com.cloud.hisavana.net.impl.IHttpCallback;
import com.cloud.hisavana.net.interceptor.RequestInterceptor;
import com.cloud.hisavana.net.ssl.HttpsTrustManager;
import com.cloud.hisavana.net.utils.ByteBufferUtil;
import com.cloud.sdk.commonutil.util.CommonLogUtil;
import com.transsin.networkmonitor.MonitorFactory;

import java.io.File;
import java.io.IOException;
import java.net.ConnectException;
import java.net.NoRouteToHostException;
import java.net.ProtocolException;
import java.net.SocketTimeoutException;
import java.net.UnknownHostException;
import java.util.concurrent.TimeUnit;

import okhttp3.Call;
import okhttp3.Callback;
import okhttp3.Dispatcher;
import okhttp3.EventListener;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;

/**
 * OkHttpClient对象
 *
 * @author chenshijun
 */

public class CommonOkHttpClient {
    public static final String TAG = "ADSDK";

    /**
     * 超时时间
     */
    private static final int TIME_OUT = 30;
    /** 当前请求的的环境 */
    public static boolean isTestRequest = false;
    /** 网络调度器 */
    private static Dispatcher adDispatcher;
    /** 网络侦听器 */
    private static EventListener.Factory mediaFactory;
    /** body返回空的错误 */
    private static Exception ERROR_BODY_EMPTY  = new Exception("http response body is empty");

    private CommonOkHttpClient() {
        throw new IllegalStateException("Utility class");
    }

    /**
     * OkHttpClient实例
     */
    public static OkHttpClient getOkHttpClient() {
        return GeneralOkHttpClient.commonOkHttpClient;
    }

    /**
     * 设置网络调度器
     * @param dispatcher 调度器
     */
    public static void setOkHttpDispatcher(Dispatcher dispatcher) {
        setOkHttpClient(dispatcher, null);
    }

    /**
     * 设置网络调度器和网络侦听器
     * @param dispatcher 调度器
     * @param factory 侦听器
     */
    public static void setOkHttpClient(Dispatcher dispatcher, EventListener.Factory factory) {
        adDispatcher = dispatcher;
        mediaFactory = factory;
    }

    private static class GeneralOkHttpClient {

        private static OkHttpClient.Builder getBuilder() {
            OkHttpClient.Builder builder;
            if (isTestRequest) {
                builder = new OkHttpClient.Builder()
                        .sslSocketFactory(HttpsTrustManager.createSSLSocketFactory(), new HttpsTrustManager())
                        .hostnameVerifier(new HttpsTrustManager.TrustAllHostnameVerifier())
                        //为构建者设置超时时间
                        .connectTimeout(TIME_OUT, TimeUnit.SECONDS)
                        .readTimeout(TIME_OUT, TimeUnit.SECONDS)
                        .writeTimeout(TIME_OUT, TimeUnit.SECONDS)
                        //设置拦截器
                        .addInterceptor(new RequestInterceptor())
                        .dispatcher(adDispatcher != null ? adDispatcher : new Dispatcher())
                        //网络监控
                        .eventListenerFactory(new MonitorFactory(mediaFactory, 0, isTestRequest));
            } else {
                builder = new OkHttpClient.Builder()
                        //为构建者设置超时时间
                        .connectTimeout(TIME_OUT, TimeUnit.SECONDS)
                        .readTimeout(TIME_OUT, TimeUnit.SECONDS)
                        .writeTimeout(TIME_OUT, TimeUnit.SECONDS)
                        //设置拦截器
                        .addInterceptor(new RequestInterceptor())
                        .dispatcher(adDispatcher != null ? adDispatcher : new Dispatcher())
                        //网络监控
                        .eventListenerFactory(new MonitorFactory(mediaFactory, 0, isTestRequest));
            }
            CommonLogUtil.netLog("getBuilder: isTestRequest = " + isTestRequest);

            return builder;
        }

        private static final OkHttpClient commonOkHttpClient = getBuilder().build();
    }

    /**
     * GET请求
     */
    public static Call get(final Request request, final IHttpCallback callback) {
        try {
            Call call = getOkHttpClient().newCall(request);
            call.enqueue(new Callback() {
                @Override
                public void onFailure(Call call, IOException e) {
                    try {
                        if (callback == null) {
                            return;
                        }
                        if (e instanceof SocketTimeoutException) {
                            callback.sendFailureMessage(HttpCode.HTTP_TIMEOUT_CODE, null, e);
                        } else if (e instanceof UnknownHostException) {
                            callback.sendFailureMessage(HttpCode.HTTP_UNKNOWN_HOST_ERROR_CODE, null, e);
                        } else if (e instanceof NoRouteToHostException) {
                            callback.sendFailureMessage(HttpCode.HTTP_NO_ROUTE_TO_HOST_ERROR_CODE, null, e);
                        } else if (e instanceof ProtocolException) {
                            callback.sendFailureMessage(HttpCode.HTTP_PROTOCOL_ERROR_CODE, null, e);
                        } else if (e instanceof ConnectException) {
                            callback.sendFailureMessage(HttpCode.HTTP_Request_ERROR_CODE, null, e);
                        } else {
                            callback.sendFailureMessage(HttpCode.HTTP_DEFAULT_RESPONSE_CODE, null, e);
                        }
                    } catch (Exception exception) {
                        if (callback != null) {
                            callback.sendFailureMessage(HttpCode.HTTP_DEFAULT_RESPONSE_CODE, null, e);
                        }
                        printException(exception);
                    }
                }

                @Override
                public void onResponse(Call call, Response response) throws IOException {
                    try {
                        if (callback == null) {
                            return;
                        }

                        if (response != null) {
                            byte[] body = response.body().bytes();
                            if (body != null && body.length > 0) {
                                callback.sendResponseMessage(response.code(), body, response.headers());
                            } else {
                                callback.sendFailureMessage(HttpCode.ERROR_RESPONSE_EMPTY, null, ERROR_BODY_EMPTY);
                            }
                        } else {
                            callback.sendFailureMessage(HttpCode.ERROR_RESPONSE_EMPTY, null, ERROR_BODY_EMPTY);
                        }
                    } catch (Exception e) {
                        if (response != null) {
                            callback.sendFailureMessage(response.code(), null, e);
                        }
                    }
                }
            });
            return call;
        } catch (Exception exception) {
            if (callback != null) {
                callback.sendFailureMessage(HttpCode.HTTP_DEFAULT_RESPONSE_CODE, null, exception);
            }
            printException(exception);
            return null;
        }
    }


    /**
     * POST请求
     */
    public static Call post(final Request request, final IHttpCallback callback) {
        if (callback != null) {
            callback.sendStartMessage();
        }
        try {
            Call call = getOkHttpClient().newCall(request);
            call.enqueue(new Callback() {
                @Override
                public void onFailure(Call call, IOException e) {
                    try {
                        if (callback == null) {
                            return;
                        }
                        if (e instanceof SocketTimeoutException) {
                            callback.sendFailureMessage(HttpCode.HTTP_TIMEOUT_CODE, null, e);
                        } else if (e instanceof UnknownHostException) {
                            callback.sendFailureMessage(HttpCode.HTTP_UNKNOWN_HOST_ERROR_CODE, null, e);
                        } else if (e instanceof NoRouteToHostException) {
                            callback.sendFailureMessage(HttpCode.HTTP_NO_ROUTE_TO_HOST_ERROR_CODE, null, e);
                        } else if (e instanceof ProtocolException) {
                            callback.sendFailureMessage(HttpCode.HTTP_PROTOCOL_ERROR_CODE, null, e);
                        } else if (e instanceof ConnectException) {
                            callback.sendFailureMessage(HttpCode.HTTP_Request_ERROR_CODE, null, e);
                        } else {
                            callback.sendFailureMessage(HttpCode.HTTP_DEFAULT_RESPONSE_CODE, null, e);
                        }
                    } catch (Exception exception) {
                        if (callback != null) {
                            callback.sendFailureMessage(HttpCode.HTTP_DEFAULT_RESPONSE_CODE, null, e);
                        }
                        printException(exception);
                    }
                }

                @Override
                public void onResponse(Call call, Response response) throws IOException {
                    try {
                        if (callback == null) {
                            return;
                        }

                        if (response != null) {
                            byte[] body = response.body().bytes();
                            if (body != null && body.length > 0) {
                                callback.sendResponseMessage(response.code(), body, response.headers());
                            } else {
                                callback.sendFailureMessage(HttpCode.ERROR_RESPONSE_EMPTY, null, ERROR_BODY_EMPTY);
                            }
                        } else {
                            callback.sendFailureMessage(HttpCode.ERROR_RESPONSE_EMPTY, null, ERROR_BODY_EMPTY);
                        }
                    } catch (Exception e) {
                        if (response != null) {
                            callback.sendFailureMessage(response.code(), null, e);
                        }
                    }
                }
            });
            return call;
        } catch (Exception exception) {
            if (callback != null) {
                callback.sendFailureMessage(HttpCode.HTTP_DEFAULT_RESPONSE_CODE, null, exception);
            }
            printException(exception);
            return null;
        }
    }

    /**
     * 请求下载图片
     */
    public static Call downLoadImg(final RequestParams params,
                                   final Request request,
                                   final IHttpCallback callback) {
        if (Log.isLoggable(CommonOkHttpClient.TAG, Log.DEBUG)) {
            CommonLogUtil.netLog("downLoadImg okhttp------> run Current Looper:" +
                    (Looper.myLooper() == Looper.getMainLooper() ? "Main Looper" : "thread Looper"));
        }
        //是否是从本地文件中读取 下载文件数据
        boolean isFromCache = false;

        try {
            if (params.isUseCache()) {
                IDiskCache diskCache = DiskLruCacheUtil.getDiskCache(params.isSwitchOffLineCache());
                if (diskCache != null) {
                    File imageFile = diskCache.get(new ImageCacheURL(request.url().toString()));
                    if (imageFile != null && imageFile.exists() && imageFile.length() > 0) {
                        isFromCache = true;
                        if (Log.isLoggable(CommonOkHttpClient.TAG, Log.DEBUG)) {
                            CommonLogUtil.netLog("image url:" + request.url());
                            CommonLogUtil.netLog("image path:" + imageFile.getPath());
                        }

                        if (callback != null) {
                            if (params.isFetchFilePath()) {
                                callback.sendResponseMessage(HTTP_IMAGE_CACHE,
                                        ByteBufferUtil.toBytes(ByteBufferUtil.fromFile(imageFile)), imageFile.getPath());
                            } else {
                                callback.sendResponseMessage(HTTP_IMAGE_CACHE,
                                        ByteBufferUtil.toBytes(ByteBufferUtil.fromFile(imageFile)));
                            }
                            return null;
                        }
                    } else {
                        return realDownLoadImage(params, request, callback);
                    }
                }
            }
        } catch (IOException e) {
            CommonLogUtil.netLog("downLoadImg: --> " + Log.getStackTraceString(e));
            if (callback != null && isFromCache) {
                callback.sendFailureMessage(FROM_DISK_FAIL, null, e);
            }
        }

        return realDownLoadImage(params, request, callback);
    }

    /**
     * 真正的发起网络请求
     */
    private static Call realDownLoadImage(final RequestParams params, final Request request, final IHttpCallback callback) {
        try {
            Call call = getOkHttpClient().newCall(request);
            call.enqueue(new Callback() {
                @Override
                public void onFailure(Call call, IOException e) {
                    try {
                        if (callback == null) {
                            return;
                        }

                        if (e instanceof SocketTimeoutException) {
                            callback.sendFailureMessage(HttpCode.HTTP_TIMEOUT_CODE, null, e);
                        } else if (e instanceof UnknownHostException) {
                            callback.sendFailureMessage(HttpCode.HTTP_UNKNOWN_HOST_ERROR_CODE, null, e);
                        } else if (e instanceof NoRouteToHostException) {
                            callback.sendFailureMessage(HttpCode.HTTP_NO_ROUTE_TO_HOST_ERROR_CODE, null, e);
                        } else if (e instanceof ProtocolException) {
                            callback.sendFailureMessage(HttpCode.HTTP_PROTOCOL_ERROR_CODE, null, e);
                        } else if (e instanceof ConnectException) {
                            callback.sendFailureMessage(HttpCode.HTTP_Request_ERROR_CODE, null, e);
                        } else {
                            callback.sendFailureMessage(HttpCode.HTTP_DEFAULT_RESPONSE_CODE, null, e);
                        }
                    } catch (Exception exception) {
                        if (callback != null) {
                            callback.sendFailureMessage(HttpCode.HTTP_DEFAULT_RESPONSE_CODE, null, e);
                        }
                        printException(exception);
                    }
                }

                @Override
                public void onResponse(Call call, Response response) {
                    try {
                        if (callback == null) {
                            return;
                        }
                        if (response != null) {
                            byte[] body = response.body().bytes();
                            if (body != null && body.length > 0) {
                                if (!params.isFetchFilePath()) {
                                    callback.sendResponseMessage(response.code(), body, response.headers());
                                }
                                String filePath = cacheToDisk(params, request.url().toString(), response.code(), body);
                                if (Log.isLoggable(CommonOkHttpClient.TAG, Log.DEBUG)) {
                                    CommonLogUtil.netLog("onResponse: url = " + request.url() + " , code = " + response.code() + " , filePath = " + filePath);
                                }
                                if (params.isFetchFilePath()) {
                                    if (isFilePathExist(filePath)) {
                                        callback.sendResponseMessage(response.code(), body, filePath);
                                    } else {
                                        callback.sendFailureMessage(HttpCode.ERROR_CACHE_FILE_TO_DISK, null, new Exception("fail to cache file to disk"));
                                    }
                                }
                            } else {
                                callback.sendFailureMessage(HttpCode.ERROR_RESPONSE_EMPTY, null, ERROR_BODY_EMPTY);
                            }
                        } else {
                            callback.sendFailureMessage(HttpCode.ERROR_RESPONSE_EMPTY, null, ERROR_BODY_EMPTY);
                        }
                    } catch (Exception e) {
                        if (callback != null && response != null) {
                            callback.sendFailureMessage(response.code(), null, e);
                        }
                    }
                    if (callback != null) {
                        callback.sendFinishMessage();
                    }
                }
            });
            return call;
        } catch (Exception exception) {
            if (callback != null) {
                callback.sendFailureMessage(HttpCode.HTTP_DEFAULT_RESPONSE_CODE, null, exception);
            }
            printException(exception);
            return null;
        }
    }

    private static String cacheToDisk(RequestParams params, String url, int code, byte[] buf) {
        String filePath = "";
        try {
            IDiskCache diskCache = DiskLruCacheUtil.getDiskCache(params.isSwitchOffLineCache());
            //返回码小于300才缓存
            if (params.isUseCache() && diskCache != null && code < HTTP_CODE_DIVIDER) {
                filePath = diskCache.put(new ImageCacheURL(url), buf);
                if (Log.isLoggable(CommonOkHttpClient.TAG, Log.DEBUG)) {
                    CommonLogUtil.netLog("url :" + url + " , write cache finish.  filePath = " + filePath);
                }
            }
        } catch (Exception e) {
            CommonLogUtil.netLog("cacheToDisk is failure, " + Log.getStackTraceString(e));
        }
        return filePath;
    }

    private static boolean isFilePathExist(String filePath) {
        if (TextUtils.isEmpty(filePath)) {
            return false;
        }
        File file = new File(filePath);
        return file.exists() && file.length() > 0;
    }

    /**
     * 打印网络日志
     */
    private static void printException(Exception e) {
        if (Log.isLoggable("ADSDK", Log.DEBUG) && e != null) {
            CommonLogUtil.netLog("onFailure:  " + Log.getStackTraceString(e));
        }
    }

}
