package com.cloud.hisavana.net.disklrucache.impl;


import android.text.TextUtils;
import android.util.Log;

import com.cloud.hisavana.net.CommonOkHttpClient;
import com.cloud.hisavana.net.disklrucache.DiskLruCache;
import com.cloud.hisavana.net.disklrucache.ImageCacheURL;
import com.cloud.hisavana.net.disklrucache.utils.DiskLruCacheUtil;
import com.cloud.hisavana.net.disklrucache.utils.SafeKeyUtils;
import com.cloud.hisavana.net.utils.StorageUtils;
import com.cloud.sdk.commonutil.util.CommonLogUtil;
import com.transsion.core.CoreUtil;

import java.io.File;
import java.io.IOException;
import java.io.OutputStream;

/**
 * DiskLruCache操作类
 * @author chenshijun
 */
public class DiskCacheImpl implements IDiskCache {
    private static final int APP_VERSION = 1;
    private static final int VALUE_COUNT = 1;

    private DiskLruCache diskLruCache;
    private File directory;
    private final long maxSize;
    /** 是否使用离线缓存 */
    private final boolean isOfflineMode;

    protected DiskCacheImpl(File directory, long maxSize, boolean isOfflineMode) {
        this.directory = directory;
        this.maxSize = maxSize;
        this.isOfflineMode = isOfflineMode;
    }

    private DiskLruCache getDiskLruCache() throws IOException {
        DiskLruCache localDiskLruCache = diskLruCache;
        if (localDiskLruCache == null) {
            synchronized(DiskCacheImpl.class){
                localDiskLruCache = diskLruCache;
                if (localDiskLruCache == null) {
                    File cacheFile = isOfflineMode ? StorageUtils.getOfflineCacheDirectory(CoreUtil.getContext(),true)
                            : StorageUtils.getCacheDirectory(CoreUtil.getContext());
                    if (Log.isLoggable(CommonOkHttpClient.TAG, Log.DEBUG)) {
                        CommonLogUtil.netLog(" ----- getDiskLruCache: directory = " + directory + " , cacheFile = " + cacheFile);
                    }
                    directory = directory == null ? cacheFile : directory;
                    localDiskLruCache =  diskLruCache = DiskLruCache.open(directory, APP_VERSION, VALUE_COUNT, maxSize);
                }
            }
        }
        return localDiskLruCache;
    }

    @Override
    public File get(ImageCacheURL key) {
        return getImageFile(key);
    }

    protected File getImageFile(ImageCacheURL imageURL) {
        if (TextUtils.isEmpty(imageURL.getOriginUrl())) {
            return null;
        }
        //获取广告素材保存的路径
        File directory = isOfflineMode ? StorageUtils.getOfflineCacheDirectory(CoreUtil.getContext(), true)
                : StorageUtils.getCacheDirectory(CoreUtil.getContext());
        if (directory == null) {
            return null;
        }
        //命名规则".0"参考代码：DiskLruCache内部类Entry的构造方法
        String fileName = SafeKeyUtils.getSafeKey(imageURL) + ".0";
        return new File(directory + File.separator + fileName);
    }

    @Override
    public String put(ImageCacheURL key, byte[] buf) {
        String filePath = "";
        OutputStream outputStream = null;
        String safeKey = SafeKeyUtils.getSafeKey(key);
        try {
            final DiskLruCache.Snapshot currentValue = getDiskLruCache().get(safeKey);
            if (currentValue != null) {
                return null;
            }
            DiskLruCache.Editor editor = diskLruCache.edit(safeKey);
            if (editor == null) {
                throw new IllegalStateException("Had two simultaneous puts for: " + safeKey);
            }
            try {
                outputStream = editor.newOutputStream(0);
                outputStream.write(buf);
                outputStream.flush();
                editor.commit();
                diskLruCache.flush();
                filePath = DiskLruCacheUtil.getCacheFilePath(key.getOriginUrl(), isOfflineMode);
                if (Log.isLoggable(CommonOkHttpClient.TAG, Log.DEBUG)) {
                    CommonLogUtil.netLog("url :" + key.getOriginUrl() + "cacheToDisk: " + filePath);
                }
            } finally {
                editor.abortUnlessCommitted();
            }
        } catch (IOException e) {
            CommonLogUtil.netLog("put: fail" + filePath);
        } finally {
            DiskLruCacheUtil.closeQuietly(outputStream);
        }
        return filePath;
    }

    @Override
    public void delete(ImageCacheURL key) {
        String safeKey = SafeKeyUtils.getSafeKey(key);
        try {
            getDiskLruCache().remove(safeKey);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    @Override
    public void clear() {
        try {
            getDiskLruCache().delete();
            resetDiskCache();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    private synchronized void resetDiskCache() {
        diskLruCache = null;
    }
}
