package com.cloud.hisavana.net.impl;

import static com.cloud.hisavana.net.constants.HttpCode.HTTP_CODE_DIVIDER;

import android.os.Handler;
import android.os.Looper;
import android.os.Message;
import android.util.Log;


import com.cloud.hisavana.net.utils.CheckUtil;

import java.io.IOException;

import okhttp3.Headers;

public abstract class HttpCallbackImpl implements IHttpCallback {
    private static final String LOG_TAG = "HttpCallbackImpl";
    public static final String DEFAULT_CHARSET = "UTF-8";
    public static final String UTF8_BOM = "\uFEFF";
    protected static final int SUCCESS_MESSAGE = 0;
    protected static final int FAILURE_MESSAGE = 1;
    protected static final int START_MESSAGE = 2;
    protected static final int FINISH_MESSAGE = 3;
    protected static final int PAUSE_MESSAGE = 7;
    private Handler handler;
    private boolean useSyncMode;
    /** 是否将回调放到请求线程。 */
    private boolean usePoolThread;
    private Looper looper = null;

    public HttpCallbackImpl() {
        this(null);
    }

    public HttpCallbackImpl(Looper looper) {
        this(looper == null ? Looper.myLooper() : looper, false);
    }

    public HttpCallbackImpl(boolean isSwitchToRequestThread) {
        this(isSwitchToRequestThread ? null : Looper.myLooper(), isSwitchToRequestThread);
    }

    private HttpCallbackImpl(Looper looper, boolean isSwitchToRequestThread) {
        if (!isSwitchToRequestThread) {
            CheckUtil.asserts(looper != null, "use looper thread, must call Looper.prepare() first!");
            this.looper = looper;
            this.handler = new ResponderHandler(this, looper);
        } else {
            CheckUtil.asserts(looper == null, "use pool thread, looper should be null!");
            this.looper = null;
            this.handler = null;
        }
        this.usePoolThread = isSwitchToRequestThread;
    }

    @Override
    public void setUseSyncMode(boolean useSyncMode) {
        if (!useSyncMode && looper == null) {
            useSyncMode = true;
            Log.w(LOG_TAG, "Current thread has not called Looper.prepare(). Forcing synchronous mode.");
        }
        if (!useSyncMode && handler == null) {
            handler = new ResponderHandler(this, looper);
        } else if (useSyncMode && handler != null) {
            handler = null;
        }
        this.useSyncMode = useSyncMode;
    }

    @Override
    public boolean getUseSyncMode() {
        return this.useSyncMode;
    }

    @Override
    public boolean getUsePoolThread() {
        return this.usePoolThread;
    }

    @Override
    public void setUsePoolThread(boolean usePoolThread) {
        if (usePoolThread) {
            looper = null;
            handler = null;
        }
        this.usePoolThread = usePoolThread;
    }

    public abstract void onSuccess(int statusCode, byte[] responseBody);

    public void onSuccess(int statusCode, byte[] responseBody, String filePath) {

    }

    public void onResponseHeader(Headers responseHeaders) {
    }

    public abstract void onFailure(int statusCode, byte[] responseBody, Throwable error);

    @Override
    public void sendSuccessMessage(int statusCode, byte[] responseBody) {
        sendMessage(obtainMessage(SUCCESS_MESSAGE, new Object[]{statusCode, responseBody}));
    }

    @Override
    public void sendSuccessMessage(int statusCode, byte[] responseBody, String filePath) {
        sendMessage(obtainMessage(SUCCESS_MESSAGE, new Object[]{statusCode, responseBody, filePath}));
    }

    @Override
    public void sendSuccessMessage(int statusCode, byte[] responseBody, Headers responseHeaders) {
        sendMessage(obtainMessage(SUCCESS_MESSAGE, new Object[]{statusCode, responseBody, responseHeaders}));
    }

    @Override
    public void sendFailureMessage(int statusCode, byte[] responseBody, Throwable error) {
        sendMessage(obtainMessage(FAILURE_MESSAGE, new Object[]{statusCode, responseBody, error}));
    }

    @Override
    public void sendFailureMessage(int statusCode, byte[] responseBody, Throwable error, Headers responseHeaders) {
        sendMessage(obtainMessage(FAILURE_MESSAGE, new Object[]{statusCode, responseBody, error, responseHeaders}));
    }

    @Override
    public void sendStartMessage() {
        sendMessage(obtainMessage(START_MESSAGE, null));
    }

    @Override
    public void sendFinishMessage() {
        sendMessage(obtainMessage(FINISH_MESSAGE, null));
    }

    @Override
    public void sendPauseMessage() {
        sendMessage(obtainMessage(PAUSE_MESSAGE, null));
    }

    protected void postRunnable(Runnable runnable) {
        if (runnable != null) {
            if (getUseSyncMode() || handler == null) {
                runnable.run();
            } else {
                handler.post(runnable);
            }
        }
    }

    protected Message obtainMessage(int msgId, Object msgData) {
        return Message.obtain(handler, msgId, msgData);
    }

    protected void sendMessage(Message msg) {
        if (getUseSyncMode() || handler == null) {
            handleMessage(msg);
        } else if (!Thread.currentThread().isInterrupted() && handler != null) {
            handler.sendMessage(msg);
        }
    }

    protected void handleMessage(Message message) {
        Object[] response;
        try {
            switch (message.what) {
                case SUCCESS_MESSAGE:
                    response = (Object[]) message.obj;
                    if (response != null && response.length >= 2) {
                        if (response.length >= 3) {
                            if (response[2] instanceof String) {
                                onSuccess((Integer) response[0], (byte[]) response[1], (String) response[2]);
                            } else if (response[2] instanceof Headers) {
                                onResponseHeader((Headers) response[2]);
                                onSuccess((Integer) response[0], (byte[]) response[1]);
                            }
                        } else {
                            onSuccess((Integer) response[0], (byte[]) response[1]);
                        }
                    } else {
                        Log.e(LOG_TAG, "SUCCESS_MESSAGE didn't got enough params");
                    }
                    break;
                case FAILURE_MESSAGE:
                    response = (Object[]) message.obj;
                    if (response != null && response.length >= 3) {
                        if (response.length >= 4 && response[3] instanceof Headers) {
                            onResponseHeader((Headers) response[3]);
                        }
                        onFailure((Integer) response[0], (byte[]) response[1], (Throwable) response[2]);
                    } else {
                        Log.e(LOG_TAG, "FAILURE_MESSAGE didn't got enough params");
                    }
                    break;
                case START_MESSAGE:
                    onStart();
                    break;
                case FINISH_MESSAGE:
                    onFinish();
                    break;
                case PAUSE_MESSAGE:
                    onPause();
                    break;
                default:
                    Log.e(LOG_TAG, "UnKnown message");
                    break;
            }
        } catch (Throwable error) {
            onUserException(error);
        }
    }


    @Override
    public void sendResponseMessage(int code, byte[] responseBody) {
        if (!Thread.currentThread().isInterrupted()) {
            if (code >= HTTP_CODE_DIVIDER) {
                sendFailureMessage(code, responseBody, new IOException());
            } else {
                sendSuccessMessage(code, responseBody);
            }
        }
    }

    @Override
    public void sendResponseMessage(int code, byte[] responseBody, String filePath) {
        if (!Thread.currentThread().isInterrupted()) {
            if (code >= HTTP_CODE_DIVIDER) {
                sendFailureMessage(code, responseBody, new IOException());
            } else {
                sendSuccessMessage(code, responseBody, filePath);
            }
        }
    }

    @Override
    public void sendResponseMessage(int code, byte[] responseBody, Headers responseHeaders) {
        if (!Thread.currentThread().isInterrupted()) {
            if (code >= HTTP_CODE_DIVIDER) {
                sendFailureMessage(code, responseBody, new IOException(), responseHeaders);
            } else {
                sendSuccessMessage(code, responseBody, responseHeaders);
            }
        }
    }

    public void onStart() {
    }

    public void onFinish() {
    }

    public void onPause() {
        Log.d(LOG_TAG, "Request got onPaused");
    }

    public void onUserException(Throwable error) {
        Log.e(LOG_TAG, "User-space exception detected!", error);
        throw new RuntimeException(error);
    }

    private static class ResponderHandler extends Handler {
        private final HttpCallbackImpl httpCallback;

        ResponderHandler(HttpCallbackImpl callback, Looper looper) {
            super(looper);
            this.httpCallback = callback;
        }

        @Override
        public void handleMessage(Message msg) {
            httpCallback.handleMessage(msg);
        }
    }
}
