package com.cloud.hisavana.net.disklrucache.utils;

import static com.cloud.hisavana.net.HttpRequest.DEFAULT_MODE;
import static com.cloud.hisavana.net.HttpRequest.OFFLINE_MODE;
import static com.cloud.hisavana.net.HttpRequest.ONLINE_MODE;
import static com.cloud.hisavana.net.HttpRequest.VIDEO_MODE;

import android.text.TextUtils;
import android.util.Log;

import com.cloud.hisavana.net.HttpRequest;
import com.cloud.hisavana.net.disklrucache.ImageCacheURL;
import com.cloud.hisavana.net.disklrucache.impl.DiskCacheProvider;
import com.cloud.hisavana.net.disklrucache.impl.IDiskCache;
import com.cloud.hisavana.net.utils.StorageUtils;
import com.cloud.sdk.commonutil.util.CommonLogUtil;
import com.cloud.sdk.commonutil.util.HSCoreUtil;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.Reader;
import java.io.StringWriter;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;

/**
 * 物料缓存工具类
 *
 * @author chenshijun
 */
public class DiskLruCacheUtil {
    public static final Charset US_ASCII = StandardCharsets.US_ASCII;
    public static final Charset UTF_8 = StandardCharsets.UTF_8;

    private static IDiskCache onlineDiskCache;
    private static IDiskCache offlineDiskCache;
    private static IDiskCache videoDiskCache;
    private static IDiskCache defaultDiskCache;

    private DiskLruCacheUtil() {
    }

    private static final char[] HEX_CHAR_ARRAY = "0123456789abcdef".toCharArray();
    /**
     * 32 bytes from sha-256 -> 64 hex chars.
     */
    private static final char[] SHA_256_CHARS = new char[64];

    /**
     * Returns the hex string of the given byte array representing a SHA256 hash.
     */
    public static String sha256BytesToHex(byte[] bytes) {
        synchronized (SHA_256_CHARS) {
            return bytesToHex(bytes, SHA_256_CHARS);
        }
    }

    /**
     * 将字节数组转换为十六进制字符串
     * http://stackoverflow.com/questions/9655181/convert-from-byte-array-to-hex-string-in-java
     */
    @SuppressWarnings("PMD.UseVarargs")
    private static String bytesToHex(byte[] bytes, char[] hexChars) {
        int v;
        for (int j = 0; j < bytes.length; j++) {
            v = bytes[j] & 0xFF;
            hexChars[j * 2] = HEX_CHAR_ARRAY[v >>> 4];
            hexChars[j * 2 + 1] = HEX_CHAR_ARRAY[v & 0x0F];
        }
        return new String(hexChars);
    }

    public static void closeQuietly(Closeable closeable) {
        if (closeable != null) {
            try {
                closeable.close();
            } catch (IOException ignored) {
                CommonLogUtil.netLog("closeQuietly --> " + Log.getStackTraceString(ignored));
            }
        }
    }

    public static String readFully(Reader reader) throws IOException {
        try (StringWriter writer = new StringWriter()) {
            char[] buffer = new char[1024];
            int count;
            while ((count = reader.read(buffer)) != -1) {
                writer.write(buffer, 0, count);
            }
            return writer.toString();
        } catch (Exception e) {
            return "";
        } finally {
            if (reader != null) {
                reader.close();
            }
        }
    }

    /**
     * Deletes the contents of {@code dir}. Throws an IOException if any file
     * could not be deleted, or if {@code dir} is not a readable directory.
     */
    public static void deleteContents(File dir) throws IOException {
        File[] files = dir.listFiles();
        if (files == null) {
            throw new IOException("not a readable directory: " + dir);
        }
        for (File file : files) {
            if (file.isDirectory()) {
                deleteContents(file);
            }
            if (!file.delete()) {
                throw new IOException("failed to delete file: " + file);
            }
        }
    }

    public static IDiskCache getDiskCache(int cacheMode) {
        switch (cacheMode) {
            case HttpRequest.DEFAULT_MODE:
                return getDefaultDIskCache();
            case HttpRequest.VIDEO_MODE:
                return getVideoDiskCache();
            default:
                return getOnlineDiskCache();

        }
    }

    private static IDiskCache getOnlineDiskCache() {
        if (onlineDiskCache == null) {
            long maxSize = HttpRequest.INSTANCE.getCacheSize() <= 0 ? IDiskCache.Factory.DEFAULT_DISK_CACHE_SIZE : HttpRequest.INSTANCE.getCacheSize() * 1024 * 1024L;
            onlineDiskCache = new DiskCacheProvider().getDiskCache(maxSize, ONLINE_MODE);
        }
        return onlineDiskCache;
    }
    private static IDiskCache getOfflineDiskCache() {
        if (offlineDiskCache == null) {
            long maxSize = HttpRequest.INSTANCE.getOfflineCacheSize() <= 0 ? IDiskCache.Factory.DEFAULT_OFFLINE_CACHE_SIZE : HttpRequest.INSTANCE.getOfflineCacheSize() * 1024 * 1024L;
            offlineDiskCache = new DiskCacheProvider().getDiskCache(maxSize, OFFLINE_MODE);
        }
        return offlineDiskCache;
    }

    private static IDiskCache getVideoDiskCache() {
        if (videoDiskCache == null) {
            long maxSize = HttpRequest.INSTANCE.getVideoCacheSize() <= 0 ? IDiskCache.Factory.VIDEO_CACHE_SIZE : HttpRequest.INSTANCE.getVideoCacheSize() * 1024 * 1024L;
            videoDiskCache = new DiskCacheProvider().getDiskCache(maxSize, VIDEO_MODE);
        }
        return videoDiskCache;
    }

    private static IDiskCache getDefaultDIskCache() {
        if (defaultDiskCache == null) {
            long maxSize = HttpRequest.INSTANCE.getDefaultCacheSize() <= 0 ? IDiskCache.Factory.BUILT_IN_CACHE_SIZE : HttpRequest.INSTANCE.getDefaultCacheSize() * 1024 * 1024L;
            defaultDiskCache = new DiskCacheProvider().getDiskCache(maxSize, DEFAULT_MODE);
        }
        return defaultDiskCache;
    }
    /**
     * 通过url生成缓存时的file路径
     */
    public static String getCacheFilePath(String url, int cacheMode) {
        if (TextUtils.isEmpty(url)) {
            return "";
        }
        //获取广告素材保存的路径
        File directory =getCacheDirectory(cacheMode);
        if (directory == null) {
            return "";
        }
        //根据url获取素材的文件名
        ImageCacheURL imageURL = new ImageCacheURL(url);
        //命名规则".0"参考代码：DiskLruCache内部类Entry的构造方法
        String fileName = SafeKeyUtils.getSafeKey(imageURL) + ".0";
        return directory + File.separator + fileName;
    }

    public static File getCacheDirectory(int cacheMode){
        switch (cacheMode){
            case HttpRequest.DEFAULT_MODE:
                return StorageUtils.getDefaultCacheDirectory(HSCoreUtil.getContext(),true);
            case HttpRequest.VIDEO_MODE:
                return StorageUtils.getVideoCacheDirectory(HSCoreUtil.getContext(),true);
            default:
                return StorageUtils.getCacheDirectory(HSCoreUtil.getContext());
        }
    }

    /**
     * 通过url来获取本地缓存真实的地址
     * @param url 物料下载url
     * @param isOfflineAd 是否是离线广告
     * @return 本地缓存真实的地址，当url本地没有对应物料时返回""
     */
    public static String getRealCacheFilePath(String url, boolean isOfflineAd) {
        if (TextUtils.isEmpty(url)) {
            return "";
        }
        //获取广告素材保存的路径
        File directory = isOfflineAd ? StorageUtils.getOfflineCacheDirectory(HSCoreUtil.getContext(), true)
                : StorageUtils.getCacheDirectory(HSCoreUtil.getContext());
        if (directory == null) {
            return "";
        }
        //根据url获取素材的文件名
        ImageCacheURL imageURL = new ImageCacheURL(url);
        //命名规则".0"参考代码：DiskLruCache内部类Entry的构造方法
        String fileName = SafeKeyUtils.getSafeKey(imageURL) + ".0";
        String filePath = directory + File.separator + fileName;
        File file = new File(filePath);
        if (file.exists()) {
            return file.getAbsolutePath();
        }
        return "";
    }

}
