package com.cloud.hisavana.net.utils;

import static android.os.Environment.MEDIA_MOUNTED;

import android.content.Context;
import android.os.Environment;
import android.util.Log;

import com.cloud.sdk.commonutil.util.HSCoreUtil;

import java.io.File;

public final class StorageUtils {

    private static final String OFFLINE_CACHE= "offline_cache";
    private static final String NORMAL_CACHE= "hisavana_net_cache";
    private static final String EXO_VIDEO_CACHE = "hisavana_video_cache";

    private static final String NORMAL_VIDEO_CACHE = "hisavana_normal_video_cache";
    private static final String DEFAULT_CACHE = "hisavana_default_cache";
    private static final String TAG= "StorageUtils";

    private volatile static File mAppCacheDir = null;

    private StorageUtils() {
    }

    /**
     * Returns application cache directory. Cache directory will be created on SD card
     * <i>("/Android/data/[app_package_name]/cache")</i> if card is mounted and app has appropriate permission. Else -
     * Android defines cache directory on device's file system.
     *
     * @param context Application context
     * @return Cache {@link File directory}.<br />
     * <b>NOTE:</b> Can be null in some unpredictable cases (if SD card is unmounted and
     * {@link Context#getCacheDir() Context.getCacheDir()} returns null).
     */
    public static File getCacheDirectory(Context context) {
        return getCacheDirectory(context, true);
    }

    /**
     * Returns application cache directory. Cache directory will be created on SD card
     * <i>("/Android/data/[app_package_name]/cache")</i> (if card is mounted and app has appropriate permission) or
     * on device's file system depending incoming parameters.
     *
     * @param context        Application context
     * @param preferExternal Whether prefer external location for cache
     * @return Cache {@link File directory}.<br />
     * <b>NOTE:</b> Can be null in some unpredictable cases (if SD card is unmounted and
     * {@link Context#getCacheDir() Context.getCacheDir()} returns null).
     * modify default dir is getFilesDir(),because third clean master can not clean
     */
    public static File getCacheDirectory(Context context, boolean preferExternal) {
        File appCacheDir = null;
        if (preferExternal) {
            appCacheDir = getExternalFileDir(context);
            if (appCacheDir != null && appCacheDir.exists()) {
                appCacheDir = new File(appCacheDir.getPath() + File.separator + NORMAL_CACHE);
//                /storage/emulated/0/Android/data/com.transsion.sdk.demo/files/netCache
                if (!appCacheDir.exists() && !appCacheDir.mkdir()) {
                    Log.w(TAG, "can't create cache file");
                }
            }
        }
        if (appCacheDir == null || (!appCacheDir.exists() && !appCacheDir.mkdirs())) {
            appCacheDir = HSCoreUtil.getExternalCacheDir(context.getApplicationContext());
            if (appCacheDir == null || !appCacheDir.exists()) {
                appCacheDir = context.getFilesDir();
            }
        }
        if (appCacheDir == null || !appCacheDir.exists()) {
            appCacheDir = context.getCacheDir();
        }
        if (appCacheDir == null) {
            String cacheDirPath = "/data/data/" + context.getPackageName() + File.separator + NORMAL_CACHE;
            Log.w(TAG, "Can't define system cache directory! '%s' will be used." + cacheDirPath);
            appCacheDir = new File(cacheDirPath);
        }
        return appCacheDir;
    }

    /**
     *
     * @param context
     * @return offline cache file
     */
    public static File getOfflineCacheDirectory(Context context, boolean preferExternal) {
        File appCacheDir = null;
        if (preferExternal) {
            appCacheDir = getExternalFileDir(context);
            if (appCacheDir != null && appCacheDir.exists()) {
                appCacheDir = new File(appCacheDir.getPath() + File.separator + OFFLINE_CACHE);
//                /storage/emulated/0/Android/data/com.transsion.sdk.demo/files/netCache
                if (!appCacheDir.exists() && !appCacheDir.mkdir()) {
                    Log.w(TAG, "can't create cache file");
                }
            }
        }
        if (appCacheDir == null) {
            String cacheDirPath = "/data/data/" + context.getPackageName() + File.separator + OFFLINE_CACHE;
            // /data/data/com.transsion.sdk.demo/offline_cache
            appCacheDir = new File(cacheDirPath);
            if (!appCacheDir.exists()) {
                boolean result = appCacheDir.mkdir();
                if (!result) {
                    Log.e(TAG, "getOfflineCacheDirectory,app cache path is not ready");
                }
            }
        }
        return appCacheDir;
    }

    /**
     *
     * @param context
     * @return 外部设备路径，优先拼接
     */
    public static File getExternalFileDir(Context context) {
        if (mAppCacheDir != null) {
            return mAppCacheDir;
        }

        //优先拼接
        String filesDir = Environment.getExternalStorageDirectory() + "/Android/data/" + context.getPackageName() + "/files";
        File file = new File(filesDir);
        if (!file.exists()) {
            boolean result = file.mkdir();
            if (result) {
                mAppCacheDir = file;
            } else {
                Log.w(TAG, "getExternalFileDir is not exists");
            }
        } else {
            mAppCacheDir = file;
        }

        if (mAppCacheDir == null) {
            String externalStorageState;
            try {
                externalStorageState = Environment.getExternalStorageState();
            } catch (NullPointerException e) { // (sh)it happens (Issue #660)
                externalStorageState = "";
            } catch (IncompatibleClassChangeError e) { // (sh)it happens too (Issue #989)
                externalStorageState = "";
            }
            if (MEDIA_MOUNTED.equals(externalStorageState)) {
                mAppCacheDir = context.getExternalFilesDir(null);
            }
        }
        return mAppCacheDir;
    }

    public static File getExoCacheDirectory(Context context, boolean preferExternal) {
        File appCacheDir = null;
        if (preferExternal) {
            appCacheDir = getExternalFileDir(context);
            if (appCacheDir != null) {
                String path = appCacheDir.getPath() + File.separator + EXO_VIDEO_CACHE;
                // /storage/emulated/0/Android/data/com.transsion.sdk.demo/files/images/hasavana_video_cache
                appCacheDir = new File(path);
                if (!appCacheDir.exists()) {
                    boolean result = appCacheDir.mkdir();
                    if (!result) {
                        Log.e(TAG, "getExoCacheDirectory,storage cache is not ready");
                    }
                }
            }
        }
        if (appCacheDir == null) {
            String cacheDirPath = "/data/data/" + context.getPackageName() + File.separator + EXO_VIDEO_CACHE;
            // /data/data/com.transsion.sdk.demo/offline_cache
            appCacheDir = new File(cacheDirPath);
            if (!appCacheDir.exists()) {
                boolean result = appCacheDir.mkdir();
                if (!result) {
                    Log.e(TAG, "getExoCacheDirectory,app cache path is not ready");
                }
            }
        }
        return appCacheDir;
    }

    public static File getVideoCacheDirectory(Context context, boolean preferExternal) {
        File appCacheDir = null;
        if (preferExternal) {
            appCacheDir = getExternalFileDir(context);
            if (appCacheDir != null) {
                String path = appCacheDir.getPath() + File.separator + NORMAL_VIDEO_CACHE;
                // /storage/emulated/0/Android/data/com.transsion.sdk.demo/files/images/hisavana_normal_video_cache
                appCacheDir = new File(path);
                if (!appCacheDir.exists()) {
                    boolean result = appCacheDir.mkdir();
                    if (!result) {
                        Log.e(TAG, "getVideoCacheDirectory,storage cache is not ready");
                    }
                }
            }
        }
        if (appCacheDir == null) {
            String cacheDirPath = "/data/data/" + context.getPackageName() + File.separator + NORMAL_VIDEO_CACHE;
            // /data/data/com.transsion.sdk.demo/hisavana_normal_video_cache
            appCacheDir = new File(cacheDirPath);
            if (!appCacheDir.exists()) {
                boolean result = appCacheDir.mkdir();
                if (!result) {
                    Log.e(TAG, "getVideoCacheDirectory,app cache path is not ready");
                }
            }
        }
        return appCacheDir;
    }

    /**
     *
     * @param context
     * @return offline cache file
     */
    public static File getDefaultCacheDirectory(Context context, boolean preferExternal) {
        File appCacheDir = null;
        if (preferExternal) {
            appCacheDir = getExternalFileDir(context);
            if (appCacheDir != null) {
                String path = appCacheDir.getPath() + File.separator + DEFAULT_CACHE;
                // /storage/emulated/0/Android/data/com.transsion.sdk.demo/files/images/offline_cache
                appCacheDir = new File(path);
                if (!appCacheDir.exists()) {
                    boolean result = appCacheDir.mkdir();
                    if (!result) {
                        Log.e(TAG, "getOfflineCacheDirectory,storage cache is not ready");
                    }
                }
            }
        }
        if (appCacheDir == null) {
            String cacheDirPath = "/data/data/" + context.getPackageName() + File.separator + DEFAULT_CACHE;
            // /data/data/com.transsion.sdk.demo/offline_cache
            appCacheDir = new File(cacheDirPath);
            if (!appCacheDir.exists()) {
                boolean result = appCacheDir.mkdir();
                if (!result) {
                    Log.e(TAG, "getOfflineCacheDirectory,app cache path is not ready");
                }
            }
        }
        return appCacheDir;
    }
}
