package com.cloud.hisavana.net.disklrucache.impl;


import android.text.TextUtils;
import android.util.Log;

import com.cloud.hisavana.net.CommonOkHttpClient;
import com.cloud.hisavana.net.disklrucache.DiskCacheWriteLocker;
import com.cloud.hisavana.net.disklrucache.DiskLruCache;
import com.cloud.hisavana.net.disklrucache.ImageCacheURL;
import com.cloud.hisavana.net.disklrucache.utils.DiskLruCacheUtil;
import com.cloud.hisavana.net.disklrucache.utils.SafeKeyUtils;
import com.cloud.sdk.commonutil.util.CommonLogUtil;
import com.cloud.sdk.commonutil.util.MD5;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import okhttp3.Response;

/**
 * DiskLruCache操作类
 * @author chenshijun
 */
public class DiskCacheImpl implements IDiskCache {
    private static final String TAG = "DiskCacheImpl";
    private static final int APP_VERSION = 1;
    private static final int VALUE_COUNT = 1;
    private final DiskCacheWriteLocker writerLocker = new DiskCacheWriteLocker();
    private DiskLruCache diskLruCache;
    private File directory;
    private final long maxSize;
    /** 是否使用离线缓存 */
    private int cacheMode;

    protected DiskCacheImpl(File directory, long maxSize, int cacheMode) {
        this.directory = directory;
        this.maxSize = maxSize;
        this.cacheMode = cacheMode;
    }

    private DiskLruCache getDiskLruCache() throws IOException {
        DiskLruCache localDiskLruCache = diskLruCache;
        if (localDiskLruCache == null) {
            synchronized(DiskCacheImpl.class){
                localDiskLruCache = diskLruCache;
                if (localDiskLruCache == null) {
                    File cacheFile = DiskLruCacheUtil.getCacheDirectory(cacheMode);
                    if (Log.isLoggable(CommonOkHttpClient.TAG, Log.DEBUG)) {
                        CommonLogUtil.netLog(" ----- getDiskLruCache: directory = " + directory + " , cacheFile = " + cacheFile);
                    }
                    directory = directory == null ? cacheFile : directory;
                    localDiskLruCache =  diskLruCache = DiskLruCache.open(directory, APP_VERSION, VALUE_COUNT, maxSize);
                }
            }
        }
        return localDiskLruCache;
    }

    @Override
    public File get(ImageCacheURL key) {
        return getImageFile(key);
    }

    @Override
    public File get(String key) {
        return getFile(key);
    }

    protected File getImageFile(ImageCacheURL imageURL) {
        if (TextUtils.isEmpty(imageURL.getOriginUrl())) {
            return null;
        }
        //获取广告素材保存的路径
        File directory = DiskLruCacheUtil.getCacheDirectory(cacheMode);
        if (directory == null) {
            return null;
        }
        //命名规则".0"参考代码：DiskLruCache内部类Entry的构造方法
        String fileName = SafeKeyUtils.getSafeKey(imageURL) + ".0";
        try {
            getDiskLruCache().get(fileName);
        } catch (Exception e) {
        }
        return new File(directory + File.separator + fileName);
    }

    protected File getFile(String fileName) {
        if (TextUtils.isEmpty(fileName)) {
            return null;
        }
        //获取广告素材保存的路径
        File directory = DiskLruCacheUtil.getCacheDirectory(cacheMode);
        if (directory == null) {
            return null;
        }
        //命名规则".0"参考代码：DiskLruCache内部类Entry的构造方法
        String key = MD5.MD5Encode(fileName) + ".0";
        try {
            getDiskLruCache().get(key);
        } catch (Exception e) {
        }
        return new File(directory + File.separator + key);
    }

    @Override
    public String put(ImageCacheURL key, byte[] buf) {
        String filePath = "";
        writerLocker.acquire(key);
        OutputStream outputStream = null;
        String safeKey = SafeKeyUtils.getSafeKey(key);
        try {
            final DiskLruCache.Snapshot currentValue = getDiskLruCache().get(safeKey);
            if (currentValue != null) {
                return null;
            }
            DiskLruCache.Editor editor = diskLruCache.edit(safeKey);
            if (editor == null) {
                throw new IllegalStateException("Had two simultaneous puts for: " + safeKey);
            }
            try {
                outputStream = editor.newOutputStream(0);
                outputStream.write(buf);
                outputStream.flush();
                editor.commit();
                diskLruCache.flush();
                filePath = DiskLruCacheUtil.getCacheFilePath(key.getOriginUrl(), cacheMode);
                if (Log.isLoggable(CommonOkHttpClient.TAG, Log.DEBUG)) {
                    CommonLogUtil.netLog("url :" + key.getOriginUrl() + " cacheToDisk with buff: " + filePath);
                }
            } finally {
                editor.abortUnlessCommitted();
            }
        } catch (IOException e) {
            CommonLogUtil.netLog("put: fail" + filePath);
        } finally {
            DiskLruCacheUtil.closeQuietly(outputStream);
            writerLocker.release(key);
        }
        return filePath;
    }

    @Override
    public String put(ImageCacheURL key, Response response) {
       if (response != null && response.body() != null ) {
           return put(key, response.body().byteStream());
       }
       return "";
    }

    @Override
    public String put(ImageCacheURL key, InputStream inputStream) {
        if (key == null || inputStream == null) {
            return "";
        }
        String filePath = "";
        writerLocker.acquire(key);
        OutputStream outputStream = null;
        String safeKey = SafeKeyUtils.getSafeKey(key);
        try {
            final DiskLruCache.Snapshot currentValue = getDiskLruCache().get(safeKey);
            if (currentValue != null) {
                return null;
            }
            DiskLruCache.Editor editor = diskLruCache.edit(safeKey);
            if (editor == null) {
                throw new IllegalStateException("Had two simultaneous puts for: " + safeKey);
            }
            try {
                outputStream = editor.newOutputStream(0);
                byte[] buffer = new byte[1024];
                int bytesRead;
                while ((bytesRead = inputStream.read(buffer)) != -1) {
                    outputStream.write(buffer, 0, bytesRead);
                }
                outputStream.flush();
                editor.commit();
                diskLruCache.flush();
                filePath = DiskLruCacheUtil.getCacheFilePath(key.getOriginUrl(), cacheMode);
                if (Log.isLoggable(CommonOkHttpClient.TAG, Log.DEBUG)) {
                    CommonLogUtil.netLog("url :" + key.getOriginUrl() + " cacheToDisk with response: " + filePath);
                }
            } finally {
                editor.abortUnlessCommitted();
            }
        } catch (Exception e) {
            CommonLogUtil.netLog("put: fail" + filePath);
        } finally {
            DiskLruCacheUtil.closeQuietly(outputStream);
            try {
                inputStream.close();
            } catch (IOException ignored) {
            }
            writerLocker.release(key);
        }
        return filePath;
    }

    @Override
    public void delete(ImageCacheURL key) {
        String safeKey = SafeKeyUtils.getSafeKey(key);
        try {
            getDiskLruCache().remove(safeKey);
        } catch (Exception ignored) {
        }
    }

    @Override
    public void clear() {
        try {
            getDiskLruCache().delete();
            resetDiskCache();
        } catch (IOException e) {
            Log.e(TAG, Log.getStackTraceString(e));
        }
    }

    @Override
    public boolean isFileExit(String url) {
        ImageCacheURL imageCacheURL = new ImageCacheURL(url);
        String safeKey = SafeKeyUtils.getSafeKey(imageCacheURL);
        try {
            final DiskLruCache.Snapshot currentValue = getDiskLruCache().get(safeKey);
           return  currentValue != null;
        }catch (Exception e){
        }
        return false;
    }

    private synchronized void resetDiskCache() {
        diskLruCache = null;
    }
}
