package com.cloud.hisavana.protocol.intercept

import com.cloud.hisavana.protocol.donwgrade.CronetInterceptListener
import com.cloud.hisavana.protocol.donwgrade.DownCountHelper
import com.cloud.hisavana.protocol.donwgrade.DowngradePolicy
import com.cloud.hisavana.protocol.extension.NONE_NET_DOWN_CODE
import com.cloud.hisavana.protocol.extension.isNetDownCount
import com.cloud.hisavana.protocol.extension.isNetDownRequest
import com.cloud.hisavana.protocol.okhttptransport.CronetInterceptor
import com.cloud.sdk.commonutil.util.CommonLogUtil
import com.transsin.networkmonitor.MonitorFactory
import com.transsin.networkmonitor.MonitorHelper
import okhttp3.Interceptor
import okhttp3.Interceptor.Chain
import okhttp3.Request
import okhttp3.Response
import org.chromium.net.CronetEngine
import java.io.IOException

/**
 * Cronet请求拦截器的重试拦截器，用于Cronet请求的时候错误降级
 */
class CronetRetryInterceptor private constructor(builder: Builder) : Interceptor {

    /**
     * Cronet的网络请求拦截器，内部处理了Cronet的网络请求的逻辑。
     */
    private val cronetInterceptor = builder.cronetInterceptor

    private val downCountHelper = builder.downCountHelper

    private val hostMap: Map<String, String>? = builder.hostMap

    private var factory: MonitorFactory? = builder.factory

    private var downgradePolicy: DowngradePolicy? = builder.downgradePolicy

    /**
     *  @param  engine Cronet的网络请求的引擎
     */
    class Builder(engine: CronetEngine) {

        internal val cronetInterceptor: CronetInterceptor =
            CronetInterceptor.newBuilder(engine).build()
        internal val downCountHelper = DownCountHelper()

        internal var hostMap: Map<String, String>? = null

        internal var factory: MonitorFactory? = null

        internal var downgradePolicy: DowngradePolicy? = null

        /**
         * 如果需要Cronet的降级请求域名降级，设置域名降级的映射表
         * @param hostMap 网络请求Host降级映射表
         */
        fun setHostMap(hostMap: Map<String, String>? = null): Builder {
            this.hostMap = hostMap
            return this
        }

        /**
         * 网络库的降级次数限制，超过限定值触发降级回调。
         * @param limitDownCount 降级次数限制,默认值是10
         */
        fun setDowngradeCount(limitDownCount: Int): Builder {
            downCountHelper.limitDownCount = limitDownCount
            return this
        }

        /**
         * 网络降级超过限定值的降级回调监听器
         * 网络库后续的请求都会降级成Okhttp
         * @param listener Cronet网络请求降级监听器
         */
        fun setDowngradeListener(listener: CronetInterceptListener? = null): Builder {
            downCountHelper.listener = listener
            return this
        }

        /**
         * 自定义降级处理策略
         */
        fun setDowngradePolicy(policy: DowngradePolicy): Builder {
            downgradePolicy = policy
            return this
        }

        /**
         * 由于Cronet请求用拦截器的方式实现，所以也会产生一条Okhttp的数据上报
         * 避免重复上报数据，传入OKhttp的MonitorFactory，内部会自动过滤重复上报和上报降级请求。
         * 如果是传入空MonitorFactory无效
         */
        fun eventListenerFactory(eventFactory: MonitorFactory?): Builder {
            if (eventFactory == null) return this
            eventFactory.setCronetFilter(true)
            this.factory = eventFactory
            return this
        }

        fun build(): CronetRetryInterceptor {
            return CronetRetryInterceptor(this)
        }
    }

    @Throws(IOException::class)
    override fun intercept(chain: Interceptor.Chain): Response {
        //触发了网络降级的限制，直接降级到降级域名的Okhttp请求
        if (downCountHelper.isAboveLimit()) {
            CommonLogUtil.netLog("request downgrade, cause failed count exceed limit, failed url is:${chain.request().url()}" )
            val buildRequest = buildDownRequest(chain.request())
            return chain.proceed(buildRequest)
        }
        try {
            val response = cronetInterceptor.intercept(chain)
            if (response.isSuccessful) {
                resetDownCount()
                return response
            } else {
                CommonLogUtil.netLog("request downgrade, cause domain failed1, failed url is:${chain.request().url()}")
                val netDownCode = response.isNetDownRequest()
                val policy = downgradePolicy
                return if (policy != null) {
                    customCodeDowngradePolicy(response, chain, netDownCode, policy)
                } else {
                    codeDowngrade(response, netDownCode, chain)
                }
            }
        } catch (e: Exception) {
            val netDownCode = e.isNetDownRequest()
            return if (netDownCode != NONE_NET_DOWN_CODE) {
                CommonLogUtil.netLog("request downgrade, cause domain failed, failed url is:${chain.request().url()}")
                val policy = downgradePolicy
                if (policy != null) {
                    if (policy.handleCode(netDownCode)) {
                        exceptionDowngrade(chain, netDownCode)
                    }else{
                        throw e
                    }
                } else {
                    exceptionDowngrade(chain, netDownCode)
                }
            } else {
                throw e
            }
        }
    }

    /**
     * 业务方自定义的Code降级策略
     */
    private fun customCodeDowngradePolicy(
        response: Response,
        chain: Chain,
        netDownCode: Int,
        downgradePolicy: DowngradePolicy
    ): Response {
        if (netDownCode == NONE_NET_DOWN_CODE) return response
        if (downgradePolicy.handleCode(netDownCode)) {
            triggerDownCount()
            return requestDowngrade(chain, netDownCode)
        } else {
            return response
        }
    }

    /**
     * httpCode导致的降级
     */
    private fun codeDowngrade(response: Response, netDownCode: Int, chain: Chain): Response {
        if (response.isNetDownCount()) {
            //错误码触发降级计数
            triggerDownCount()
            if (netDownCode != NONE_NET_DOWN_CODE) {
                //错误码触发降级请求
                return requestDowngrade(chain, netDownCode)
            }
        }
        return response
    }

    /**
     * 异常导致的降级
     */
    private fun exceptionDowngrade(chain: Chain,netDownCode: Int): Response {
        //异常触发降级请求和计数
        triggerDownCount()
        return requestDowngrade(chain, netDownCode)
    }

    /**
     * 执行网络降级请求
     */
    private fun requestDowngrade(chain: Interceptor.Chain, errorCode: Int): Response {
        val buildRequest = buildDownRequest(chain.request())
        MonitorHelper.downgrade(chain, buildRequest, errorCode)
        return chain.proceed(buildRequest)
    }

    /**
     * 网络请求触发失败计数
     */
    private fun triggerDownCount() {
        downCountHelper.triggerDown {
            factory?.setCronetFilter(false)
        }
    }

    /**
     * 网络请求错误计数充值
     */
    private fun resetDownCount() {
        downCountHelper.resetCount()
    }

    /**
     * 构建域名降级请求
     */
    private fun buildDownRequest(request: Request): Request {
        val url = request.url()
        val host = url.host()
        val downHost = hostMap?.get(host)
        //如果降级的Host不为空，则降级Host
        return if (downHost != null) {
            request.newBuilder()
                .url(url.newBuilder().host(downHost).build())
                .build()
        } else {
            request
        }
    }
}
