/*
 * Copyright (C) 2022 Google LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.hisavana.admob.util;

import androidx.annotation.IntDef;

import com.cloud.hisavana.sdk.common.constant.TaErrorCode;
import com.google.android.gms.ads.AdError;
import com.google.android.gms.ads.AdRequest;

/** Convenience factory class to create AdError objects for custom events. */
public class HisavanaCustomEventError {

  private HisavanaCustomEventError() {}

  public static final String SAMPLE_SDK_DOMAIN = "com.hisavana.admob.util.sdk";
  public static final String CUSTOM_EVENT_ERROR_DOMAIN = "com.hisavana.admob.util.customevent";
  public static final int NETWORK_ERROR = 9000;
  public static final int NO_AD_CODE = 1002;
  public static final int UNKNOWN_ERROR_CODE = 10001;
  public static final int INVALID_REQUEST = 1017;

  @IntDef(value = {ERROR_NO_AD_UNIT_ID, ERROR_AD_NOT_AVAILABLE, ERROR_NO_ACTIVITY_CONTEXT})
  public @interface SampleCustomEventErrorCode {}

  /** Error raised when the custom event adapter cannot obtain the ad unit id. */
  public static final int ERROR_NO_AD_UNIT_ID = 101;

  /**
   * Error raised when the custom event adapter does not have an ad available when trying to show
   * the ad.
   */
  public static final int ERROR_AD_NOT_AVAILABLE = 102;

  /** Error raised when the custom event adapter cannot obtain the activity context. */
  public static final int ERROR_NO_ACTIVITY_CONTEXT = 103;

  public static AdError createCustomEventNoAdIdError() {
    return new AdError(ERROR_NO_AD_UNIT_ID, "Ad unit id is empty", CUSTOM_EVENT_ERROR_DOMAIN);
  }

  public static AdError createCustomEventAdNotAvailableError() {
    return new AdError(ERROR_AD_NOT_AVAILABLE, "No ads to show", CUSTOM_EVENT_ERROR_DOMAIN);
  }

  public static AdError createCustomEventNoActivityContextError() {
    return new AdError(
        ERROR_NO_ACTIVITY_CONTEXT,
        "An activity context is required to show the sample ad",
        CUSTOM_EVENT_ERROR_DOMAIN);
  }
  /**
   * Creates a custom event {@code AdError}. This error wraps the underlying error thrown by the SDK.
   *
   * @param errorCode A {@code TaErrorCode} to be reported.
   */
  public static AdError createSdkError(TaErrorCode errorCode) {
    return new AdError(getMediationErrorCode(errorCode), errorCode.getErrorMessage(), SAMPLE_SDK_DOMAIN);
  }

  /**
   * Converts the TaErrorCode to AdErrorCode
   */
  private static int getMediationErrorCode(TaErrorCode errorCode) {
    switch (errorCode.getErrorCode()) {
      case UNKNOWN_ERROR_CODE:
        return AdRequest.ERROR_CODE_INTERNAL_ERROR;
      case INVALID_REQUEST:
        return AdRequest.ERROR_CODE_INVALID_REQUEST;
      case NETWORK_ERROR:
        return AdRequest.ERROR_CODE_NETWORK_ERROR;
      case NO_AD_CODE:
        return AdRequest.ERROR_CODE_NO_FILL;
    }
    return errorCode.getErrorCode();
  }
}
