/*
 * Copyright (C) 2014 Google, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.hisavana.admob.adapter;

import android.content.Context;
import android.os.Bundle;
import android.text.TextUtils;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.cloud.hisavana.sdk.api.config.AdManager;
import com.cloud.hisavana.sdk.config.AdxServerConfig;
import com.cloud.sdk.commonutil.athena.PostConstant;
import com.cloud.sdk.commonutil.util.Preconditions;
import com.google.android.gms.ads.AdSize;
import com.google.android.gms.ads.mediation.Adapter;
import com.google.android.gms.ads.mediation.InitializationCompleteCallback;
import com.google.android.gms.ads.mediation.MediationAdRequest;
import com.google.android.gms.ads.mediation.MediationConfiguration;
import com.google.android.gms.ads.mediation.NativeMediationAdRequest;
import com.google.android.gms.ads.mediation.VersionInfo;
import com.google.android.gms.ads.mediation.customevent.CustomEventBanner;
import com.google.android.gms.ads.mediation.customevent.CustomEventBannerListener;
import com.google.android.gms.ads.mediation.customevent.CustomEventInterstitial;
import com.google.android.gms.ads.mediation.customevent.CustomEventInterstitialListener;
import com.google.android.gms.ads.mediation.customevent.CustomEventNative;
import com.google.android.gms.ads.mediation.customevent.CustomEventNativeListener;
import com.hisavana.admob.BuildConfig;
import com.hisavana.admob.ad.HisavanaBanner;
import com.hisavana.admob.ad.HisavanaInterstitial;
import com.hisavana.admob.ad.HisavanaNative;
import com.hisavana.admob.util.AdapterUtil;
import com.transsion.core.CoreUtil;
import com.transsion.ga.AthenaAnalytics;

import java.util.List;

public class SampleAdapter extends Adapter implements CustomEventBanner, CustomEventInterstitial, CustomEventNative {
    private HisavanaBanner hisavanaBanner;
    private HisavanaInterstitial hisavanaInterstitial;
    private HisavanaNative hisavanaNative;

    @Override
    public void initialize(
            Context context,
            InitializationCompleteCallback initializationCompleteCallback,
            List<MediationConfiguration> mediationConfigurations) {
        if (context == null
                || mediationConfigurations.size() == 0
                || mediationConfigurations.get(0) == null
                || mediationConfigurations.get(0).getServerParameters() == null
                || TextUtils.isEmpty(mediationConfigurations.get(0).getServerParameters().getString("parameter"))) {
            AdapterUtil.logW("HiSavanaMediationAdapter --> initialize Failed");
            initializationCompleteCallback.onInitializationFailed("Initialization Failed: Context is null.");
            return;
        }
        Bundle serverParameters = mediationConfigurations.get(0).getServerParameters();
        AdapterUtil.logW("HiSavanaMediationAdapter --> initialize appid -->" + serverParameters.getString("parameter"));
        Preconditions.runOnMainThread(() -> {
            CoreUtil.init(context.getApplicationContext());
            if (AdapterUtil.isDebug()) {
                AthenaAnalytics.setTest(true);
                AdxServerConfig.setAppModle(AdxServerConfig.TEST);
            }
            AdManager.init(new AdManager.AdConfigBuilder()
                    .testRequest(false)
                    .setDebug(AdapterUtil.isDebug())
                    .setAppId(serverParameters.getString("parameter"))
                    .build());

            initializationCompleteCallback.onInitializationSucceeded();
            AdapterUtil.logW("HiSavanaMediationAdapter --> initialize --> AdManager init success");
        });
    }

    @Override
    public VersionInfo getVersionInfo() {
        String versionString = BuildConfig.VERSION_NAME;
        String[] splits = versionString.split("\\.");

        if (splits.length >= 4) {
            int major = Integer.parseInt(splits[0]);
            int minor = Integer.parseInt(splits[1]);
            int micro = Integer.parseInt(splits[2]) * 100 + Integer.parseInt(splits[3]);
            return new VersionInfo(major, minor, micro);
        }

        return new VersionInfo(0, 0, 0);
    }

    @Override
    public VersionInfo getSDKVersionInfo() {
        String versionString = PostConstant.getSdkVersion();
        String[] splits = versionString.split("\\.");

        if (splits.length >= 3) {
            int major = Integer.parseInt(splits[0]);
            int minor = Integer.parseInt(splits[1]);
            int micro = Integer.parseInt(splits[2]);
            return new VersionInfo(major, minor, micro);
        }

        return new VersionInfo(0, 0, 0);
    }

    @Override
    public void requestBannerAd(@NonNull Context context, @NonNull CustomEventBannerListener listener, @Nullable String s, @NonNull AdSize adSize, @NonNull MediationAdRequest mediationAdRequest, @Nullable Bundle customEventExtras) {
        CoreUtil.init(context);
        hisavanaBanner = new HisavanaBanner();
        hisavanaBanner.requestBannerAd(context, listener, s, adSize, mediationAdRequest, customEventExtras);
    }

    @Override
    public void onDestroy() {
        if (hisavanaBanner != null) {
            hisavanaBanner.onDestroy();
        }
        if (hisavanaInterstitial != null) {
            hisavanaInterstitial.onDestroy();
        }
        if (hisavanaNative != null) {
            hisavanaNative.onDestroy();
        }
    }

    @Override
    public void onPause() {

    }

    @Override
    public void onResume() {

    }

    @Override
    public void requestInterstitialAd(@NonNull Context context, @NonNull CustomEventInterstitialListener customEventInterstitialListener, @Nullable String s, @NonNull MediationAdRequest mediationAdRequest, @Nullable Bundle bundle) {
        CoreUtil.init(context);
        hisavanaInterstitial = new HisavanaInterstitial();
        hisavanaInterstitial.requestInterstitialAd(context, customEventInterstitialListener, s, mediationAdRequest, bundle);
    }

    @Override
    public void showInterstitial() {
        if (hisavanaInterstitial != null) {
            hisavanaInterstitial.showInterstitial();
        }
    }

    @Override
    public void requestNativeAd(@NonNull Context context, @NonNull CustomEventNativeListener customEventNativeListener, @Nullable String s, @NonNull NativeMediationAdRequest nativeMediationAdRequest, @Nullable Bundle bundle) {
        CoreUtil.init(context);
        hisavanaNative = new HisavanaNative();
        hisavanaNative.requestNativeAd(context, customEventNativeListener, s, nativeMediationAdRequest, bundle);
    }
}
