package com.cloud.sdk.commonutil.util;

import static com.cloud.sdk.commonutil.util.CommonLogUtil.MEASURE_TAG;

import android.app.Activity;
import android.app.Application;
import android.os.Bundle;
import android.support.annotation.Nullable;
import android.util.Log;


import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * @author wenjing.luo
 * @date :2022/5/26 15:13
 * @description:
 */
public class LifecycleUtil {

    /**
     * 保存应用启动创建的所有Activity，需要注意如果不是在应用创建的时候注册的话将丢失很多Activity。
     */
    private static final List<WeakReference<Activity>> activityList = new ArrayList<>();
    private static final List<LifecycleListener> listenerList = new ArrayList<>();

    /**
     * 获取 应用创建的第一个Activity
     */
    public static Activity getMainActivity() {
        if (activityList.isEmpty() || activityList.get(0) == null || activityList.get(0).get() == null) {
            return null;
        }
        return activityList.get(0).get();
    }

    public static void addListener(LifecycleListener listener) {
        if (listener != null && !listenerList.contains(listener)) {
            listenerList.add(listener);
        }
    }

    /**
     * 监听整个应用的生命周期
     */
    public static void registerActivityLifecycleCallbacks(Application application) {
        if (null == application) {
            return;
        }

        application.registerActivityLifecycleCallbacks(new Application.ActivityLifecycleCallbacks() {

            @Override
            public void onActivityCreated(Activity activity, @Nullable Bundle savedInstanceState) {
                activityList.add(new WeakReference(activity));
            }

            @Override
            public void onActivityStarted(Activity activity) {
                if (!listenerList.isEmpty()) {
                    for (LifecycleListener listener : listenerList) {
                        listener.onActivityStarted(activity);
                    }
                }
            }

            @Override
            public void onActivityResumed(Activity activity) {
            }

            @Override
            public void onActivityPaused(Activity activity) {
            }

            @Override
            public void onActivityStopped(Activity activity) {
                if (!listenerList.isEmpty()) {
                    for (LifecycleListener listener : listenerList) {
                        listener.onActivityStopped(activity);
                    }
                }
            }

            @Override
            public void onActivitySaveInstanceState(Activity activity, Bundle outState) {
            }

            @Override
            public void onActivityDestroyed(Activity activity) {
                if(listenerList.isEmpty()){
                    return;
                }
                try{
                    Iterator<WeakReference<Activity>> iterator = activityList.iterator();
                    while (iterator.hasNext()) {
                        WeakReference<Activity> activityWeakReference = iterator.next();
                        if (null != activityWeakReference && activityWeakReference.get() == activity) {
                            iterator.remove();
                        }
                    }
                }catch (Exception e){
                }
            }
        });
    }

    public abstract static class LifecycleListener {

        public void onActivityStarted(Activity activity) {
        }

        public void onActivityStopped(Activity activity) {
        }
    }
}
