package com.cloud.sdk.commonutil.athena;

import static android.content.Context.TELEPHONY_SERVICE;
import static android.content.Context.WIFI_SERVICE;

import android.Manifest;
import android.annotation.SuppressLint;
import android.app.ActivityManager;
import android.bluetooth.BluetoothAdapter;
import android.bluetooth.BluetoothManager;
import android.content.Context;
import android.content.ContextWrapper;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.content.pm.ResolveInfo;
import android.hardware.Camera;
import android.hardware.camera2.CameraManager;
import android.location.LocationManager;
import android.net.wifi.ScanResult;
import android.net.wifi.WifiInfo;
import android.net.wifi.WifiManager;
import android.os.BatteryManager;
import android.os.Build;
import android.os.Bundle;
import android.os.Environment;
import android.os.StatFs;
import android.os.SystemClock;
import android.provider.Settings;
import android.telephony.TelephonyManager;
import android.text.TextUtils;
import android.util.AndroidRuntimeException;
import android.util.Log;
import android.view.accessibility.AccessibilityManager;

import androidx.core.content.ContextCompat;

import com.cloud.sdk.commonutil.util.AESUtils;
import com.transsion.core.CoreUtil;
import com.transsion.core.utils.SharedPreferencesUtil;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.InputStream;
import java.lang.reflect.Method;
import java.net.NetworkInterface;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * @author xinbing.zhang
 * @date :2022/8/12 11:08
 * @description: 反作弊埋点业务工具类
 * <p>
 * 反作弊埋点详细设计 --> https://transsioner.feishu.cn/docs/doccnEvnfFjWBSvJ3ezbtbTfxFh#
 * <p>
 * 在线进制转换工具 --> https://tool.oschina.net/hexconvert/
 * <p>
 * 黑产识别逻辑 --> https://transsioner.feishu.cn/docs/doccnk9Y06oSLZSDBPGksHkQZDc#
 */
public class AntiFraudUtil {

    /**
     * is_charge             01-是否正在充电，00表示否，01表示是，10表示采集不到
     * is_sim_card           23-是否有sim卡，00表示否，01表示是，10表示采集不到
     * is_cameras            45-是否有摄像头，00表示否，01表示是，10表示采集不到
     * is_root               6-是否root
     * is_change_frame       7-是否篡改架构
     * is_adb_enable         8-是否开通adb权限
     * is_accessibility      9-是否开通辅助权限
     * is_hook               10-是否被hook
     * is_open_sys           11-是否操作系统多开
     * is_open_app           12-是否多开app
     * is_emulator           13-是否模拟器
     * is_use_vpn            14-是否使用vpn
     * is_use_proxy          15-是否使用代理
     * is_mock_location      16-是否允许模拟定位
     * is_monkey             17-是否在跑monkey测试
     * <p>
     * 00000000000000000000000000000000
     */
    private AntiFraudUtil() {
        //
    }

    private static final String TAG = "anti_fraud_log";
    public static final boolean IS_TEST = false;
    private static final boolean IS_LOGCAT_TAG = Log.isLoggable(TAG, Log.DEBUG);

    public static void log(String msg) {
        if (IS_TEST || IS_LOGCAT_TAG) {
            Log.w(TAG, msg);
        }
    }


    /**
     * =============================================================================================
     * is_charge             01-是否正在充电，00表示否，01表示是，10表示采集不到
     * <p>
     * 00000000000000000000000000000000
     * 0000 0000 0000 0000 0000 0000 0000 0000 --> 没有在充电 --> 0
     * <p>
     * 00000000000000000000000000000001
     * 0000 0000 0000 0000 0000 0000 0000 0001 --> 正在充电 --> 1
     * <p>
     * 00000000000000000000000000000010
     * 0000 0000 0000 0000 0000 0000 0000 0010 --> 没有获取到状态 --> 2
     */
    private static final int IS_CHARGE_FALSE = 0;
    private static final int IS_CHARGE_TRUE = 1;
    private static final int IS_CHARGE_UNKNOWN = 2;

    private static int isCharge() {
        try {
            Intent batteryBroadcast = CoreUtil.getContext().registerReceiver(null, new IntentFilter(Intent.ACTION_BATTERY_CHANGED));
            int status = batteryBroadcast.getIntExtra(BatteryManager.EXTRA_STATUS, -1);
            if (status == BatteryManager.BATTERY_STATUS_CHARGING) {
                return IS_CHARGE_TRUE;
            } else if (status == BatteryManager.BATTERY_STATUS_UNKNOWN) {
                return IS_CHARGE_UNKNOWN;
            } else {
                return IS_CHARGE_FALSE;
            }
        } catch (Exception e) {
            log(Log.getStackTraceString(e));
        }
        return IS_CHARGE_UNKNOWN;
    }


    /**
     * =============================================================================================
     * is_sim_card           23-是否有sim卡，00表示否，01表示是，10表示采集不到
     * <p>
     * 00000000000000000000000000000000 --> 没有sim卡 --> 0
     * 00000000000000000000000000000100 --> 有sim卡 --> 4
     * 00000000000000000000000000001000 --> 没有获取到状态 --> 8
     */
    private static final int IS_SIM_CARD_FALSE = 0;
    private static final int IS_SIM_CARD_TRUE = 4;
    private static final int IS_SIM_CARD_UNKNOWN = 8;
    private static int isSimCardValue = -1;

    @SuppressLint({"MissingPermission", "HardwareIds"})
    private static int isSimCard() {
        if (isSimCardValue != -1) {
            return isSimCardValue;
        }
        try {
            isSimCardValue = IS_SIM_CARD_UNKNOWN;
            String imsi = "";
            TelephonyManager manager = (TelephonyManager) CoreUtil.getContext().getSystemService(TELEPHONY_SERVICE);
            // 这个需要权限
            if (null != manager) {
                imsi = manager.getSubscriberId();
            }
            if (imsi == null || imsi.length() <= 0) {
                isSimCardValue = IS_SIM_CARD_FALSE;
            } else {
                isSimCardValue = IS_SIM_CARD_TRUE;
            }
        } catch (Exception e) {
            log(Log.getStackTraceString(e));
        }
        return isSimCardValue;
    }


    /**
     * =============================================================================================
     * is_cameras            45-是否有摄像头，00表示否，01表示是，10表示采集不到
     * <p>
     * 00000000000000000000000000000000 --> 没有摄像头 --> 0
     * 00000000000000000000000000010000 --> 有摄像头 --> 16
     * 00000000000000000000000000100000 --> 没有获取到状态 --> 32
     */
    private static final int IS_CAMERAS_FALSE = 0;
    private static final int IS_CAMERAS_TRUE = 16;
    private static final int IS_CAMERAS_UNKNOWN = 32;
    private static int isCamerasValue = -1;

    private static int isCameras() {
        if (isCamerasValue != -1) {
            return isCamerasValue;
        }
        try {
            isCamerasValue = IS_CAMERAS_UNKNOWN;
            boolean hasCamera = false;
            PackageManager pm = CoreUtil.getContext().getPackageManager();
            if (null != pm) {
                hasCamera = pm.hasSystemFeature(PackageManager.FEATURE_CAMERA_ANY);
            }
            isCamerasValue = hasCamera ? IS_CAMERAS_TRUE : IS_CAMERAS_FALSE;
        } catch (Exception e) {
            log(Log.getStackTraceString(e));
        }
        return isCamerasValue;
    }


    /**
     * =============================================================================================
     * is_root               6-是否root
     * <p>
     * 00000000000000000000000001000000
     * <p>
     * * https://www.jb51.net/article/235532.htm
     */
    private static final int IS_ROOT_FALSE = 0;
    private static final int IS_ROOT_TRUE = 64;
    private static int isRootValue = -1;

    private static final String[] ROOT_RELATED_DIRS = new String[]{
            "/su", "/su/bin/su", "/sbin/su",
            "/data/local/xbin/su", "/data/local/bin/su", "/data/local/su",
            "/system/xbin/su",
            "/system/bin/su", "/system/sd/xbin/su", "/system/bin/failsafe/su",
            "/system/bin/cufsdosck", "/system/xbin/cufsdosck", "/system/bin/cufsmgr",
            "/system/xbin/cufsmgr", "/system/bin/cufaevdd", "/system/xbin/cufaevdd",
            "/system/bin/conbb", "/system/xbin/conbb"};

    private static int isRoot() {
        if (isRootValue != -1) {
            return isRootValue;
        }
        try {
            isRootValue = IS_ROOT_FALSE;
            boolean hasRootDir = false;
            for (String dir : ROOT_RELATED_DIRS) {
                if ((new File(dir)).exists()) {
                    hasRootDir = true;
                    break;
                }
            }
            boolean isRoot = (Build.TAGS != null && Build.TAGS.contains("test-keys")) || hasRootDir;
            isRootValue = isRoot ? IS_ROOT_TRUE : IS_ROOT_FALSE;
        } catch (Exception e) {
            log(Log.getStackTraceString(e));
        }
        return isRootValue;
    }


    /**
     * =============================================================================================
     * is_change_frame       7-是否篡改架构
     * <p>
     * 00000000000000000000000010000000
     */
    private static final int IS_CHANGE_FRAME_FALSE = 0;
    private static final int IS_CHANGE_FRAME_TRUE = 128;

    private static int isChangeFrame() {
        return IS_CHANGE_FRAME_FALSE;
    }


    /**
     * =============================================================================================
     * is_adb_enable         8-是否开通adb权限
     * <p>
     * 00000000000000000000000100000000
     * <p>
     * https://www.csdn.net/tags/NtzaIgzsNjUxOS1ibG9n.html
     * 近期需要用到判断Android手机系统开发者选项是否打开的,
     * 遇事不决,先搜索下，一顿操作后发现都是使用adb调试（Settings.Global.ADB_ENABLED）是否打开来判断的，
     * 但是由于国内各大厂商对系统源码的定制化，部分手机会出现adb调试开启与否和开发者选项无关；
     * 最后还是在源码内找到了控制开发者选项的开关
     * Settings.Global.DEVELOPMENT_SETTINGS_ENABLED
     */
    private static final int IS_ADB_ENABLE_FALSE = 0;
    private static final int IS_ADB_ENABLE_TRUE = 256;
    private static int isAdbEnableValue = -1;

    private static int isAdbEnable() {
        if (isAdbEnableValue != -1) {
            return isAdbEnableValue;
        }
        try {
            isAdbEnableValue = IS_ADB_ENABLE_FALSE;
            boolean isOpenDevelopmentSetting = Settings.Secure.getInt(CoreUtil.getContext().getContentResolver(), Settings.Global.DEVELOPMENT_SETTINGS_ENABLED, 0) != 0;
            boolean isUsbDebugSetting = Settings.Secure.getInt(CoreUtil.getContext().getContentResolver(), Settings.Global.ADB_ENABLED, 0) != 0;
            boolean isAdbEnable = isOpenDevelopmentSetting || isUsbDebugSetting;
            isAdbEnableValue = isAdbEnable ? IS_ADB_ENABLE_TRUE : IS_ADB_ENABLE_FALSE;
        } catch (Exception e) {
            log(Log.getStackTraceString(e));
        }
        return isAdbEnableValue;
    }


    /**
     * =============================================================================================
     * is_accessibility      9-是否开通辅助权限
     * <p>
     * 00000000000000000000001000000000
     * <p>
     * 标准Android系统提供AccessibilityService以满足残障人群使用App的需求，同时提供talkback服务以语音播报的方式提供服务，
     * 开发者也可能通过继承实现AccessibilityService实现自己的无障碍支持。
     * 目前市面上已经存在一些无障碍支持软件如保益悦听、点明安卓、解说等，这类软件同样是基于语音播报的方式提供服务。
     * 此外，还有些软件如爱奇异App也使用了AccessibilityService，然而此类App不是为了提供无障碍服务，
     * 而是使用AccessibilityService以实现诸如自动安装等功能，所以在进行无障碍服务是否开启的判断中要排除此类App使用AccessibilityService的方式。
     * 所以如何准确判断用户是否开启无障碍服务需要考虑以上所有场景。
     * <p>
     * 需要权限
     */
    private static final String SCREEN_READER_INTENT_ACTION = "android.accessibilityservice.AccessibilityService";
    private static final String SCREEN_READER_INTENT_CATEGORY = "android.accessibilityservice.category.FEEDBACK_SPOKEN";
    private static final int IS_ACCESSIBILITY_FALSE = 0;
    private static final int IS_ACCESSIBILITY_TRUE = 512;
    private static int isAccessibilityValue = -1;

    private static boolean isScreenReaderActive() {
        // 通过Intent方式判断是否存在以语音播报方式提供服务的Service，还需要判断开启状态
        Intent screenReaderIntent = new Intent(SCREEN_READER_INTENT_ACTION);
        screenReaderIntent.addCategory(SCREEN_READER_INTENT_CATEGORY);
        @SuppressLint("QueryPermissionsNeeded") List<ResolveInfo> screenReaders = CoreUtil.getContext().getPackageManager().queryIntentServices(screenReaderIntent, 0);
        // 如果没有，返回false
        if (screenReaders.isEmpty()) {
            return false;
        }
        boolean hasActiveScreenReader = false;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            // 高版本可以直接判断服务是否处于开启状态
            for (ResolveInfo screenReader : screenReaders) {
                hasActiveScreenReader |= isAccessibilitySettingsOn(CoreUtil.getContext(), screenReader.serviceInfo.packageName + "/" + screenReader.serviceInfo.name);
            }
        } else {
            // 判断正在运行的Service里有没有上述存在的Service
            List<String> runningServices = new ArrayList<>();
            ActivityManager manager = (ActivityManager) CoreUtil.getContext().getSystemService(Context.ACTIVITY_SERVICE);
            for (ActivityManager.RunningServiceInfo service : manager.getRunningServices(Integer.MAX_VALUE)) {
                runningServices.add(service.service.getPackageName());
            }
            for (ResolveInfo screenReader : screenReaders) {
                if (runningServices.contains(screenReader.serviceInfo.packageName)) {
                    hasActiveScreenReader = true;
                    break;
                }
            }
        }
        return hasActiveScreenReader;
    }

    private static boolean isAccessibilitySettingsOn(Context context, String service) {
        TextUtils.SimpleStringSplitter mStringColonSplitter = new TextUtils.SimpleStringSplitter(':');
        String settingValue = Settings.Secure.getString(
                context.getApplicationContext().getContentResolver(),
                Settings.Secure.ENABLED_ACCESSIBILITY_SERVICES);
        if (settingValue != null) {
            mStringColonSplitter.setString(settingValue);
            while (mStringColonSplitter.hasNext()) {
                String accessibilityService = mStringColonSplitter.next();
                if (accessibilityService.equalsIgnoreCase(service)) {
                    return true;
                }
            }
        }
        return false;
    }

    private static int isAccessibility() {
        if (isAccessibilityValue != -1) {
            return isAccessibilityValue;
        }
        try {
            isAccessibilityValue = IS_ACCESSIBILITY_FALSE;
            // 检查AccessibilityService是否开启
            AccessibilityManager am = (AccessibilityManager) CoreUtil.getContext().getSystemService(android.content.Context.ACCESSIBILITY_SERVICE);
            boolean isAccessibilityEnabledFlag = am.isEnabled();
            // 检查无障碍服务是否以语音播报的方式开启
            boolean isExploreByTouchEnabledFlag = isScreenReaderActive();
            boolean isAccessibilityEnabled = (isAccessibilityEnabledFlag && isExploreByTouchEnabledFlag);
            isAccessibilityValue = isAccessibilityEnabled ? IS_ACCESSIBILITY_TRUE : IS_ACCESSIBILITY_FALSE;
        } catch (Exception e) {
            log(Log.getStackTraceString(e));
        }
        return isAccessibilityValue;
    }


    /**
     * =============================================================================================
     * is_hook               10-是否被hook
     * <p>
     * 00000000000000000000010000000000
     */
    private static final int IS_HOOK_FALSE = 0;
    private static final int IS_HOOK_TRUE = 1024;
    private static int isHookValue = -1;

    /*** 查找设备安装目录中是否存在hook工具*/
    private static boolean findHookAppName() {
        PackageManager packageManager = CoreUtil.getContext().getPackageManager();
        @SuppressLint("QueryPermissionsNeeded") List<ApplicationInfo> applicationInfoList = packageManager.getInstalledApplications(PackageManager.GET_META_DATA);
        for (ApplicationInfo applicationInfo : applicationInfoList) {
            if ("de.robv.android.xposed.installer".equals(applicationInfo.packageName)) {
                return true;
            }
            if ("com.saurik.substrate".equals(applicationInfo.packageName)) {
                return true;
            }
        }
        return false;
    }

    /*** 查找设备存储中是否存在hook安装文件*/
    private static boolean findHookAppFile() {
        Set<String> libraries = new HashSet<>();
        String mapsFilename = "/proc/" + android.os.Process.myPid() + "/maps";

        try (
                FileReader fileReader = new FileReader(mapsFilename);
                BufferedReader reader = new BufferedReader(fileReader)
        ) {
            String line;
            while ((line = reader.readLine()) != null) {
                if (line.endsWith(".so") || line.endsWith(".jar")) {
                    int n = line.lastIndexOf(" ");
                    libraries.add(line.substring(n + 1));
                }
            }
            for (String library : libraries) {
                if (library.contains("com.saurik.substrate")) {
                    return true;
                }
                if (library.contains("XposedBridge.jar")) {
                    return true;
                }
            }
        } catch (Exception e) {
            log(Log.getStackTraceString(e));
        }

        return false;
    }

    /*** 查找程序运行的栈中是否存在hook相关类*/
    private static boolean findHookStack() {
        try {
            throw new AndroidRuntimeException("findhook");
        } catch (Exception e) {
            if (extracted(e)) {
                return true;
            }
        }
        return false;
    }

    private static boolean extracted(Exception e) {
        int zygoteInitCallCount = 0;
        for (StackTraceElement stackTraceElement : e.getStackTrace()) {
            if ("com.android.internal.os.ZygoteInit".equals(stackTraceElement.getClassName())) {
                zygoteInitCallCount++;
                if (zygoteInitCallCount == 2) {
                    return true;
                }
            } else if ("com.saurik.substrate.MS$2".equals(stackTraceElement.getClassName()) && "invoked".equals(stackTraceElement.getMethodName())) {
                return true;
            } else if ("de.robv.android.xposed.XposedBridge".equals(stackTraceElement.getClassName()) && "main".equals(stackTraceElement.getMethodName())) {
                return true;
            } else if ("de.robv.android.xposed.XposedBridge".equals(stackTraceElement.getClassName()) && "handleHookedMethod".equals(stackTraceElement.getMethodName())) {
                return true;
            }
        }
        return false;
    }

    private static int isHook() {
        if (isHookValue != -1) {
            return isHookValue;
        }
        try {
            isHookValue = IS_HOOK_FALSE;
            boolean isHook = (findHookAppName() || findHookAppFile() || findHookStack());
            isHookValue = isHook ? IS_HOOK_TRUE : IS_HOOK_FALSE;
        } catch (Exception e) {
            log(Log.getStackTraceString(e));
        }
        return isHookValue;
    }


    /**
     * =============================================================================================
     * is_open_sys           11-是否操作系统多开
     * <p>
     * 00000000000000000000100000000000
     */
    private static final int IS_OPEN_SYS_FALSE = 0;
    private static final int IS_OPEN_SYS_TRUE = 2048;

    private static int isOpenSys() {
        return IS_OPEN_SYS_FALSE;
    }


    /**
     * =============================================================================================
     * is_open_app           12-是否多开app
     * <p>
     * 00000000000000000001000000000000
     * <p>
     * //多开分身本身的包名
     * //chaos引擎
     * //平行空间
     * //双开助手
     * //VirtualXposed，VirtualApp
     * //360分身大师
     * //DualGenius/双开精灵
     * //GO Multiple/Go双开
     */
    private static final int IS_OPEN_APP_FALSE = 0;
    private static final int IS_OPEN_APP_TRUE = 4096;
    private static int isOpenAppValue = -1;

    private static final String[] PACKAGE_NAME = {
            "com.bly.dkplat",
            "com.by.chaos",
            "com.lbe.parallel",
            "com.excelliance.dualaid",
            "com.lody.virtual",
            "com.qihoo.magic",
            "com.dual.dualgenius",
            "com.jiubang.commerce.gomultiple"
    };

    /***通过读取文件包的方式进行比对*/
    private static int isOpenApp() {
        if (isOpenAppValue != -1) {
            return isOpenAppValue;
        }
        try {
            isOpenAppValue = IS_OPEN_APP_FALSE;
            boolean isOpenApp = false;
            String path = CoreUtil.getContext().getFilesDir().getPath();
            for (String vtpkg : PACKAGE_NAME) {
                if (path.contains(vtpkg)) {
                    isOpenApp = true;
                    break;
                }
            }
            isOpenAppValue = isOpenApp ? IS_OPEN_APP_TRUE : IS_OPEN_APP_FALSE;
        } catch (Exception e) {
            log(Log.getStackTraceString(e));
        }
        return isOpenAppValue;
    }


    /**
     * =============================================================================================
     * is_emulator           13-是否模拟器
     * <p>
     * 00000000000000000010000000000000
     */
    private static final int IS_EMULATOR_FALSE = 0;
    private static final int IS_EMULATOR_TRUE = 8192;
    private static int isEmulatorValue = -1;

    private static final String[] KNOWN_PIPES = {
            "/dev/socket/qemud",
            "/dev/qemu_pipe"
    };

    private static final String[] KNOWN_QEMU_DRIVERS = {
            "goldfish"
    };

    private static final String[] KNOWN_FILES = {
            "/system/lib/libc_malloc_debug_qemu.so",
            "/sys/qemu_trace",
            "/system/bin/qemu-props"
    };

    /**
     * 检测“/dev/socket/qemud”，“/dev/qemu_pipe”这两个通道
     */
    private static boolean checkPipes() {
        for (String pipes : KNOWN_PIPES) {
            File qemuSocket = new File(pipes);
            if (qemuSocket.exists()) {
                return true;
            }
        }
        return false;
    }

    /**
     * 检测驱动文件内容
     * 读取文件内容，然后检查已知QEmu的驱动程序的列表
     */
    private static boolean checkQemuDriverFile() {
        File driverFile = new File("/proc/tty/drivers");
        if (driverFile.exists() && driverFile.canRead()) {
            byte[] data = new byte[1024];
            try (
                    InputStream inStream = new FileInputStream(driverFile);
            ) {
                inStream.read(data);
                String driverData = new String(data);
                for (String knownQemuDriver : KNOWN_QEMU_DRIVERS) {
                    if (driverData.contains(knownQemuDriver)) {
                        return true;
                    }
                }
            } catch (Exception e) {
                log(Log.getStackTraceString(e));
            }
        }
        return false;
    }

    /**
     * 检测模拟器上特有的几个文件
     */
    private static boolean checkEmulatorFiles() {
        for (String fileName : KNOWN_FILES) {
            File qemuFile = new File(fileName);
            if (qemuFile.exists()) {
                return true;
            }
        }
        return false;
    }

    /**
     * 检测手机上的一些硬件信息
     */
    private static boolean checkEmulatorBuild() {
        return "android".equalsIgnoreCase(android.os.Build.BRAND) || android.os.Build.MODEL.toLowerCase().contains("sdk");
    }

    private static int isEmulator() {
        if (isEmulatorValue != -1) {
            return isEmulatorValue;
        }
        try {
            isEmulatorValue = IS_EMULATOR_FALSE;
            boolean isEmulator = checkEmulatorBuild() || checkEmulatorFiles() || checkQemuDriverFile() || checkPipes();
            isEmulatorValue = isEmulator ? IS_EMULATOR_TRUE : IS_EMULATOR_FALSE;
        } catch (Exception e) {
            log(Log.getStackTraceString(e));
        }
        return isEmulatorValue;
    }


    /**
     * =============================================================================================
     * is_use_vpn            14-是否使用vpn
     * <p>
     * 00000000000000000100000000000000
     */
    private static final int IS_USE_VPN_FALSE = 0;
    private static final int IS_USE_VPN_TRUE = 16384;
    private static int isUseVpnValue = -1;

    private static int isUseVpn() {
        if (isUseVpnValue != -1) {
            return isUseVpnValue;
        }
        try {
            isUseVpnValue = IS_USE_VPN_FALSE;
            boolean isUseVpn = false;
            Enumeration<NetworkInterface> niList = NetworkInterface.getNetworkInterfaces();
            if (niList != null) {
                for (NetworkInterface intf : Collections.list(niList)) {
                    if (!intf.isUp() || intf.getInterfaceAddresses().isEmpty()) {
                        log("网络未连接");
                        //continue
                    } else if ("tun0".equals(intf.getName()) || "ppp0".equals(intf.getName())) {
                        isUseVpn = true;
                        break;
                    }
                }
            }
            isUseVpnValue = isUseVpn ? IS_USE_VPN_TRUE : IS_USE_VPN_FALSE;
        } catch (Exception e) {
            log(Log.getStackTraceString(e));
        }
        return isUseVpnValue;
    }


    /**
     * =============================================================================================
     * is_use_proxy          15-是否使用代理
     * <p>
     * 00000000000000001000000000000000
     */
    private static final int IS_USE_PROXY_FALSE = 0;
    private static final int IS_USE_PROXY_TRUE = 32768;
    private static int isUseProxyValue = -1;

    private static int isUseProxy() {
        if (isUseProxyValue != -1) {
            return isUseProxyValue;
        }
        try {
            isUseProxyValue = IS_USE_PROXY_FALSE;
            String proxyAddress = System.getProperty("http.proxyHost");
            String portStr = System.getProperty("http.proxyPort");
            int proxyPort = Integer.parseInt((portStr != null ? portStr : "-1"));
            boolean isUseProxy = (!TextUtils.isEmpty(proxyAddress)) && (proxyPort != -1);
            isUseProxyValue = isUseProxy ? IS_USE_PROXY_TRUE : IS_USE_PROXY_FALSE;
        } catch (Exception e) {
            log(Log.getStackTraceString(e));
        }
        return isUseProxyValue;
    }


    /**
     * =============================================================================================
     * is_mock_location      16-是否允许模拟定位
     * <p>
     * 00000000000000010000000000000000
     * <p>
     * Android 6.0 及以上：没有【允许模拟位置】选项，同时弃用了Settings.Secure.ALLOW_MOCK_LOCATION，无法通过上面的方法判断。
     * 增加了【选择模拟位置信息应用】的方法，需要选择使用模拟位置的应用。
     * 但是不知道怎么获取当前选择的应用，因此通过是否能够成功执行addTestProvider方法来进行判断，如果没有选择当前的应用，则addTestProvider会抛出异常。
     */
    private static final int IS_MOCK_LOCATION_FALSE = 0;
    private static final int IS_MOCK_LOCATION_TRUE = 65536;
    private static int isMockLocationValue = -1;

    private static int isMockLocation() {
        if (isMockLocationValue != -1) {
            return isMockLocationValue;
        }
        try {
            isMockLocationValue = IS_MOCK_LOCATION_FALSE;
            boolean isMockLocation = false;
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                // android 6.0 以上需要动态的授权 定位权限
                if (ContextCompat.checkSelfPermission(CoreUtil.getContext(), Manifest.permission.ACCESS_FINE_LOCATION) != PackageManager.PERMISSION_GRANTED
                        && ContextCompat.checkSelfPermission(CoreUtil.getContext(), Manifest.permission.ACCESS_COARSE_LOCATION) != PackageManager.PERMISSION_GRANTED) {
                    // 如果连权限都没有，那就不存在虚拟位置一说
                    //isMockLocation = false
                    log("当前没有定位权限");
                } else {
                    LocationManager locationManager = (LocationManager) CoreUtil.getContext().getSystemService(Context.LOCATION_SERVICE);
                    String providerStr = LocationManager.GPS_PROVIDER;
                    locationManager.setTestProviderEnabled(providerStr, true);
                    // 模拟位置可用
                    isMockLocation = true;
                }
            } else {
                // Android 6.0 以下：使用Settings.Secure.ALLOW_MOCK_LOCATION判断。
                // Android 6.0 以下：是否开启【允许模拟位置】
                // boolean canMockPosition = Settings.Secure.getInt(getContentResolver(), Settings.Secure.ALLOW_MOCK_LOCATION, 0) != 0
                isMockLocation = Settings.Secure.getInt(CoreUtil.getContext().getContentResolver(), Settings.Secure.ALLOW_MOCK_LOCATION, 0) != 0;
            }
            isMockLocationValue = isMockLocation ? IS_MOCK_LOCATION_TRUE : IS_MOCK_LOCATION_FALSE;
        } catch (Exception e) {
            log(Log.getStackTraceString(e));
        }
        return isMockLocationValue;
    }


    /**
     * =============================================================================================
     * is_monkey             17-是否在跑monkey测试
     * <p>
     * 00000000000000100000000000000000
     */
    private static final int IS_MONKEY_FALSE = 0;
    private static final int IS_MONKEY_TRUE = 131072;
    private static boolean isMonkey = false;

    private static int isMonkey() {
        return isMonkey ? IS_MONKEY_TRUE : IS_MONKEY_FALSE;
    }

    /**
     * 这个需要媒体传入
     *
     * @param monkey is monkey
     */
    public static void setMonkey(boolean monkey) {
        isMonkey = monkey;
    }


    /**
     * 存储解析的数据
     */
    private static final Map<String, Integer> STATE_MAP = new HashMap<>();


    // =============================================================================================
    // =============================================================================================
    // =============================================================================================


    private static final int MAX_PARAMS = 65535;

    /**
     * 将两个int数保存在一个int数中
     * 客户端调用
     * <p>
     * 65535 --> 1111 1111 1111 1111
     *
     * @param numA 高16位保存的数
     * @param numB 低16位保存的数
     * @return 压缩之后的数据
     */
    public static int encryptInt(int numA, int numB) {
        if (numA > MAX_PARAMS || numB > MAX_PARAMS) {
            return -1;
        }
        // 将第一个数 左移16位
        int numC = numA << 16;
        // 将左移之后的结果与第二位数 按位异或得到压缩之后的结果将其返回 相同为0，不同为1
        return numC ^ numB;
    }

    /**
     * 解压 需要注意 这个数转换成二进制之后需要在16位以内
     * <p>
     * 65535 --> 1111 1111 1111 1111
     * <p>
     * 服务端调用
     *
     * @param encryptValue 压缩之后的数
     * @deprecated
     */
    private static void decryptInt(int encryptValue) {
        //int numA = encryptValue >>> 16
        //int numB = (encryptValue << 16) >>> 16
    }

    /**
     * 统一处理 字符串长度小于100
     *
     * @param tag 目标字符串
     * @return 长度小于100的字符串
     */
    private static String subString(String tag) {
        if (null != tag && tag.length() > 100) {
            tag = tag.substring(0, 100);
        }
        return tag;
    }


    // =============================================================================================
    // =============================================================================================
    // =============================================================================================


    /**
     * 添加一个状态值
     * 客户端调用
     *
     * @param verify 当前状态的int值
     * @param type   当前的状态
     * @return 返回操作后的状态的int值
     */
    private static int addState(int verify, int type) {
        return verify | type;
    }

    private static final String IS_CHARGE = "is_charge";
    private static final String IS_SIM_CARD = "is_sim_card";
    private static final String IS_CAMERAS = "is_cameras";

    /**
     * 获取当前状态的字符串拼接
     * <p>
     * ConcurrentModificationException
     *
     * @param verify 当前状态的int值
     */
    private static String getCurrentState(int verify) {
        STATE_MAP.clear();

        if ((IS_CHARGE_TRUE & verify) == IS_CHARGE_TRUE) {
            STATE_MAP.put(IS_CHARGE, 1);
        } else if ((IS_CHARGE_UNKNOWN & verify) == IS_CHARGE_UNKNOWN) {
            STATE_MAP.put(IS_CHARGE, 2);
        } else {
            STATE_MAP.put(IS_CHARGE, 0);
        }

        if ((IS_SIM_CARD_TRUE & verify) == IS_SIM_CARD_TRUE) {
            STATE_MAP.put(IS_SIM_CARD, 1);
        } else if ((IS_SIM_CARD_UNKNOWN & verify) == IS_SIM_CARD_UNKNOWN) {
            STATE_MAP.put(IS_SIM_CARD, 2);
        } else {
            STATE_MAP.put(IS_SIM_CARD, 0);
        }

        if ((IS_CAMERAS_TRUE & verify) == IS_CAMERAS_TRUE) {
            STATE_MAP.put(IS_CAMERAS, 1);
        } else if ((IS_CAMERAS_UNKNOWN & verify) == IS_CAMERAS_UNKNOWN) {
            STATE_MAP.put(IS_CAMERAS, 2);
        } else {
            STATE_MAP.put(IS_CAMERAS, 0);
        }

        if ((IS_ROOT_TRUE & verify) == IS_ROOT_TRUE) {
            STATE_MAP.put("is_root", 1);
        } else {
            STATE_MAP.put("is_root", 0);
        }

        if ((IS_CHANGE_FRAME_TRUE & verify) == IS_CHANGE_FRAME_TRUE) {
            STATE_MAP.put("is_change_frame", 1);
        } else {
            STATE_MAP.put("is_change_frame", 0);
        }

        if ((IS_ADB_ENABLE_TRUE & verify) == IS_ADB_ENABLE_TRUE) {
            STATE_MAP.put("is_adb_enable", 1);
        } else {
            STATE_MAP.put("is_adb_enable", 0);
        }

        if ((IS_ACCESSIBILITY_TRUE & verify) == IS_ACCESSIBILITY_TRUE) {
            STATE_MAP.put("is_accessibility", 1);
        } else {
            STATE_MAP.put("is_accessibility", 0);
        }

        if ((IS_HOOK_TRUE & verify) == IS_HOOK_TRUE) {
            STATE_MAP.put("is_hook", 1);
        } else {
            STATE_MAP.put("is_hook", 0);
        }

        if ((IS_OPEN_SYS_TRUE & verify) == IS_OPEN_SYS_TRUE) {
            STATE_MAP.put("is_open_sys", 1);
        } else {
            STATE_MAP.put("is_open_sys", 0);
        }

        if ((IS_OPEN_APP_TRUE & verify) == IS_OPEN_APP_TRUE) {
            STATE_MAP.put("is_open_app", 1);
        } else {
            STATE_MAP.put("is_open_app", 0);
        }

        if ((IS_EMULATOR_TRUE & verify) == IS_EMULATOR_TRUE) {
            STATE_MAP.put("is_emulator", 1);
        } else {
            STATE_MAP.put("is_emulator", 0);
        }

        if ((IS_USE_VPN_TRUE & verify) == IS_USE_VPN_TRUE) {
            STATE_MAP.put("is_use_vpn", 1);
        } else {
            STATE_MAP.put("is_use_vpn", 0);
        }

        if ((IS_USE_PROXY_TRUE & verify) == IS_USE_PROXY_TRUE) {
            STATE_MAP.put("is_use_proxy", 1);
        } else {
            STATE_MAP.put("is_use_proxy", 0);
        }

        if ((IS_MOCK_LOCATION_TRUE & verify) == IS_MOCK_LOCATION_TRUE) {
            STATE_MAP.put("is_mock_location", 1);
        } else {
            STATE_MAP.put("is_mock_location", 0);
        }

        if ((IS_MONKEY_TRUE & verify) == IS_MONKEY_TRUE) {
            STATE_MAP.put("is_monkey", 1);
        } else {
            STATE_MAP.put("is_monkey", 0);
        }

        return STATE_MAP.toString();
    }

    /**
     * 服务端调用
     */
    private static String getCpMap(int currentState) {
        return getCurrentState(currentState);
    }

    /**
     * 32位整数，0-17位含义如下，18-31位用0填充：
     */
    private static int getCpInt() {
        int currentState = 0;
        currentState = addState(currentState, isCharge());
        currentState = addState(currentState, isSimCard());
        currentState = addState(currentState, isCameras());
        currentState = addState(currentState, isRoot());
        currentState = addState(currentState, isChangeFrame());
        currentState = addState(currentState, isAdbEnable());
        currentState = addState(currentState, isAccessibility());
        currentState = addState(currentState, isHook());
        currentState = addState(currentState, isOpenSys());
        currentState = addState(currentState, isOpenApp());
        currentState = addState(currentState, isEmulator());
        currentState = addState(currentState, isUseVpn());
        currentState = addState(currentState, isUseProxy());
        currentState = addState(currentState, isMockLocation());
        currentState = addState(currentState, isMonkey());
        return currentState;
    }


    // =============================================================================================
    // =============================================================================================
    // =============================================================================================


    /**
     * =============================================================================================
     * 本机的蓝牙名称
     * <p>
     * 这个需要权限
     * <p>
     * bn
     * bluetooth_name
     * 蓝牙名称
     * 字符串
     * 长度小于100
     */
    private static String currentBtName = "bluetooth_name";

    @SuppressLint("MissingPermission")
    private static String getCurrentBtName() {
        if (!"bluetooth_name".equals(currentBtName)) {
            return currentBtName;
        }
        try {
            BluetoothManager bluetoothManager = (BluetoothManager) CoreUtil.getContext().getSystemService(Context.BLUETOOTH_SERVICE);
            BluetoothAdapter adapter = bluetoothManager.getAdapter();
            currentBtName = null != adapter ? (subString(adapter.getName())) : "";
        } catch (Exception e) {
            currentBtName = "";
            log(Log.getStackTraceString(e));
        }
        return currentBtName;
    }


    /**
     * =============================================================================================
     * bbv
     * base_band_version
     * 基带版本号
     * 字符串
     * 长度小于100
     * <p>
     * value --> 220218V439
     */
    private static String baseBandVersion = "base_band_version";

    private static String getBaseBandVersion() {
        if (!"base_band_version".equals(baseBandVersion)) {
            return baseBandVersion;
        }
        try {
            baseBandVersion = subString(Build.VERSION.INCREMENTAL);
        } catch (Exception e) {
            baseBandVersion = "";
            log(Log.getStackTraceString(e));
        }
        return baseBandVersion;
    }


    /**
     * =============================================================================================
     * bt
     * boot_time
     * 系统启动时间，开机时间到当前时间秒数
     * 整型
     * 32位整型
     * sdk将各参数拼接成字符串时整型用16进制字符串表示
     * <p>
     * 毫秒转秒之后在转16进制
     */
    private static String bootTime = "boot_time";

    private static String getBootTime() {
        if (!"boot_time".equals(bootTime)) {
            return bootTime;
        }
        try {
            int time = (int) ((System.currentTimeMillis() - SystemClock.elapsedRealtime()) / 1000);
            bootTime = Integer.toHexString(time);
        } catch (Exception e) {
            bootTime = "";
            log(Log.getStackTraceString(e));
        }
        return bootTime;
    }


    /**
     * =============================================================================================
     * phn
     * phone_name
     * 手机名称
     * 字符串
     * 长度小于100
     * <p>
     * value --> TECNO-CE9
     */
    private static String phoneName = "phone_name";

    private static String getPhoneName() {
        if (!"phone_name".equals(phoneName)) {
            return phoneName;
        }
        try {
            phoneName = subString(Build.DEVICE);
        } catch (Exception e) {
            phoneName = "";
            log(Log.getStackTraceString(e));
        }
        return phoneName;
    }


    /**
     * =============================================================================================
     * si
     * ssid
     * wifi名称 当前连接的Wi-Fi名称
     * 字符串
     * 长度小于100
     */
    private static String ssid = "ssid";

    private static String getSsid() {
        if (!"ssid".equals(ssid)) {
            return ssid;
        }
        try {
            WifiManager wifiManager = (WifiManager) CoreUtil.getContext().getApplicationContext().getSystemService(WIFI_SERVICE);
            WifiInfo wifiInfo = wifiManager.getConnectionInfo();
            ssid = null != wifiInfo ? subString(wifiInfo.getSSID()) : "";
        } catch (Exception e) {
            ssid = "";
            log(Log.getStackTraceString(e));
        }
        return ssid;
    }


    /**
     * =============================================================================================
     * bl
     * wifi_list
     * wifi列表
     * 字符串数组
     * 数组长度不超过5，字符串长度小于100
     */
    private static boolean isWifiListLoaded = false;
    private static String[] wifiListValue = new String[]{};

    private static String[] getWifiList() {
        if (isWifiListLoaded) {
            return wifiListValue;
        }
        try {
            isWifiListLoaded = true;
            WifiManager wifiManager = (WifiManager) CoreUtil.getContext().getApplicationContext().getSystemService(WIFI_SERVICE);
            List<ScanResult> scanWifiList = wifiManager.getScanResults();
            List<String> wifiList = new ArrayList<>();
            if (scanWifiList != null && !scanWifiList.isEmpty()) {
                for (int i = 0; i < scanWifiList.size(); i++) {
                    ScanResult scanResult = scanWifiList.get(i);
                    if (wifiList.size() > 4) {
                        break;
                    }
                    if (!scanResult.SSID.isEmpty()) {
                        wifiList.add(subString(scanResult.SSID));
                    }
                }
            }
            wifiListValue = wifiList.toArray(new String[0]);
        } catch (Exception e) {
            log(Log.getStackTraceString(e));
        }
        return wifiListValue;
    }


    /**
     * =============================================================================================
     * sb
     * screen_brightness
     * screen_brightness_capacity
     * 当前屏幕亮度和屏幕亮度最大值
     * 整型
     * 32位整数，低16位表示屏幕亮度最大值，高16位表示当前屏幕亮度
     * 举例：当前屏幕亮度：234，屏幕亮度最大值321，sb=(234<<16)|321=15335745
     * 二进制：
     * (11101010<<16)|101000001=111010100000000101000001
     * 十六进制：
     * (ea<<16)|141=00ea0141
     * 各参数拼接成字符串时整型用16进制字符串表示
     */
    private static String screenBrightness = "screen_brightness";

    private static String getScreenBrightnessCapacity() {
        if (!"screen_brightness".equals(screenBrightness)) {
            return screenBrightness;
        }
        try {
            // 需要注意的是，返回的亮度是介于0~255之间的int类型值
            int systemBrightness = Settings.System.getInt(CoreUtil.getContext().getContentResolver(), Settings.System.SCREEN_BRIGHTNESS);
            int i = encryptInt(systemBrightness, 255);
            screenBrightness = Integer.toHexString(i);
        } catch (Exception e) {
            screenBrightness = "";
            log(Log.getStackTraceString(e));
        }
        return screenBrightness;
    }


    /**
     * =============================================================================================
     * bpc
     * battery_percent
     * battery_capacity
     * <p>
     * 当前电量和电池容量
     * 整型
     * 32位整数，低16位表示电池容量，高16位表示当前电量
     * 各参数拼接成字符串时整型用16进制字符串表示
     */
    private static String batteryPercent = "battery_percent";

    /*** 获取电池的容量*/
    @SuppressLint("PrivateApi")
    private static int getBatteryTotal() {
        int batteryCapacity = 0;
        try {
            final String powerProfileClass = "com.android.internal.os.PowerProfile";
            Object mPowerProfile = Class.forName(powerProfileClass).getConstructor(Context.class).newInstance(CoreUtil.getContext().getApplicationContext());
            Method method = Class.forName(powerProfileClass).getMethod("getBatteryCapacity");
            // 直接转int 会有问题，需要中转一下
            double battery = (double) method.invoke(mPowerProfile);
            batteryCapacity = (int) battery;
        } catch (Exception e) {
            log(Log.getStackTraceString(e));
        }
        return batteryCapacity;
    }

    /*** 获取当前电量百分比*/
    private static int getBatteryCurrent() {
        int capacity = 0;
        try {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                BatteryManager manager = (BatteryManager) CoreUtil.getContext().getSystemService(Context.BATTERY_SERVICE);
                capacity = manager.getIntProperty(BatteryManager.BATTERY_PROPERTY_CAPACITY);
            } else {
                // 21以下怎么获取电池的电量呢
                Intent intent = new ContextWrapper(CoreUtil.getContext()).registerReceiver(null, new IntentFilter(Intent.ACTION_BATTERY_CHANGED));
                //capacity = (intent.getIntExtra(BatteryManager.EXTRA_LEVEL, -1) * 100) / (intent.getIntExtra(BatteryManager.EXTRA_SCALE, -1))
                capacity = intent.getIntExtra(BatteryManager.EXTRA_LEVEL, -1);
            }
        } catch (Exception e) {
            log(Log.getStackTraceString(e));
        }
        return capacity;
    }

    private static String getBatteryPercent() {
        if (!"battery_percent".equals(batteryPercent)) {
            return batteryPercent;
        }
        try {
            int i = encryptInt(getBatteryCurrent(), getBatteryTotal());
            batteryPercent = Integer.toHexString(i);
        } catch (Exception e) {
            batteryPercent = "";
            log(Log.getStackTraceString(e));
        }
        return batteryPercent;
    }


    /**
     * =============================================================================================
     * ct
     * charge_type
     * 充电类型
     * <p>
     * 枚举
     * 0-未知
     * 1-充电器
     * 2-usb接口
     * 3-无线充电
     */
    private static String chargeType = "charge_type";

    private static String getChargeType() {
        if (!"charge_type".equals(chargeType)) {
            return chargeType;
        }
        try {
            Intent batteryBroadcast = CoreUtil.getContext().registerReceiver(null, new IntentFilter(Intent.ACTION_BATTERY_CHANGED));
            // 0 means we are discharging, anything else means charging --> 0 表示正在放电，其他表示充电
            // boolean isCharging = batteryBroadcast.getIntExtra(BatteryManager.EXTRA_PLUGGED, -1) != 0
            int chargePlug = batteryBroadcast.getIntExtra(BatteryManager.EXTRA_PLUGGED, -1);
            if (chargePlug == BatteryManager.BATTERY_PLUGGED_AC) {
                chargeType = "1";
            } else if (chargePlug == BatteryManager.BATTERY_PLUGGED_USB) {
                chargeType = "2";
            } else if (chargePlug == BatteryManager.BATTERY_PLUGGED_WIRELESS) {
                chargeType = "3";
            } else {
                chargeType = "0";
            }
        } catch (Exception e) {
            chargeType = "0";
            log(Log.getStackTraceString(e));
        }
        return chargeType;
    }


    /**
     * =============================================================================================
     * rn
     * rom_name
     * rom名称
     * 字符串
     * 长度小于100
     * <p>
     * value --> TECNO MOBILE LIMITED
     * <p>
     * 这行返回的是rom定制商的名称
     */
    private static String romName = "rom_name";

    private static String getRomName() {
        if (!"rom_name".equals(romName)) {
            return romName;
        }
        try {
            romName = subString(android.os.Build.MANUFACTURER);
        } catch (Exception e) {
            romName = "";
            log(Log.getStackTraceString(e));
        }
        return romName;
    }


    /**
     * =============================================================================================
     * rv
     * rom_version
     * rom版本号
     * 字符串
     * 长度小于100
     * https://blog.csdn.net/weixin_32964881/article/details/117505714
     * https://blog.csdn.net/sheikhdz/article/details/50505054
     * https://www.jianshu.com/p/5891275fc525
     * https://blog.csdn.net/weixin_42510717/article/details/117611622?spm=1001.2101.3001.6650.4&utm_medium=distribute.pc_relevant.none-task-blog-2%7Edefault%7ECTRLIST%7ERate-4-117611622-blog-50505054.t0_eslanding_v1&depth_1-utm_source=distribute.pc_relevant.none-task-blog-2%7Edefault%7ECTRLIST%7ERate-4-117611622-blog-50505054.t0_eslanding_v1&utm_relevant_index=9
     * <p>
     * infinix、tecno、itel 这三个都  ro.build.display.id
     * <p>
     * 安全问题 不上报
     */
    private static String getRomVersion() {
        return "";
    }


    /**
     * =============================================================================================
     * rb
     * rom_build_time
     * rom编译时间
     * 长整型
     * 13位时间戳
     * 各参数拼接成字符串时整型用16进制字符串表示
     * <p>
     * ro.build.date.utc是编译时间
     */
    private static String romBuildTime = "rom_build_time";

    private static String getRomBuildTime() {
        if (!"rom_build_time".equals(romBuildTime)) {
            return romBuildTime;
        }
        try {
            return Long.toHexString(Build.TIME);
        } catch (Exception e) {
            romBuildTime = "";
            log(Log.getStackTraceString(e));
        }
        return romBuildTime;
    }


    /**
     * =============================================================================================
     * oa
     * os_api
     * Android os api版本号
     * 整型
     * 16位整数
     * 各参数拼接成字符串时整型用16进制字符串表示
     * <p>
     * value --> 11
     */
    private static String osApi = "os_api";

    private static String getOsApi() {
        if (!"os_api".equals(osApi)) {
            return osApi;
        }
        try {
            osApi = Integer.toHexString(Build.VERSION.SDK_INT);
        } catch (Exception e) {
            osApi = "";
            log(Log.getStackTraceString(e));
        }
        return osApi;
    }

    /**
     * =============================================================================================
     * bi
     * build_id
     * <p>
     * build系统id，用来获取设备信息的
     * 字符串
     * 长度小于100
     * <p>
     * 设备版本号。
     * <p>
     * value --> RP1A.200720.011
     */
    private static String buildId = "build_id";

    private static String getBuildId() {
        if (!"build_id".equals(buildId)) {
            return buildId;
        }
        try {
            buildId = subString(Build.ID);
        } catch (Exception e) {
            buildId = "";
            log(Log.getStackTraceString(e));
        }
        return buildId;
    }


    /**
     * =============================================================================================
     * cc
     * cpu_cores
     * 核心处理器，cpu型号
     * 字符串
     * 长度小于100
     * <p>
     * value --> arm64-v8a
     */
    private static String cpuCores = "cpu_cores";

    private static String getCpuCores() {
        if (!"cpu_cores".equals(cpuCores)) {
            return cpuCores;
        }
        try {
            cpuCores = subString(Build.CPU_ABI);
        } catch (Exception e) {
            cpuCores = "";
            log(Log.getStackTraceString(e));
        }
        return cpuCores;
    }


    /**
     * =============================================================================================
     * ccf
     * cpu_cores_cnt
     * cpu核数和cpu主频
     * 整型
     * 低16位表示cpu主频，高16位表示cpu核数
     * 各参数拼接成字符串时整型用16进制字符串表示
     * <p>
     * 实时获取CPU当前频率（单位KHZ）
     * CPU频率它决定计算机的运行速度，随着计算机的发展，主频由过去MHZ发展到了当前的GHZ（1GHZ=10^3MHZ=10^6KHZ= 10^9HZ）。
     */
    private static final String K_CPU_INFO_CUR_FREQ_FILEPATH = "/sys/devices/system/cpu/cpu0/cpufreq/scaling_cur_freq";
    private static String cpuCoresCnt = "cpu_cores_cnt";

    /*** @return MHZ */
    private static int getCurCpuFreq() {
        int result = 0;
        try (
                FileReader fr = new FileReader(K_CPU_INFO_CUR_FREQ_FILEPATH);
                BufferedReader br = new BufferedReader(fr)
        ) {
            String text = br.readLine();
            result = Integer.parseInt(text.trim());
        } catch (Exception e) {
            log(Log.getStackTraceString(e));
        }
        return result;
    }

    private static String getCpuCoresCnt() {
        if (!"cpu_cores_cnt".equals(cpuCoresCnt)) {
            return cpuCoresCnt;
        }
        try {
            // 获取 CPU 核数
            int ccf = Runtime.getRuntime().availableProcessors();
            // MHZ
            int curCpuFreq = getCurCpuFreq() / 1000;
            int i = encryptInt(ccf, curCpuFreq);
            cpuCoresCnt = Integer.toHexString(i);
        } catch (Exception e) {
            cpuCoresCnt = "";
            log(Log.getStackTraceString(e));
        }
        return cpuCoresCnt;
    }


    /**
     * =============================================================================================
     * ca
     * cpu_abi
     * cpu架构
     * 字符串
     * 长度小于100
     * <p>
     * value --> arm64-v8a
     */
    private static String cpuAbi = "cpu_abi";

    private static String getCpuAbi() {
        if (!"cpu_abi".equals(cpuAbi)) {
            return cpuAbi;
        }
        try {
            cpuAbi = subString(android.os.Build.CPU_ABI);
        } catch (Exception e) {
            cpuAbi = "";
            log(Log.getStackTraceString(e));
        }
        return cpuAbi;
    }


    /**
     * =============================================================================================
     * ck
     * cpu_kernel
     * cpu内核 --> cpu_kernel暂时取abi2字段
     * 字符串
     * 长度小于100
     */
    private static String cpuKernel = "cpu_kernel";

    private static String getCpuKernel() {
        if (!"cpu_kernel".equals(cpuKernel)) {
            return cpuKernel;
        }
        try {
            cpuKernel = subString(Build.CPU_ABI2);
        } catch (Exception e) {
            cpuKernel = "";
            log(Log.getStackTraceString(e));
        }
        return cpuKernel;
    }


    /**
     * =============================================================================================
     * bd
     * board
     * 主板
     * 字符串
     * 长度小于100
     * <p>
     * value --> TECNO-CE9
     */
    private static String board = "board";

    private static String getBoard() {
        if (!"board".equals(board)) {
            return board;
        }
        try {
            board = subString(Build.BOARD);
        } catch (Exception e) {
            board = "";
            log(Log.getStackTraceString(e));
        }
        return board;
    }


    /**
     * =============================================================================================
     * hw
     * hard_ware
     * 主板硬件参数
     * 字符串
     * 长度小于100
     * <p>
     * 设备硬件名称,一般和基板名称一样（BOARD）
     * <p>
     * value --> mt6785
     */
    private static String hardWare = "hard_ware";

    private static String getHardWare() {
        if (!"hard_ware".equals(hardWare)) {
            return hardWare;
        }
        try {
            hardWare = subString(android.os.Build.HARDWARE);
        } catch (Exception e) {
            hardWare = "";
            log(Log.getStackTraceString(e));
        }
        return hardWare;
    }


    /**
     * =============================================================================================
     * fp
     * finger_print
     * 主板指纹
     * 字符串
     * 长度小于100
     * <p>
     * value --> TECNO/CE9-OP/TECNO-CE9:11/RP1A.200720.011/220218V439:user/release-keys
     */
    private static String fingerPrint = "finger_print";

    private static String getFingerPrint() {
        if (!"finger_print".equals(fingerPrint)) {
            return fingerPrint;
        }
        try {
            fingerPrint = subString(android.os.Build.FINGERPRINT);
        } catch (Exception e) {
            fingerPrint = "";
            log(Log.getStackTraceString(e));
        }
        return fingerPrint;
    }


    /**
     * =============================================================================================
     * noc
     * number_of_cameras
     * 摄像头数量
     * 整型
     * 16位整数
     * 各参数拼接成字符串时整型用16进制字符串表示
     */
    private static int numberOfCameras = 0;
    private static boolean isNumberOfCamerasLoaded = false;

    private static String getNumberOfCameras() {
        if (isNumberOfCamerasLoaded) {
            return Integer.toHexString(numberOfCameras);
        }
        try {
            isNumberOfCamerasLoaded = true;
            if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.LOLLIPOP) {
                CameraManager cameraManager = (CameraManager) CoreUtil.getContext().getSystemService(Context.CAMERA_SERVICE);
                String[] cameraIdList = cameraManager.getCameraIdList();
                if (cameraIdList.length > 0) {
                    //cameraIdList.length 是否有摄像头和摄像头个数
                    numberOfCameras = cameraIdList.length;
                }
            } else {
                // Released publicly as Android 5.0 in November 2014.
                int camNum = Camera.getNumberOfCameras();
                if (camNum > 0) {
                    numberOfCameras = camNum;
                }
            }
        } catch (Exception e) {
            log(Log.getStackTraceString(e));
        }
        return Integer.toHexString(numberOfCameras);
    }


    /**
     * =============================================================================================
     * ps
     * phone_height
     * phone_width
     * 手机尺寸长宽
     * 整型
     * <p>
     * 低16位表示长，高16位表示宽
     * 各参数拼接成字符串时整型用16进制字符串表示
     */
    private static String phoneHeight = "phone_height";

    private static String getPhoneHeightWidth() {
        if (!"phone_height".equals(phoneHeight)) {
            return phoneHeight;
        }
        try {
            int i = encryptInt(PostConstant.getScreenWidth(), PostConstant.getScreenHeight());
            phoneHeight = Integer.toHexString(i);
        } catch (Exception e) {
            phoneHeight = "";
            log(Log.getStackTraceString(e));
        }
        return phoneHeight;
    }


    /**
     * =============================================================================================
     * tm
     * total_memory
     * 总内存（M）
     * 整型
     * 32位整数
     * 各参数拼接成字符串时整型用16进制字符串表示
     */
    private static int totalMemory = 0;
    private static boolean isTotalMemoryLoaded = false;

    private static String getTotalMemory() {
        if (isTotalMemoryLoaded) {
            return Integer.toHexString(totalMemory);
        }
        isTotalMemoryLoaded = true;
        // 系统内存信息文件
        String str1 = "/proc/meminfo";
        int initialMemory;
        try (
                FileReader localFileReader = new FileReader(str1);
                BufferedReader localBufferedReader = new BufferedReader(localFileReader, 8192)
        ) {
            // 读取meminfo第一行，系统总内存大小
            String str2 = localBufferedReader.readLine();
            String[] arrayOfString = str2.split("\\s+");
            // 获得系统总内存，单位是KB
            initialMemory = Integer.parseInt(arrayOfString[1]);
            //log("总内存（M）--> initialMemory = " + (initialMemory / (1024)))
            totalMemory = initialMemory / (1024);
        } catch (Exception e) {
            log(Log.getStackTraceString(e));
        }
        //return Formatter.formatFileSize(context, initial_memory);// Byte转换为KB或者MB，内存大小规格化
        //System.out.println("总运存--->>>" + initial_memory / (1024))
        return Integer.toHexString(totalMemory);
    }


    /**
     * =============================================================================================
     * am
     * available_memory
     * 剩余内存大小（M）
     * 整型
     * 32位整数
     * 各参数拼接成字符串时整型用16进制字符串表示
     * <p>
     * 获取android当前可用运行内存大小
     * https://blog.csdn.net/weixin_42061754/article/details/82969311
     */
    private static String getAvailableMemory() {
        try {
            ActivityManager am = (ActivityManager) CoreUtil.getContext().getSystemService(Context.ACTIVITY_SERVICE);
            ActivityManager.MemoryInfo mi = new ActivityManager.MemoryInfo();
            am.getMemoryInfo(mi);
            // mi.availMem; 当前系统的可用内存
            //return Formatter.formatFileSize(context, mi.availMem)// 将获取的内存大小规格化
            //log("剩余内存大小（M）--> size = " + (int) (mi.availMem / (1024 * 1024)))
            return Integer.toHexString((int) (mi.availMem / (1024 * 1024)));
        } catch (Exception e) {
            log(Log.getStackTraceString(e));
        }
        return "";
    }


    /**
     * =============================================================================================
     * 磁盘大小
     */
    private static int totalDiskSize = 0;
    private static int availableDiskSize = 0;
    private static boolean isTotalDiskSizeLoaded = false;

    private static void getDiskSize() {
        String state = Environment.getExternalStorageState();
        if (Environment.MEDIA_MOUNTED.equals(state)) {
            File dataFileDir = Environment.getExternalStorageDirectory();
            // 获得一个磁盘状态对象
            StatFs stat = new StatFs(dataFileDir.getPath());
            // 获得一个扇区的大小
            long blockSize = stat.getBlockSizeLong();
            if (blockSize == 0) {
                return;
            }
            // 获得扇区的总数
            long totalBlocks = stat.getBlockCountLong();
            // 获得可用的扇区数量
            long availableBlocks = stat.getAvailableBlocksLong();
            //String totalMemory = Formatter.formatFileSize(CoreUtil.getContext(), totalBlocks * blockSize)
            //String availableMemory = Formatter.formatFileSize(CoreUtil.getContext(), availableBlocks * blockSize)
            //log("总空间: " + totalMemory + " ----- 可用空间: " + availableMemory)
            totalDiskSize = (int) ((totalBlocks * blockSize) / (1024 * 1024));
            availableDiskSize = (int) ((availableBlocks * blockSize) / (1024 * 1024));
            //log("总空间: " + totalDiskSize + " ----- 可用空间: " + availableDiskSize)
        }
    }

    /**
     * td
     * total_disk
     * 总磁盘大小（M）
     * 整型
     * <p>
     * 32位整数
     * 各参数拼接成字符串时整型用16进制字符串表示
     * http://t.zoukankan.com/tuncaysanli-p-2423593.html
     */
    private static String getTotalDisk() {
        if (isTotalDiskSizeLoaded) {
            return Integer.toHexString(totalDiskSize);
        }
        try {
            isTotalDiskSizeLoaded = true;
            getDiskSize();
            return Integer.toHexString(totalDiskSize);
        } catch (Exception e) {
            log(Log.getStackTraceString(e));
        }
        return "";
    }

    /**
     * ad
     * available_disk
     * 剩余磁盘大小（M）
     * 整型
     * 32位整数
     * 各参数拼接成字符串时整型用16进制字符串表示
     */
    private static String getAvailableDisk() {
        if (isTotalDiskSizeLoaded) {
            return Integer.toHexString(availableDiskSize);
        }
        try {
            return Integer.toHexString(availableDiskSize);
        } catch (Exception e) {
            log(Log.getStackTraceString(e));
        }
        return "";
    }


    // =============================================================================================
    // =============================================================================================
    // =============================================================================================


    /**
     * 分隔符&用\u0001替换，分隔符=用\u0002替换
     */
    private static final String TAG_KEY = "\u0001";
    private static final String TAG_VALUE = "\u0002";

    /***预加载标识 默认加载中*/
    private static final AtomicBoolean IS_PRELOAD_LOADING = new AtomicBoolean(true);

    /***完整版的反作弊埋点（除传感器之外的数据）*/
    private static final Bundle ANTI_FRAUD_FULL_BUNDLE = new Bundle();
    private static String antiFraudFullString;
    /***服务端反作弊埋点缓存*/
    private static String antiFraudLiteString;

    /***SP 本地保存 name\key */
    private static final String ANTI_FRAUD_SP_NAME = "anti_fraud_sp_name";
    private static final String IS_OPEN_ANTI_FRAUD_SP_KEY = "is_open_anti_fraud_sp_key";

    /***缓存这个状态 不需要每次都读取 SP，SP是耗时的*/
    private static volatile Boolean isOpenAntiFraudCache;

    /***将服务端的结果保存在本地 SSP、聚合调用设置*/
    public static void saveAntiFraudSwitch(boolean isOpenAntiFraud) {
        log("获取到云控 isOpenAntiFraud = " + isOpenAntiFraud);
        // 不需要每次都保存SP的了
        if (null != isOpenAntiFraudCache && isOpenAntiFraudCache == isOpenAntiFraud) {
            return;
        }
        isOpenAntiFraudCache = isOpenAntiFraud;
        if (isOpenAntiFraud && IS_PRELOAD_LOADING.get() ){
            log("saveAntiFraudSwitch --> 没有预加载");
            preloadAntiFraud();
        }
        SharedPreferencesUtil.getInstance(ANTI_FRAUD_SP_NAME).putBoolean(IS_OPEN_ANTI_FRAUD_SP_KEY, isOpenAntiFraud);
    }

    /***获取本地保存的 反作弊开关 默认关闭*/
    public static boolean isOpenAntiFraud() {
        if (null == isOpenAntiFraudCache) {
            isOpenAntiFraudCache = SharedPreferencesUtil.getInstance(ANTI_FRAUD_SP_NAME).getBoolean(IS_OPEN_ANTI_FRAUD_SP_KEY, false);
        }
        return isOpenAntiFraudCache;
    }

    /**
     * 优化 在初始化的时候 预加载数据 在子线程中完成
     */
    public static void preloadAntiFraud() {
        if (!AntiFraudUtil.isOpenAntiFraud()) {
            log("preloadAntiFraud --> 云控关闭了反作弊埋点上报");
            return;
        }

        try {
            SensorUtil.register();
            getAntiFraudFullBundle();
            IS_PRELOAD_LOADING.set(false);
            log("preloadAntiFraud --> 初始化完成 设置标识 IS_PRELOAD_LOADING = false");
        } catch (Exception e) {
            // 失败了，当前就不会在上报反作弊埋点了
            log(Log.getStackTraceString(e));
        }
    }

    /**
     * 统一对外 --> 新增反作弊埋点字段
     * <p>
     * 埋点上报都在一个线程完成的
     * <p>
     * ad_ssp_request、ad_ssp_return、ad_ssp_show、ad_ssp_click
     *
     * @param bundle bundle
     */
    public static void setAntiFraudFullString(Bundle bundle) {
        if (null == bundle) {
            return;
        }

        if (!TextUtils.isEmpty(antiFraudFullString)) {
            bundle.putString("fraud_data_set", antiFraudFullString);
            return;
        }

        if (!isOpenAntiFraud()) {
            log("======= setAntiFraudFullString 服务端关闭了 =========");
            return;
        }

        try {
            // 如果正在预加载中，将不上报当前字段
            if (!IS_PRELOAD_LOADING.get()) {
                // 判断是否已经添加过传感器数据，如果没有那就添加，否则不添加。 ots 是传感器的数据key。
                // 因为不知道传感器什么时候获取到数据
                boolean finish = SensorUtil.getInstance().isFinish();
                ANTI_FRAUD_FULL_BUNDLE.putAll(SensorUtil.getSensorBundle());
                // 将全量反作弊字段加密处理
                String antiFraudString = getAntiFraudString(ANTI_FRAUD_FULL_BUNDLE);
                bundle.putString("fraud_data_set", antiFraudString);
                if (finish) {
                    ANTI_FRAUD_FULL_BUNDLE.clear();
                    antiFraudFullString = antiFraudString;
                }
            } else {
                bundle.putString("fraud_data_set", "");
            }
        } catch (Exception e) {
            log(Log.getStackTraceString(e));
            bundle.putString("fraud_data_set", "");
        }
    }

    /**
     * 获取 完整版的 反作弊埋点 加密数据
     *
     * @param bundle bundle
     * @return string
     */
    private static String getAntiFraudString(Bundle bundle) {
        if (null == bundle) {
            return "";
        }
        // 数据格式处理 总共37个字段
        StringBuilder sb = new StringBuilder();
        try {
            // 数据拼接
            Set<String> keySet = bundle.keySet();
            // 移除第一个拼接的 &符号
            // bundle.putString("bn", getCurrentBtName())
            sb.append("bn").append(TAG_VALUE).append(getCurrentBtName());
            for (String key : keySet) {
                Object ob = bundle.get(key);
                if (ob instanceof String) {
                    if (TextUtils.isEmpty((String) ob)) {
                        ob = "\"\"";
                    }
                    sb.append(TAG_KEY).append(key).append(TAG_VALUE).append(ob);
                } else if (ob instanceof Float) {
                    sb.append(TAG_KEY).append(key).append(TAG_VALUE).append(ob);
                } else if (ob instanceof float[]) {
                    sb.append(TAG_KEY).append(key).append(TAG_VALUE).append(Arrays.toString((float[]) ob));
                } else if (ob instanceof String[]) {
                    sb.append(TAG_KEY).append(key).append(TAG_VALUE).append(Arrays.toString((String[]) ob));
                }
            }
        } catch (Exception e) {
            log(Log.getStackTraceString(e));
            sb.append("");
        }
        //log("getAntiFraudString = " + sb)
        // 加密返回
        return AESUtils.encrypt(sb.toString());
    }

    /**
     * 客户端埋点完整版 （缺少传感器相关数据）
     */
    private static void getAntiFraudFullBundle() {
        Bundle bundle = ANTI_FRAUD_FULL_BUNDLE;
        bundle.putString("bbv", getBaseBandVersion());
        bundle.putString("bt", getBootTime());
        bundle.putString("phn", getPhoneName());
        bundle.putString("si", getSsid());
        bundle.putStringArray("bl", getWifiList());
        bundle.putString("sb", getScreenBrightnessCapacity());
        bundle.putString("bpc", getBatteryPercent());
        bundle.putString("ct", getChargeType());
        bundle.putString("rn", getRomName());
        bundle.putString("rv", getRomVersion());
        bundle.putString("rb", getRomBuildTime());
        bundle.putString("oa", getOsApi());
        bundle.putString("bi", getBuildId());
        bundle.putString("cc", getCpuCores());
        bundle.putString("ccf", getCpuCoresCnt());
        bundle.putString("ca", getCpuAbi());
        bundle.putString("ck", getCpuKernel());
        bundle.putString("bd", getBoard());
        bundle.putString("hw", getHardWare());
        bundle.putString("fp", getFingerPrint());
        bundle.putString("noc", getNumberOfCameras());
        bundle.putString("ps", getPhoneHeightWidth());
        bundle.putString("tm", getTotalMemory());
        bundle.putString("td", getTotalDisk());
        bundle.putString("am", getAvailableMemory());
        bundle.putString("ad", getAvailableDisk());
        // 传感器部分
        //bundle.putAll(SensorUtil.getSensorBundle());
        // 组合开关
        bundle.putString("cp", Integer.toHexString(getCpInt()));
    }

    /**
     * 服务端埋点精简版
     * 服务端曝光、点击、请求 的时候上报这些新增的参数
     * <p>
     * 分隔符&用\u0001替换，分隔符=用\u0002替换
     *
     * 再多线程情况下调用，只有一个赋值操作
     */
    public static String getAntiFraudLiteString() {
        // 已经加载过了，直接返回
        if (!TextUtils.isEmpty(antiFraudLiteString)) {
            return antiFraudLiteString;
        }
        if (!isOpenAntiFraud()) {
            log("=========== getAntiFraudLiteString 服务端关闭了 ===========");
            return "";
        }
        if (IS_PRELOAD_LOADING.get()) {
            // 如果正在预加载将不上报数据
            return "";
        }
        try {
            Bundle bundle = new Bundle();
            bundle.putString("phn", getPhoneName());
            bundle.putString("bt", getBootTime());
            bundle.putString("si", getSsid());
            bundle.putStringArray("bl", getWifiList());
            bundle.putString("sb", getScreenBrightnessCapacity());
            bundle.putString("bpc", getBatteryPercent());
            bundle.putString("rn", getRomName());
            bundle.putString("rv", getRomVersion());
            bundle.putString("rb", getRomBuildTime());
            bundle.putString("oa", getOsApi());
            bundle.putString("cp", Integer.toHexString(getCpInt()));
            antiFraudLiteString = getAntiFraudString(bundle);
        } catch (Exception e) {
            log(Log.getStackTraceString(e));
        }
        if(TextUtils.isEmpty(antiFraudLiteString)){
            antiFraudLiteString = "";
        }
        return antiFraudLiteString;
    }

}
