package com.cloud.sdk.commonutil.athena;

import static android.content.Context.SENSOR_SERVICE;

import static com.cloud.sdk.commonutil.athena.AntiFraudUtil.log;

import android.hardware.Sensor;
import android.hardware.SensorEvent;
import android.hardware.SensorEventListener;
import android.hardware.SensorManager;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.util.Log;

import com.transsion.core.CoreUtil;

import java.util.concurrent.atomic.AtomicBoolean;

/**
 * @author xinbing.zhang
 * @date :2022/8/16 14:57
 * @description: 处理传感器的 统一处理
 */
public class SensorUtil implements SensorEventListener {

    private final SensorManager mSensorManager;

    /***方向传感器 xyz三轴数据 float数组、数组长度为3，浮点数保留小数点后8位*/
    private float[] orientationValues = new float[]{};
    /***光线传感器 float、浮点数保留小数点后8位*/
    private float lightValue;
    /***磁场传感器 float数组、数组长度为3，浮点数保留小数点后8位*/
    private float[] magneticFieldValue = new float[]{};
    /***距离传感器 float、浮点数保留小数点后8位*/
    private float proximityValue;
    /***温度传感器 float、浮点数保留小数点后8位*/
    private float temperatureValue;
    /***陀螺仪传感器 float数组、数组长度为3，浮点数保留小数点后8位*/
    private float[] gyroscopeValue = new float[]{};
    /***压力传感器 float、浮点数保留小数点后8位*/
    private float pressureValue;
    /***重力传感器 float、数组长度为3，浮点数保留小数点后8位*/
    private float[] gravityValue = new float[]{};
    /***加速度传感器 float数组、数组长度为3，浮点数保留小数点后8位*/
    private float[] accelerometerValue = new float[]{};

    /***2000ms后自动注销注册*/
    private final AtomicBoolean isFinish = new AtomicBoolean(false);


    private static class SingletonHolder {
        private static final SensorUtil INSTANCE = new SensorUtil();
    }

    public static SensorUtil getInstance() {
        return SingletonHolder.INSTANCE;
    }

    private SensorUtil() {
        mSensorManager = (SensorManager) CoreUtil.getContext().getSystemService(SENSOR_SERVICE);
        //Log.w(TAG, "SensorManagerUtil: init mSensorManager")
    }

    @Override
    public void onSensorChanged(SensorEvent event) {

        if (null == event || null == event.sensor || null == event.values || event.values.length < 1) {
            return;
        }

        // 获取触发event的传感器类型
        int sensorType = event.sensor.getType();
        switch (sensorType) {
            case Sensor.TYPE_ORIENTATION:
                orientationValues = event.values;
                break;
            case Sensor.TYPE_LIGHT:
                lightValue = event.values[0];
                break;
            case Sensor.TYPE_MAGNETIC_FIELD:
                magneticFieldValue = event.values;
                break;
            case Sensor.TYPE_PROXIMITY:
                proximityValue = event.values[0];
                break;
            case Sensor.TYPE_TEMPERATURE:
                temperatureValue = event.values[0];
                break;
            case Sensor.TYPE_GYROSCOPE:
                gyroscopeValue = event.values;
                break;
            case Sensor.TYPE_PRESSURE:
                pressureValue = event.values[0];
                break;
            case Sensor.TYPE_GRAVITY:
                gravityValue = event.values;
                break;
            case Sensor.TYPE_ACCELEROMETER:
                accelerometerValue = event.values;
                break;
            default:
                break;
        }

    }

    @Override
    public void onAccuracyChanged(Sensor sensor, int accuracy) {
        //Log.w(TAG, "onAccuracyChanged: accuracy = " + accuracy)
    }

    /**
     * 开启  这里需要用到 原子计数
     */
    public void onStart() {
        // 方向传感器
        mSensorManager.registerListener(this, mSensorManager.getDefaultSensor(Sensor.TYPE_ORIENTATION), SensorManager.SENSOR_DELAY_FASTEST);
        // 光线传感器
        mSensorManager.registerListener(this, mSensorManager.getDefaultSensor(Sensor.TYPE_LIGHT), SensorManager.SENSOR_DELAY_FASTEST);
        // 磁场传感器
        mSensorManager.registerListener(this, mSensorManager.getDefaultSensor(Sensor.TYPE_MAGNETIC_FIELD), SensorManager.SENSOR_DELAY_FASTEST);
        // 距离传感器
        mSensorManager.registerListener(this, mSensorManager.getDefaultSensor(Sensor.TYPE_PROXIMITY), SensorManager.SENSOR_DELAY_FASTEST);
        // 温度传感器
        mSensorManager.registerListener(this, mSensorManager.getDefaultSensor(Sensor.TYPE_TEMPERATURE), SensorManager.SENSOR_DELAY_FASTEST);
        // 陀螺仪传感器
        mSensorManager.registerListener(this, mSensorManager.getDefaultSensor(Sensor.TYPE_GYROSCOPE), SensorManager.SENSOR_DELAY_FASTEST);
        // 压力传感器
        mSensorManager.registerListener(this, mSensorManager.getDefaultSensor(Sensor.TYPE_PRESSURE), SensorManager.SENSOR_DELAY_FASTEST);
        // 重力传感器
        mSensorManager.registerListener(this, mSensorManager.getDefaultSensor(Sensor.TYPE_GRAVITY), SensorManager.SENSOR_DELAY_FASTEST);
        // 加速度传感器
        mSensorManager.registerListener(this, mSensorManager.getDefaultSensor(Sensor.TYPE_ACCELEROMETER), SensorManager.SENSOR_DELAY_FASTEST);
    }

    /**
     * 关闭
     */
    public void onStop() {
        new Handler(Looper.getMainLooper()).postDelayed(() -> {
            isFinish.set(true);
            mSensorManager.unregisterListener(SensorUtil.this);
        }, 1000);
    }

    public boolean isFinish() {
        return isFinish.get();
    }

    public float[] getOrientationSensor() {
        return orientationValues;
    }

    public float getLightSensor() {
        return lightValue;
    }

    public float[] getMagneticFieldSensor() {
        return magneticFieldValue;
    }

    public float getProximitySensor() {
        return proximityValue;
    }

    public float getTemperatureSensor() {
        return temperatureValue;
    }

    public float[] getGyroscopeSensor() {
        return gyroscopeValue;
    }

    public float getPressureSensor() {
        return pressureValue;
    }

    public float[] getGravitySensor() {
        return gravityValue;
    }

    public float[] getAccelerometerSensor() {
        return accelerometerValue;
    }

    public static void register() {
        try {
            SensorUtil.getInstance().onStart();
            SensorUtil.getInstance().onStop();
        } catch (Exception e) {
            log(Log.getStackTraceString(e));
        }
    }

    /**
     * 获取传感器数据
     *
     * @return bundle
     */
    public static Bundle getSensorBundle() {
        // 传感器部分
        Bundle bundle = new Bundle();
        try {
            bundle.putFloatArray("ots", SensorUtil.getInstance().getOrientationSensor());
            bundle.putFloat("ls", SensorUtil.getInstance().getLightSensor());
            bundle.putFloatArray("ms", SensorUtil.getInstance().getMagneticFieldSensor());
            bundle.putFloat("pxs", SensorUtil.getInstance().getProximitySensor());
            bundle.putFloat("tps", SensorUtil.getInstance().getTemperatureSensor());
            bundle.putFloatArray("gs", SensorUtil.getInstance().getGyroscopeSensor());
            bundle.putFloat("pss", SensorUtil.getInstance().getPressureSensor());
            bundle.putFloatArray("gvs", SensorUtil.getInstance().getGravitySensor());
            bundle.putFloatArray("ams", SensorUtil.getInstance().getAccelerometerSensor());
        } catch (Exception e) {
            log(Log.getStackTraceString(e));
        }
        return bundle;
    }


    /**
     * 测试输出结果
     */
//    public void testSensor() {
//        TranssionPoolManager.getInstance().addTask(() -> {
//
//            // 注册回调
//            onStart();
//
//            // 开启之后做一个延时 为了回调能获取到数据
//            try {
//                Thread.sleep(200);
//            } catch (InterruptedException e) {
//                Thread.currentThread().interrupt();
//                Log.getStackTraceString(e);
//            }
//
//            Bundle bundle = new Bundle();
//            float[] orientationSensor = SensorUtil.getInstance().getOrientationSensor();
//            bundle.putFloatArray("ots", orientationSensor);
//            log("ots --> orientationSensor = " + Arrays.toString(orientationSensor));
//
//            float lightSensor = SensorUtil.getInstance().getLightSensor();
//            bundle.putFloat("ls", lightSensor);
//            log("ls --> lightSensor = " + lightSensor);
//
//            float[] magneticFieldSensor = SensorUtil.getInstance().getMagneticFieldSensor();
//            bundle.putFloatArray("ms", magneticFieldSensor);
//            log("ms --> magneticFieldSensor = " + Arrays.toString(magneticFieldSensor));
//
//            float proximitySensor = SensorUtil.getInstance().getProximitySensor();
//            bundle.putFloat("pxs", proximitySensor);
//            log("pxs --> proximitySensor = " + proximitySensor);
//
//            float temperatureSensor = SensorUtil.getInstance().getTemperatureSensor();
//            bundle.putFloat("tps", temperatureSensor);
//            log("tps --> temperatureSensor = " + temperatureSensor);
//
//            float[] gyroscopeSensor = SensorUtil.getInstance().getGyroscopeSensor();
//            bundle.putFloatArray("gs", gyroscopeSensor);
//            log("gs --> gyroscopeSensor = " + Arrays.toString(gyroscopeSensor));
//
//            float pressureSensor = SensorUtil.getInstance().getPressureSensor();
//            bundle.putFloat("pss", pressureSensor);
//            log("pss --> pressureSensor = " + pressureSensor);
//
//            float[] gravitySensor = SensorUtil.getInstance().getGravitySensor();
//            bundle.putFloatArray("gvs", gravitySensor);
//            log("gvs --> gravitySensor = " + Arrays.toString(gravitySensor));
//
//            float[] accelerometerSensor = SensorUtil.getInstance().getAccelerometerSensor();
//            bundle.putFloatArray("ams", accelerometerSensor);
//            log("ams --> accelerometerSensor = " + Arrays.toString(accelerometerSensor));
//
//            // 数据格式处理 总共37个字段
//            StringBuilder sb = new StringBuilder();
//            Set<String> keySet = bundle.keySet();
//            for (String key : keySet) {
//                Object ob = bundle.get(key);
//                if (ob instanceof String) {
//                    sb.append("&").append(key).append("=").append(ob);
//                } else if (ob instanceof Float) {
//                    sb.append("&").append(key).append("=").append(ob);
//                } else if (ob instanceof float[]) {
//                    sb.append("&").append(key).append("=").append(Arrays.toString((float[]) ob));
//                } else if (ob instanceof String[]) {
//                    sb.append("&").append(key).append("=").append(Arrays.toString((String[]) ob));
//                }
//            }
//
//            // 真机测试日志
//            // W/bury_log: ots --> orientationSensor = [318.68314, -54.434685, 0.469732]
//            //W/bury_log: ls --> lightSensor = 425.0
//            //W/bury_log: ms --> magneticFieldSensor = [0.28125, -131.53125, -431.34378]
//            //W/bury_log: pxs --> proximitySensor = 10.0
//            //W/bury_log: tps --> temperatureSensor = 0.0
//            //W/bury_log: gs --> gyroscopeSensor = null
//            //W/bury_log: pss --> pressureSensor = 0.0
//            //W/bury_log: gvs --> gravitySensor = null
//            //W/bury_log: ams --> accelerometerSensor = [0.043050002, 8.083051, 5.6569505]
//            //W/bury_log: SensorManagerUtil --> 测试传感器 --> s = &ls=425.0&ms=[0.28125, -131.53125, -431.34378]&ams=[0.043050002, 8.083051, 5.6569505]&ots=[318.68314, -54.434685, 0.469732]&pss=0.0&pxs=10.0&tps=0.0
//            log("SensorManagerUtil --> 测试传感器 --> s = " + sb);
//
//            // 关闭回调
//            onStop();
//        });
//    }

}
