package com.cloud.sdk.commonutil.util;

import android.text.TextUtils;
import android.util.Base64;
import android.util.Log;

import com.transsion.core.CoreUtil;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.security.KeyFactory;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import javax.crypto.Cipher;

public class RSAUtils {

	public static final String TAG = "RSAUtils";// 非对称加密密钥算法
	public static final String RSA = "RSA";// 非对称加密密钥算法
	public static final String ECB_PKCS1_PADDING = "RSA/ECB/PKCS1Padding";// 加密填充方式
	public static final int DEFAULT_KEY_SIZE = 2048;// 秘钥默认长度
	public static final byte[] DEFAULT_SPLIT = "#PART#".getBytes(); // 当要加密的内容超过bufferSize，则采用partSplit进行分块加密
	public static final int DEFAULT_BUFFERSIZE = (DEFAULT_KEY_SIZE / 8) - 11;// 当前秘钥支持加密的最大字节数
	private static byte[] publicKey = new byte[0];//公钥
	public static String PUBLICKEY_ASSETS = "hisavana_rsa_public_key.pem";//公钥地址

	/**
	 * 用公钥对字符串进行加密
	 *
	 * @param data 原文
	 */
	public static byte[] encryptByPublicKey(byte[] data, byte[] publicKey) throws Exception {
		// 得到公钥
		X509EncodedKeySpec keySpec = new X509EncodedKeySpec(publicKey);
		KeyFactory kf = KeyFactory.getInstance(RSA);
		PublicKey keyPublic = kf.generatePublic(keySpec);
		// 加密数据
		Cipher cipher = Cipher.getInstance(ECB_PKCS1_PADDING);
		cipher.init(Cipher.ENCRYPT_MODE, keyPublic);
		return cipher.doFinal(data);
	}

	/**
	 * 使用私钥进行解密
	 */
	public static byte[] decryptByPrivateKey(byte[] encrypted, byte[] privateKey) throws Exception {
		// 得到私钥
		PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(privateKey);
		KeyFactory kf = KeyFactory.getInstance(RSA);
		PrivateKey keyPrivate = kf.generatePrivate(keySpec);

		// 解密数据
		Cipher cp = Cipher.getInstance(ECB_PKCS1_PADDING);
		cp.init(Cipher.DECRYPT_MODE, keyPrivate);
		return cp.doFinal(encrypted);
	}

	/**
	 * 用公钥对字符串进行分段加密
	 */
	public static byte[] encryptByPublicKeyForSpilt(byte[] data, byte[] publicKey) throws Exception {
		int dataLen = data.length;
		if (dataLen <= DEFAULT_BUFFERSIZE) {
			return encryptByPublicKey(data, publicKey);
		}
		List<Byte> allBytes = new ArrayList<Byte>(2048);
		int bufIndex = 0;
		int subDataLoop = 0;
		byte[] buf = new byte[DEFAULT_BUFFERSIZE];
		for (int i = 0; i < dataLen; i++) {
			Objects.requireNonNull(buf)[bufIndex] = data[i];
			if (++bufIndex == DEFAULT_BUFFERSIZE || i == dataLen - 1) {
				subDataLoop++;
				if (subDataLoop != 1) {
					for (byte b : DEFAULT_SPLIT) {
						allBytes.add(b);
					}
				}
				byte[] encryptBytes = encryptByPublicKey(buf, publicKey);
				for (byte b : encryptBytes) {
					allBytes.add(b);
				}
				bufIndex = 0;
				if (i == dataLen - 1) {
					buf = null;
				} else {
					buf = new byte[Math.min(DEFAULT_BUFFERSIZE, dataLen - i - 1)];
				}
			}
		}
		byte[] bytes = new byte[allBytes.size()];
		{
			int i = 0;
			for (Byte b : allBytes) {
				bytes[i++] = b.byteValue();
			}
		}
		return bytes;
	}
	/**
	 * 使用私钥分段解密
	 *
	 */
	public static byte[] decryptByPrivateKeyForSpilt(byte[] encrypted, byte[] privateKey) throws Exception {
		int splitLen = DEFAULT_SPLIT.length;
		if (splitLen <= 0) {
			return decryptByPrivateKey(encrypted, privateKey);
		}
		int dataLen = encrypted.length;
		List<Byte> allBytes = new ArrayList<Byte>(1024);
		int latestStartIndex = 0;
		for (int i = 0; i < dataLen; i++) {
			byte bt = encrypted[i];
			boolean isMatchSplit = false;
			if (i == dataLen - 1) {
				// 到data的最后了
				byte[] part = new byte[dataLen - latestStartIndex];
				System.arraycopy(encrypted, latestStartIndex, part, 0, part.length);
				byte[] decryptPart = decryptByPrivateKey(part, privateKey);
				for (byte b : decryptPart) {
					allBytes.add(b);
				}
				latestStartIndex = i + splitLen;
				i = latestStartIndex - 1;
			} else if (bt == DEFAULT_SPLIT[0]) {
				// 这个是以split[0]开头
				if (splitLen > 1) {
					if (i + splitLen < dataLen) {
						// 没有超出data的范围
						for (int j = 1; j < splitLen; j++) {
							if (DEFAULT_SPLIT[j] != encrypted[i + j]) {
								break;
							}
							if (j == splitLen - 1) {
								// 验证到split的最后一位，都没有break，则表明已经确认是split段
								isMatchSplit = true;
							}
						}
					}
				} else {
					// split只有一位，则已经匹配了
					isMatchSplit = true;
				}
			}
			if (isMatchSplit) {
				byte[] part = new byte[i - latestStartIndex];
				System.arraycopy(encrypted, latestStartIndex, part, 0, part.length);
				byte[] decryptPart = decryptByPrivateKey(part, privateKey);
				for (byte b : decryptPart) {
					allBytes.add(b);
				}
				latestStartIndex = i + splitLen;
				i = latestStartIndex - 1;
			}
		}
		byte[] bytes = new byte[allBytes.size()];
		{
			int i = 0;
			for (Byte b : allBytes) {
				bytes[i++] = b.byteValue();
			}
		}
		return bytes;
	}

	public static void test() {
		try {
			String publicKeyStr = getFromAssets("hisavana_rsa_public_key.pem");
			Log.e(TAG, "公钥---->" + publicKeyStr);
			byte[] publicKey = Base64.decode(publicKeyStr, Base64.DEFAULT);


			String privateKeyStr = getFromAssets("rsa_private_key.pem");
			Log.e(TAG, "私钥---->" + privateKeyStr);
			byte[] privateKey = Base64.decode(privateKeyStr, Base64.DEFAULT);

			String jsonData = "jsonData";

			// 公钥加密
			long start = System.currentTimeMillis();
			byte[] encryptBytes = encryptByPublicKeyForSpilt(jsonData.getBytes(), publicKey);
			long end = System.currentTimeMillis();
			Log.e(TAG, "公钥加密耗时 cost time---->" + (end - start));
			String encryStr = Base64.encodeToString(encryptBytes, Base64.DEFAULT);
			Log.e(TAG, "加密后json数据 --1-->" + encryStr);
			Log.e(TAG, "加密后json数据长度 --1-->" + encryStr.length());
			// 私钥解密
			start = System.currentTimeMillis();
			byte[] decryptBytes = RSAUtils.decryptByPrivateKeyForSpilt(Base64.decode(encryStr, Base64.DEFAULT),
					privateKey);
			String decryStr = new String(decryptBytes);
			end = System.currentTimeMillis();
			Log.e(TAG, "私钥解密耗时 cost time---->" + (end - start));
			Log.e(TAG, "解密后json数据 --1-->" + decryStr);
		} catch (Exception e) {
			CommonLogUtil.Log().d(TAG, Log.getStackTraceString(e));
		}

	}

	/**
	 * 加密arraylist
	 * @param arrayList
	 * @return
	 */
	public static ArrayList<String> encryptByPublicKeyForSpilt(ArrayList<String> arrayList){
		ArrayList<String> newArrayList = new ArrayList<>();
		try {
			if (arrayList != null && !arrayList.isEmpty()){
				for(String s : arrayList){
					if (!TextUtils.isEmpty(s)){
						String encryStr = encryptByPublicKeyForSpilt(s);
						newArrayList.add(encryStr);
					}
				}
			}
		}catch (Exception e){
			CommonLogUtil.Log().d(TAG, Log.getStackTraceString(e));
			Log.e(TAG, "加密失败-->" + e.getMessage());
		}
		return newArrayList;
	}

	/**
	 * 加密字符串
	 * @param str
	 * @return
	 */
	public static String encryptByPublicKeyForSpilt(String str){
		String encryStr = "";
		try {
			if (publicKey.length == 0){
				String publicKeyStr = getFromAssets("hisavana_rsa_public_key.pem");
				publicKey = Base64.decode(publicKeyStr, Base64.DEFAULT);
			}
			if (!TextUtils.isEmpty(str)){
				byte[] encryptBytes = encryptByPublicKeyForSpilt(str.getBytes(), publicKey);
				encryStr = Base64.encodeToString(encryptBytes, Base64.DEFAULT);
			}
		}catch (Exception e){
			CommonLogUtil.Log().d(TAG, Log.getStackTraceString(e));
			Log.e(TAG, "加密失败-->" + e.getMessage());
		}
		return encryStr;
	}
	/**
	 * 获取密钥
	 */
	public static void setPublicKey() {
		if (publicKey.length == 0) {
			String publicKeyStr = getFromAssets(PUBLICKEY_ASSETS);
			publicKey = Base64.decode(publicKeyStr, Base64.DEFAULT);
		}
	}

	public static String getFromAssets(String fileName){
		try (InputStreamReader inputReader = new InputStreamReader(CoreUtil.getContext().getResources().getAssets().open(fileName) );
			 BufferedReader bufReader = new BufferedReader(inputReader)){
			String line;
			StringBuilder Result= new StringBuilder();
			while((line = bufReader.readLine()) != null){
				Result.append(line);
			}
			return Result.toString();
		} catch (Exception e) {
			CommonLogUtil.Log().d(TAG, Log.getStackTraceString(e));
		}
		return "";
	}

}
