package com.cloud.sdk.commonutil.util.imagerender

import android.graphics.drawable.Drawable
import android.widget.ImageView
import com.bumptech.glide.Glide
import com.bumptech.glide.Priority
import com.bumptech.glide.load.engine.DiskCacheStrategy
import com.bumptech.glide.load.resource.gif.GifDrawable
import com.bumptech.glide.request.RequestOptions
import com.cloud.sdk.commonutil.util.CommonLogUtil
import com.cloud.sdk.commonutil.util.HSCoreUtil

object GlideRender: ImageRenderable {

    /**
     * 宿主是否依赖了 Glide
     */
    var isGlideDependency = false
        private set

    /**
     * 宿主设置是否可以使用 Glide
     */
    @JvmStatic
    var canUseGlide = true

    /***
     * Glide 是否可用。
     * true: 依赖了glide且媒体设置可以使用glide;
     * false: 未依赖glide 或者媒体设置不可使用
     */
    private val available: Boolean
        get() = (isGlideDependency && canUseGlide)

    fun setup() {
        try {
            if (Class.forName("com.bumptech.glide.Glide") != null) {
                isGlideDependency = true
                CommonLogUtil.Log().d(HisavanaImageRender.TAG, "init glide succeed")
            } else {
                CommonLogUtil.Log().d(HisavanaImageRender.TAG, "init glide error, cannot found com.bumptech.glide.Glide")
            }
        } catch (e: Throwable) {
            CommonLogUtil.Log()
                .w(HisavanaImageRender.TAG, "init glide error=${e.message}")
        }
    }

    @JvmStatic
    fun loadImage(url: String?, highPriority: Boolean, imageView: ImageView?): Boolean {
        try {
            if (!available || url.isNullOrBlank() || imageView == null) {
                return false
            }

            var options = RequestOptions().diskCacheStrategy(DiskCacheStrategy.NONE)
            if (highPriority) {
                options = options.useUnlimitedSourceGeneratorsPool(true).priority(Priority.IMMEDIATE)
            }
            Glide.with(HSCoreUtil.getContext())
                .load(url)
                .apply(options)
                .into(imageView)
            CommonLogUtil.Log().d(HisavanaImageRender.TAG, "loadImage with glide, url=${url}")
        } catch (e: Throwable) {
            CommonLogUtil.Log().w(HisavanaImageRender.TAG, "loadImage with glide failed, error=${e.message}")
            return false
        }
        return true
    }

    // 使用 Glide 加载 GIF 并获取 GifDrawable
    override fun getGifDrawableFromData(bytes: ByteArray): Drawable? {
        try {
            if (!available) {
                return null
            }

            val result = Glide.with(HSCoreUtil.getContext())
                .asGif() // 明确指定加载 GIF
                .useUnlimitedSourceGeneratorsPool(true)
                .priority(Priority.IMMEDIATE)
                .diskCacheStrategy(DiskCacheStrategy.NONE)
                .load(bytes)
                .submit()
                .get()
            CommonLogUtil.Log().d(HisavanaImageRender.TAG, "glide: getGifDrawableFromData succees")
            return result
        } catch (e: InterruptedException) {
            CommonLogUtil.Log().w(HisavanaImageRender.TAG, "glide: getGifDrawableFromData failed, error=${e.message}")
            return null
        }
    }

    override fun activatedDrawable(drawable: Drawable?): Drawable? {
        try {
            if (available && drawable is GifDrawable) {
                drawable.start()
            }
        } catch (_: Exception) {}
        return drawable
    }
}