package com.cloud.sdk.commonutil.pool;

import android.os.Process;
import android.os.StrictMode;
import androidx.annotation.NonNull;
import android.util.Log;
import java.io.File;
import java.io.FilenameFilter;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.FutureTask;
import java.util.concurrent.RunnableFuture;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;

public final class HSPoolExecutor extends ThreadPoolExecutor {
    public static final String DEFAULT_SOURCE_EXECUTOR_NAME = "source";
    public static final String TRANSSION_EXECUTOR_NAME = "zero";
    public static final String DEFAULT_DISK_CACHE_EXECUTOR_NAME = "disk-cache";
    public static final int DEFAULT_DISK_CACHE_EXECUTOR_THREADS = 1;
    private static final String TAG = "ZeroPoolExecutor";
    private static final String CPU_NAME_REGEX = "cpu[0-9]+";
    private static final String CPU_LOCATION = "/sys/devices/system/cpu/";
    private static final int MAXIMUM_AUTOMATIC_THREAD_COUNT = 4;
    private final boolean executeSynchronously;

    public static HSPoolExecutor newDiskCacheExecutor() {
        return newDiskCacheExecutor(1, "disk-cache", HSPoolExecutor.UncaughtThrowableStrategy.DEFAULT);
    }

    public static HSPoolExecutor newDiskCacheExecutor(int threadCount, String name, HSPoolExecutor.UncaughtThrowableStrategy uncaughtThrowableStrategy) {
        return new HSPoolExecutor(threadCount, name, uncaughtThrowableStrategy, true, false);
    }

    public static HSPoolExecutor newSourceExecutor() {
        return newSourceExecutor(calculateBestThreadCount(), "source", HSPoolExecutor.UncaughtThrowableStrategy.DEFAULT);
    }

    public static HSPoolExecutor newSourceExecutor(int threadCount, String name, HSPoolExecutor.UncaughtThrowableStrategy uncaughtThrowableStrategy) {
        return new HSPoolExecutor(threadCount, name, uncaughtThrowableStrategy, false, false);
    }

    public static HSPoolExecutor newTranssionExecutor() {
        return new HSPoolExecutor(20, "zero", HSPoolExecutor.UncaughtThrowableStrategy.DEFAULT, false, false);
    }

    public static HSPoolExecutor newTranssionExecutor(int maxPoolSize) {
        if (maxPoolSize <= 0) {
            maxPoolSize = 20;
        }

        return new HSPoolExecutor(maxPoolSize, "zero", HSPoolExecutor.UncaughtThrowableStrategy.DEFAULT, false, false);
    }

    public HSPoolExecutor(int maxPoolSize, String name, HSPoolExecutor.UncaughtThrowableStrategy uncaughtThrowableStrategy, boolean preventNetworkOperations, boolean executeSynchronously) {
        super(0, Integer.MAX_VALUE, 60L, TimeUnit.MILLISECONDS, new SynchronousQueue(), new HSPoolExecutor.DefaultThreadFactory(name, uncaughtThrowableStrategy, preventNetworkOperations));
        this.executeSynchronously = executeSynchronously;

        try {
            Log.d("ThreadPool", "size is :" + this.getPoolSize());
        } catch (Exception var7) {
            var7.printStackTrace();
        }

    }

    public void execute(Runnable command) {
        if (this.executeSynchronously) {
            command.run();
        } else {
            super.execute(command);
        }

    }

    @NonNull
    public Future<?> submit(Runnable task) {
        return this.maybeWait(super.submit(task));
    }

    private <T> Future<T> maybeWait(Future<T> future) {
        if (this.executeSynchronously) {
            try {
                future.get();
            } catch (ExecutionException | InterruptedException var3) {
                throw new RuntimeException(var3);
            }
        }

        return future;
    }

    @NonNull
    public <T> Future<T> submit(Runnable task, T result) {
        return this.maybeWait(super.submit(task, result));
    }

    public <T> Future<T> submit(Callable<T> task) {
        return this.maybeWait(super.submit(task));
    }

    protected <T> RunnableFuture<T> newTaskFor(Runnable runnable, T value) {
        return new HSPoolExecutor.ComparableFutureTask(runnable, value);
    }

    protected <T> RunnableFuture<T> newTaskFor(Callable<T> callable) {
        return new HSPoolExecutor.ComparableFutureTask(callable);
    }

    public static int calculateBestThreadCount() {
        File[] cpus = null;

        try {
            File cpuInfo = new File("/sys/devices/system/cpu/");
            final Pattern cpuNamePattern = Pattern.compile("cpu[0-9]+");
            cpus = cpuInfo.listFiles(new FilenameFilter() {
                public boolean accept(File file, String s) {
                    return cpuNamePattern.matcher(s).matches();
                }
            });
        } catch (Throwable var3) {
            Log.d("ZeroPoolExecutor", "Failed to calculate accurate cpu count");
        }

        int cpuCount = cpus != null ? cpus.length : 0;
        int availableProcessors = Math.max(1, Runtime.getRuntime().availableProcessors());
        return Math.min(4, Math.max(availableProcessors, cpuCount));
    }

    private static final class DefaultThreadFactory implements ThreadFactory {
        private final String name;
        private final HSPoolExecutor.UncaughtThrowableStrategy uncaughtThrowableStrategy;
        private final boolean preventNetworkOperations;
        private int threadNum;

        DefaultThreadFactory(String name, HSPoolExecutor.UncaughtThrowableStrategy uncaughtThrowableStrategy, boolean preventNetworkOperations) {
            this.name = name;
            this.uncaughtThrowableStrategy = uncaughtThrowableStrategy;
            this.preventNetworkOperations = preventNetworkOperations;
        }

        public synchronized Thread newThread(@NonNull Runnable runnable) {
            Thread result = new Thread(runnable, "zero-" + this.name + "-thread-" + this.threadNum) {
                public void run() {
                    Process.setThreadPriority(10);
                    if (HSPoolExecutor.DefaultThreadFactory.this.preventNetworkOperations) {
                        StrictMode.setThreadPolicy((new StrictMode.ThreadPolicy.Builder()).detectNetwork().penaltyDeath().build());
                    }

                    try {
                        super.run();
                    } catch (Throwable var2) {
                        HSPoolExecutor.DefaultThreadFactory.this.uncaughtThrowableStrategy.handle(var2);
                    }

                }
            };
            ++this.threadNum;
            return result;
        }
    }

    public static enum UncaughtThrowableStrategy {
        IGNORE,
        LOG {
            protected void handle(Throwable t) {
                Log.d("ZeroPoolExecutor", "Request threw uncaught throwable");
            }
        },
        THROW {
            protected void handle(Throwable t) {
                super.handle(t);
                if (t != null) {
                    throw new RuntimeException("Request threw uncaught throwable", t);
                }
            }
        };

        public static final HSPoolExecutor.UncaughtThrowableStrategy DEFAULT = LOG;

        private UncaughtThrowableStrategy() {
        }

        protected void handle(Throwable t) {
        }
    }

    protected class ComparableFutureTask<V> extends FutureTask<V> implements Comparable<HSPoolExecutor.ComparableFutureTask<V>> {
        private Object object;

        public ComparableFutureTask(Callable<V> callable) {
            super(callable);
            this.object = callable;
        }

        public ComparableFutureTask(Runnable runnable, V result) {
            super(runnable, result);
            this.object = runnable;
        }

        public int compareTo(HSPoolExecutor.ComparableFutureTask<V> o) {
            if (this == o) {
                return 0;
            } else if (o == null) {
                return -1;
            } else {
                return this.object != null && o.object != null && this.object.getClass().equals(o.object.getClass()) && this.object instanceof Comparable ? ((Comparable)this.object).compareTo(o.object) : 0;
            }
        }
    }
}

