package com.cloud.sdk.commonutil.util

import android.os.Handler
import android.os.HandlerThread

/**
 * HandlerThread 工厂类，负责创建和管理 HandlerThread 对象及其关联的 Handler。
 */
object HandlerThreadFactory {
    private val handlerThreadMap: MutableMap<String, HandlerThread> = HashMap()
    private val handlerMap: MutableMap<String, Handler> = HashMap()

    /**
     * 根据线程名称创建或获取 HandlerThread 对象。
     *
     * @param threadName 线程名称
     * @return HandlerThread 对象
     */
    @Synchronized
    private fun getHandlerThread(threadName: String): HandlerThread {
        var handlerThread = handlerThreadMap[threadName]
        if (handlerThread == null) {
            handlerThread = HandlerThread(threadName)
            handlerThread.start()
            handlerThreadMap[threadName] = handlerThread
        }
        return handlerThread
    }

    /**
     * 根据线程名称创建或获取与 HandlerThread 关联的 Handler 对象。
     *
     * @param threadName 线程名称
     * @return Handler 对象
     */
    @Synchronized
    fun getHandler(threadName: String): Handler? {
        var handler = handlerMap[threadName]
        if (handler == null) {
            val handlerThread = getHandlerThread(threadName)
            val looper = handlerThread.looper
            if (looper != null) {
                handler = Handler(looper)
                handlerMap[threadName] = handler
            }
        }
        return handler
    }

    /**
     * 退出指定名称的 HandlerThread，并移除对应的 Handler 和 HandlerThread 对象。
     *
     * @param threadName 线程名称
     */
    @Synchronized
    fun quitHandlerThread(threadName: String) {
        val handlerThread = handlerThreadMap[threadName]
        if (handlerThread != null) {
            handlerThread.quitSafely()
            handlerThreadMap.remove(threadName)
        }
        handlerMap.remove(threadName)
    }

    /**
     * 退出所有的 HandlerThread，并移除所有的 Handler 和 HandlerThread 对象。
     */
    @Synchronized
    fun quitAllHandlerThreads() {
        for (handlerThread in handlerThreadMap.values) {
            handlerThread.quitSafely()
        }
        handlerThreadMap.clear()
        handlerMap.clear()
    }
}