package com.cloud.hisavana.sdk.common.tracking;

import android.os.Build;
import android.text.TextUtils;

import com.cloud.hisavana.sdk.BuildConfig;
import com.cloud.hisavana.sdk.api.config.SspAd;
import com.cloud.hisavana.sdk.common.constant.TaErrorCode;
import com.cloud.hisavana.sdk.common.util.AESUtils;
import com.cloud.hisavana.sdk.common.util.AdLogUtil;
import com.cloud.hisavana.sdk.common.util.DeviceUtil;
import com.cloud.hisavana.sdk.common.util.GPSTracker;
import com.cloud.hisavana.sdk.common.util.MitNetUtil;
import com.cloud.hisavana.sdk.common.util.Preconditions;
import com.cloud.hisavana.sdk.common.util.UserAgentUtil;
import com.transsion.core.deviceinfo.DeviceInfo;
import com.transsion.core.pool.TranssionPoolManager;
import com.transsion.core.utils.AppUtil;
import com.transsion.http.HttpClient;
import com.transsion.http.impl.StringCallback;

import java.util.List;

/**
 * 落地页跟踪
 */
public class TrackingManager {

    /**
     * 超时时间
     */
    private static final int TIMEOUT_IN_MILLIONS = 15 * 1000;


    // =============================================================================================


    /**
     * 当广告被点击后，触发此字段中的 clickTrackingUrls
     */
    public static void clickTrackingUrlsExposure(List<String> urls) {
        if (urls == null || urls.size() < 1) {
            AdLogUtil.trackLog("clickTrackingUrls --> urls==null || urls.size()=<1 --> return ");
            return;
        }

        // 发起多个请求
        for (final String url : urls) {
            TranssionPoolManager.getInstance().addTask(new Runnable() {
                @Override
                public void run() {
                    sendRequestToServer(url, null);
                }
            });
        }
    }


    // =============================================================================================


    /**
     * deepLinkUrl  应用深链接地址
     * clickUrl     点击链接地址
     */
    public static void deepLinkUrlExposure(String url, DownUpPointBean pointBean, TrackingCallBack callBack) {
        if (null == url || TextUtils.isEmpty(url) || null == pointBean) {
            AdLogUtil.trackLog("deepLinkUrlExposure --> null == url || null == pointBean");
            return;
        }

        // 发起请求
        sendRequestToServer(processUrl(url, pointBean), callBack);
    }


    // =============================================================================================


    /**
     * 发起网络请求
     *
     * @param url
     */
    private static void sendRequestToServer(String url, final TrackingCallBack callBack) {
        AdLogUtil.trackLog("sendRequestToServer --> url = " + url);

        if (url == null) {
            return;
        }
        HttpClient.get()
                .log(SspAd.isDebug())
                .connectTimeout(TIMEOUT_IN_MILLIONS)
                .readTimeout(TIMEOUT_IN_MILLIONS)
                .url(url)
                .addHeader("User-Agent", UserAgentUtil.getUserAgent())
                .build()
                .execute(new StringCallback(true) {
                             @Override
                             public void onSuccess(final int statusCode, final String response) {
                                 AdLogUtil.trackLog("sendRequestToServer --> onSuccess --> statusCode = " + statusCode + " response=" + response);
                                 //AdLogUtil.trackLog("sendRequestToServer --> onSuccess --> statusCode = "+ statusCode + " response=");

                                 if (callBack == null) {
                                     return;
                                 }

                                 Preconditions.runOnMainThread(new Preconditions.Callback() {
                                     @Override
                                     public void onRun() {
                                         if (!TextUtils.isEmpty(response)) {
                                             callBack.onServerRequestSuccess(statusCode, response);
                                         } else {
                                             callBack.onServerRequestFailure(TaErrorCode.HTTP_RESPONSE_ERROR.getErrorCode(),
                                                     TaErrorCode.HTTP_RESPONSE_ERROR.getErrorMessage(), null);
                                         }
                                     }
                                 });

                             }

                             @Override
                             public void onFailure(final int statusCode, final String response, final Throwable throwable) {
                                 AdLogUtil.trackLog("sendRequestToServer --> onFailure --> statusCode = " + statusCode + " response = " + response);
                                 if (callBack != null) {
                                     Preconditions.runOnMainThread(new Preconditions.Callback() {
                                         @Override
                                         public void onRun() {
                                             callBack.onServerRequestFailure(statusCode, response, throwable);
                                         }
                                     });
                                 }
                             }
                         }
                );
    }

    /**
     * 将结果 导出
     */
    public interface TrackingCallBack {

        /**
         * 请求失败
         */
        void onServerRequestFailure(int statusCode, String responseBody, Throwable throwable);

        /**
         * 请求成功
         */
        void onServerRequestSuccess(int statusCode, String response);

    }


    // =============================================================================================


    /**
     * 处理 url
     * <p>
     * xd=_X-DOWN_&yd=_Y-DOWN_&xu=_X-UP_&yu=_Y-UP_&ai=_APPID_&pn=_PACKAGE-NAME_&ve=_VERSION_&sv=_SDK-VERSION_&ot=_OS-TYPE_&ov=_OS-VERSION_&nc=_NET-CONN-TYPE_&op=_OPERATOR-TYPE_&ga=_GAID_&dt=_DEVICE-TYPE_&br=_BRAND_&mo=_MODEL_&ma=_MAKER_&la=_LATITUDE_&lo=_LONGITUDE_&iw=_IMAGE-WIDTH_&ih=_IMAGE-HEIGHT_
     */
    public static String processUrl(String url, DownUpPointBean pointBean) {

        if (TextUtils.isEmpty(url) || null == pointBean) {
            return url;
        }

        // 组装Url
        AdLogUtil.trackLog("TrackingManager --> processUrl --> 处理前 url = " + url);
        AdLogUtil.trackLog("TrackingManager --> processUrl --> pointBean = " + pointBean.toString());

        StringBuilder sb = new StringBuilder();
        sb.append("xd=" + pointBean.getDownX()); // X-DOWN         点击坐标（按）     String
        sb.append("&yd=" + pointBean.getDownY()); // Y-DOWN         点击坐标（按）     String
        sb.append("&xu=" + pointBean.getUpX()); // X-UP           点击坐标（弹）     String
        sb.append("&yu=" + pointBean.getUpY()); // Y-UP           点击坐标（弹）     String
        sb.append("&ai=" + SspAd.AppId); //  APPID          点击的应用id     String

        sb.append("&pn=" + AppUtil.getPkgName()); //  PACKAGE-NAME   点击的app包名    String
        sb.append("&ve=" + AppUtil.getVersionName()); // VERSION        点击的app版本号   String
        sb.append("&sv=" + BuildConfig.VERSION_NAME); // SDK-VERSION    点击的sdk版本号   String
        sb.append("&ot=" + 1); // OS-TYPE        操作系统类型      String //1、Android；2、iOS
        sb.append("&ov=" + DeviceUtil.getSystemVersion()); // OS-VERSION     操作系统版本号     String

        sb.append("&nc=" + MitNetUtil.getNetType()); //  NET-CONN-TYPE  网络连接类型      String
        sb.append("&op=" + DeviceUtil.getMNC()+DeviceUtil.getMCC()); // OPERATOR-TYPE  运营商类型       String
        sb.append("&ga=" + DeviceUtil.getGAId()); // GAID           点击时的gaid    String
        sb.append("&dt=" + (DeviceInfo.isPad() ? 2 : 1)); // DEVICE-TYPE    设备类型        String
        sb.append("&br=" + Build.BRAND); // BRAND          机型品牌        String
        sb.append("&mo=" + Build.MODEL); // MODEL          客户端手机型号     String
        sb.append("&ma=" + Build.MANUFACTURER); // MAKER          设备制造商        String

        GPSTracker gps = new GPSTracker();
        sb.append("&la=" + gps.getLatitude()); // LATITUDE       点击时的纬度      String
        sb.append("&lo=" + gps.getLongitude()); // LONGITUDE      点击时的经度      String

        sb.append("&iw=" + pointBean.getImageW()); // IMAGE-WIDTH    点击时素材宽      String
        sb.append("&ih=" + pointBean.getImageH()); // IMAGE-HEIGHT   点击时素材高      String

        AdLogUtil.trackLog("埋点字段 --> sb=" + sb.toString());
        url = url + "&p=" + AESUtils.encrypt(sb.toString());
        AdLogUtil.trackLog("TrackingManager --> processUrl --> 处理后 url = " + url);

        return url;
    }

}
