package com.cloud.hisavana.sdk.common.util;

import android.support.annotation.Nullable;

import com.transsion.core.utils.AppUtil;

import java.io.File;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.io.Serializable;

/**
 * 图片缓存帮助类
 * 包含内存缓存LruCache和磁盘缓存DiskLruCache
 *
 * @author Javen
 */
public class CacheUtil {

    /**
     * 缓存文件保存目录
     */
    private static final String CACHE_DIR = "ZERO_AD_SDK_LRUCACHE";
    /**
     * 硬盘缓存
     */
//    private static LruCache<String, Bitmap> mLruCache;

//    private static volatile CacheUtil cacheUtil;
    /**
     * 内存缓存
     */
    private static DiskLruCache mDiskLruCache;

    /**
     * 最多缓存100m文件，可以认为是永久缓存
     */
    private static final long MAX_SIZE = 100 * 1024 * 1024;


    /**
     * 构造方法
     */
    private CacheUtil() {
        /**
         *  获取应用可占内存的1/16作为缓存
         */
//        int maxSize = (int) (Runtime.getRuntime().maxMemory() / 16);
//        /**
//         * 实例化LruCaceh对象
//         */
//        mLruCache = new LruCache<String, Bitmap>(maxSize) {
//            @Override
//            protected int sizeOf(String key, Bitmap bitmap) {
//                return bitmap.getRowBytes() * bitmap.getHeight();
//            }
//        };
        try {
            /**
             *获取DiskLruCahce对象
             */
            mDiskLruCache = DiskLruCache.open(getCacheDir(), AppUtil.getVersionCode(), 1, MAX_SIZE);
        } catch (Throwable e) {
            e.printStackTrace();
        }
    }

    private static class CacheUtilInstance {
        private static final CacheUtil INSTANCE = new CacheUtil();
    }

    public static CacheUtil getInstance() {
        return CacheUtilInstance.INSTANCE;
    }

    /**
     * 获得磁盘的String值
     *
     * @param key
     * @return
     */
    @Nullable
    public synchronized String getString(String key) {
        if (mDiskLruCache == null || mDiskLruCache.isClosed()) {
            return null;
        }
        try {
            DiskLruCache.Snapshot snapshot = mDiskLruCache.get(key);
            if (snapshot != null) {
                String value = snapshot.getString(0);
                AdLogUtil.LOG.d( "get content successful.key=" + key + ", value=" + value);
                return value;
            }
        } catch (Throwable e) {
            AdLogUtil.LOG.e( "read cache error: " + e.toString());
        }
        return null;
    }

    /**
     * String加入磁盘缓存
     *
     * @param key
     * @param content
     */
    public synchronized void putString(String key, String content) {
        if (mDiskLruCache == null || mDiskLruCache.isClosed()) {
            return;
        }
        DiskLruCache.Editor editor = null;
        try {
            editor = mDiskLruCache.edit(key);
            if (editor != null) {
                editor.set(0, content);
                editor.commit();
                mDiskLruCache.flush();
                AdLogUtil.LOG.d( "put content successful. key=" + key + ", value=" + content);
            }
        } catch (Throwable e) {
            if (editor != null) {
                try {
                    editor.abort();
                } catch (Throwable e1) {
                    AdLogUtil.LOG.e( "write cache error: " + e.toString());
                }
            }
        }
    }


    /**
     * 存放可序列化对象
     *
     * @param key
     * @param value
     */
    public void putSerializable(String key, Serializable value) {
        DiskLruCache.Editor editor = null;
        ObjectOutputStream oos = null;
        try {
            editor = mDiskLruCache.edit(key);
            if (editor == null) {
                return;
            }
            OutputStream os = editor.newOutputStream(0);
            oos = new ObjectOutputStream(os);
            oos.writeObject(value);
            oos.flush();
            editor.commit();
        } catch (Throwable e) {
            e.printStackTrace();
            try {
                editor.abort();
            } catch (Throwable e1) {
                e1.printStackTrace();
            }
        } finally {
            try {
                if (oos != null) {
                    oos.close();
                }
            } catch (Throwable e) {
                e.printStackTrace();
            }
        }
    }

    public <T> T getAsSerializable(String key) {
        T t = null;
        ObjectInputStream ois = null;
        try {
            DiskLruCache.Snapshot snapshot = mDiskLruCache.get(key);
            if (snapshot == null) {
                return null;
            }
            InputStream is = snapshot.getInputStream(0);
            if (is == null) {
                return null;
            }
            ois = new ObjectInputStream(is);
            t = (T) ois.readObject();
        } catch (Throwable e) {
            e.printStackTrace();
        } finally {
            try {
                if (ois != null) {
                    ois.close();
                }
            } catch (Throwable e) {
                e.printStackTrace();
            }
        }
        return t;
    }


//    /**
//     * 从缓存（内存缓存，磁盘缓存）中获取Bitmap
//     */
//    public Bitmap getBitmap(String url) {
//        if (mLruCache.get(url) != null) {
//            // 从LruCache缓存中取
//            AdLogUtil.TAN.d( "从LruCahce获取");
//            return mLruCache.get(url);
//        } else {
//            try {
//                if (mDiskLruCache.get(url) != null) {
//                    // 从DiskLruCahce取
//                    DiskLruCache.Snapshot snapshot = mDiskLruCache.get(url);
//                    Bitmap bitmap = null;
//                    if (snapshot != null) {
//                        bitmap = BitmapFactory.decodeStream(snapshot.getInputStream(0));
////                        // 存入LruCache缓存
////                        mLruCache.put(url, bitmap);
//                        AdLogUtil.TAN.d( "从DiskLruCahce获取");
//                    }
//                    return bitmap;
//                }
//            } catch (IOException e) {
//                e.printStackTrace();
//            }
//        }
//        return null;
//    }

//    /**
//     * 存入缓存（内存缓存，磁盘缓存）
//     */
//    public void putBitmap(String url, Bitmap bitmap) {
//        // 存入LruCache缓存
//        mLruCache.put(url, bitmap);
//        // 判断是否存在DiskLruCache缓存，若没有存入
//        try {
//            if (mDiskLruCache.get(url) == null) {
//                DiskLruCache.Editor editor = mDiskLruCache.edit(url);
//                if (editor != null) {
//                    OutputStream outputStream = editor.newOutputStream(0);
//                    if (bitmap.compress(CompressFormat.JPEG, 100, outputStream)) {
//                        editor.commit();
//                    } else {
//                        editor.abort();
//                    }
//                }
//                mDiskLruCache.flush();
//            }
//        } catch (IOException e) {
//            e.printStackTrace();
//        }
//
//    }

    /**
     * 移除磁盘的key
     *
     * @param key
     */
    public void diskRemove(String key) {
        if (mDiskLruCache == null || mDiskLruCache.isClosed()) {
            return;
        }
        try {
            mDiskLruCache.remove(key);
        } catch (Throwable e) {
            e.printStackTrace();
        }
    }


    /**
     * 文件缓存目录
     *
     * @return
     */
    private File getCacheDir() {
        File dir = new File(FileUtil.getAppDateCachePath(), CACHE_DIR);
        if (!dir.exists()) {
            dir.mkdirs();
        }
        // TODO: 2018/5/18 修改
        return dir;
    }

}