/*
 * Decompiled with CFR 0.152.
 */
package com.cloud.hisavana.sdk.common.util;

import com.cloud.hisavana.sdk.common.util.AdLogUtil;
import java.io.BufferedInputStream;
import java.io.BufferedWriter;
import java.io.Closeable;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.StringWriter;
import java.io.Writer;
import java.lang.reflect.Array;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

public final class DiskLruCache
implements Closeable {
    static final String JOURNAL_FILE = "journal";
    static final String JOURNAL_FILE_TMP = "journal.tmp";
    static final String MAGIC = "libcore.io.DiskLruCache";
    static final String VERSION_1 = "1";
    static final long ANY_SEQUENCE_NUMBER = -1L;
    private static final String CLEAN = "CLEAN";
    private static final String DIRTY = "DIRTY";
    private static final String REMOVE = "REMOVE";
    private static final String READ = "READ";
    private static final Charset UTF_8 = Charset.forName("UTF-8");
    private static final int IO_BUFFER_SIZE = 8192;
    private final File directory;
    private final File journalFile;
    private final File journalFileTmp;
    private final int appVersion;
    private final long maxSize;
    private final int valueCount;
    private long size = 0L;
    private Writer journalWriter;
    private final LinkedHashMap<String, Entry> lruEntries = new LinkedHashMap(0, 0.75f, true);
    private int redundantOpCount;
    private long nextSequenceNumber = 0L;
    private final ExecutorService executorService = new ThreadPoolExecutor(0, 1, 60L, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>());
    private final Callable<Void> cleanupCallable = new Callable<Void>(){

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public Void call() throws Exception {
            DiskLruCache diskLruCache = DiskLruCache.this;
            synchronized (diskLruCache) {
                if (DiskLruCache.this.journalWriter == null) {
                    return null;
                }
                DiskLruCache.this.trimToSize();
                if (DiskLruCache.this.journalRebuildRequired()) {
                    DiskLruCache.this.rebuildJournal();
                    DiskLruCache.this.redundantOpCount = 0;
                }
            }
            return null;
        }
    };

    private static <T> T[] copyOfRange(T[] original, int start, int end) {
        int originalLength = original.length;
        if (start > end) {
            throw new IllegalArgumentException();
        }
        if (start < 0 || start > originalLength) {
            throw new ArrayIndexOutOfBoundsException();
        }
        int resultLength = end - start;
        int copyLength = Math.min(resultLength, originalLength - start);
        Object[] result = (Object[])Array.newInstance(original.getClass().getComponentType(), resultLength);
        System.arraycopy(original, start, result, 0, copyLength);
        return result;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static String readFully(Reader reader) throws IOException {
        try {
            int count;
            StringWriter writer = new StringWriter();
            char[] buffer = new char[1024];
            while ((count = reader.read(buffer)) != -1) {
                writer.write(buffer, 0, count);
            }
            String string = writer.toString();
            return string;
        }
        finally {
            reader.close();
        }
    }

    public static String readAsciiLine(InputStream in) throws IOException {
        StringBuilder result = new StringBuilder(80);
        while (true) {
            int c2;
            if ((c2 = in.read()) == -1) {
                throw new EOFException();
            }
            if (c2 == 10) break;
            result.append((char)c2);
        }
        int length = result.length();
        if (length > 0 && result.charAt(length - 1) == '\r') {
            result.setLength(length - 1);
        }
        return result.toString();
    }

    public static void closeQuietly(Closeable closeable) {
        if (closeable != null) {
            try {
                closeable.close();
            }
            catch (RuntimeException rethrown) {
                throw rethrown;
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
    }

    public static void deleteContents(File dir) throws IOException {
        File[] files = dir.listFiles();
        if (files == null) {
            AdLogUtil.LOG.d((Object)("not a directory: " + dir));
            return;
        }
        for (File file : files) {
            if (file.isDirectory()) {
                DiskLruCache.deleteContents(file);
            }
            if (file.delete()) continue;
            throw new IOException("failed to delete file: " + file);
        }
    }

    private DiskLruCache(File directory, int appVersion, int valueCount, long maxSize) {
        this.directory = directory;
        this.appVersion = appVersion;
        this.journalFile = new File(directory, JOURNAL_FILE);
        this.journalFileTmp = new File(directory, JOURNAL_FILE_TMP);
        this.valueCount = valueCount;
        this.maxSize = maxSize;
    }

    public static DiskLruCache open(File directory, int appVersion, int valueCount, long maxSize) throws IOException {
        if (maxSize <= 0L) {
            throw new IllegalArgumentException("maxSize <= 0");
        }
        if (valueCount <= 0) {
            throw new IllegalArgumentException("valueCount <= 0");
        }
        DiskLruCache cache = new DiskLruCache(directory, appVersion, valueCount, maxSize);
        if (cache.journalFile.exists()) {
            try {
                cache.readJournal();
                cache.processJournal();
                cache.journalWriter = new BufferedWriter(new FileWriter(cache.journalFile, true), 8192);
                return cache;
            }
            catch (IOException journalIsCorrupt) {
                cache.delete();
            }
        }
        directory.mkdirs();
        cache = new DiskLruCache(directory, appVersion, valueCount, maxSize);
        cache.rebuildJournal();
        return cache;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void readJournal() throws IOException {
        BufferedInputStream in = new BufferedInputStream(new FileInputStream(this.journalFile), 8192);
        try {
            String magic = DiskLruCache.readAsciiLine(in);
            String version = DiskLruCache.readAsciiLine(in);
            String appVersionString = DiskLruCache.readAsciiLine(in);
            String valueCountString = DiskLruCache.readAsciiLine(in);
            String blank = DiskLruCache.readAsciiLine(in);
            if (!(MAGIC.equals(magic) && VERSION_1.equals(version) && Integer.toString(this.appVersion).equals(appVersionString) && Integer.toString(this.valueCount).equals(valueCountString) && "".equals(blank))) {
                throw new IOException("unexpected journal header: [" + magic + ", " + version + ", " + valueCountString + ", " + blank + "]");
            }
            boolean isLoop = true;
            while (isLoop) {
                try {
                    this.readJournalLine(DiskLruCache.readAsciiLine(in));
                }
                catch (Throwable endOfJournal) {
                    // empty catch block
                    break;
                }
            }
        }
        finally {
            DiskLruCache.closeQuietly(in);
        }
    }

    private void readJournalLine(String line) throws IOException {
        String[] parts = line.split(" ");
        if (parts.length < 2) {
            throw new IOException("unexpected journal line: " + line);
        }
        String key = parts[1];
        if (parts[0].equals(REMOVE) && parts.length == 2) {
            this.lruEntries.remove(key);
            return;
        }
        Entry entry = this.lruEntries.get(key);
        if (entry == null) {
            entry = new Entry(key);
            this.lruEntries.put(key, entry);
        }
        if (parts[0].equals(CLEAN) && parts.length == 2 + this.valueCount) {
            entry.readable = true;
            entry.currentEditor = null;
            entry.setLengths(DiskLruCache.copyOfRange(parts, 2, parts.length));
        } else if (parts[0].equals(DIRTY) && parts.length == 2) {
            entry.currentEditor = new Editor(entry);
        } else if (!parts[0].equals(READ) || parts.length != 2) {
            throw new IOException("unexpected journal line: " + line);
        }
    }

    private void processJournal() throws IOException {
        DiskLruCache.deleteIfExists(this.journalFileTmp);
        Iterator<Entry> i2 = this.lruEntries.values().iterator();
        while (i2.hasNext()) {
            int t;
            Entry entry = i2.next();
            if (entry.currentEditor == null) {
                for (t = 0; t < this.valueCount; ++t) {
                    this.size += entry.lengths[t];
                }
                continue;
            }
            entry.currentEditor = null;
            for (t = 0; t < this.valueCount; ++t) {
                DiskLruCache.deleteIfExists(entry.getCleanFile(t));
                DiskLruCache.deleteIfExists(entry.getDirtyFile(t));
            }
            i2.remove();
        }
    }

    private synchronized void rebuildJournal() throws IOException {
        if (this.journalWriter != null) {
            this.journalWriter.close();
        }
        BufferedWriter writer = new BufferedWriter(new FileWriter(this.journalFileTmp), 8192);
        writer.write(MAGIC);
        writer.write("\n");
        writer.write(VERSION_1);
        writer.write("\n");
        writer.write(Integer.toString(this.appVersion));
        writer.write("\n");
        writer.write(Integer.toString(this.valueCount));
        writer.write("\n");
        writer.write("\n");
        for (Entry entry : this.lruEntries.values()) {
            if (entry.currentEditor != null) {
                writer.write("DIRTY " + entry.key + '\n');
                continue;
            }
            writer.write("CLEAN " + entry.key + entry.getLengths() + '\n');
        }
        ((Writer)writer).close();
        this.journalFileTmp.renameTo(this.journalFile);
        this.journalWriter = new BufferedWriter(new FileWriter(this.journalFile, true), 8192);
    }

    private static void deleteIfExists(File file) throws IOException {
        if (file.exists() && !file.delete()) {
            throw new IOException();
        }
    }

    public synchronized Snapshot get(String key) throws IOException {
        this.checkNotClosed();
        this.validateKey(key);
        Entry entry = this.lruEntries.get(key);
        if (entry == null) {
            return null;
        }
        if (!entry.readable) {
            return null;
        }
        InputStream[] ins = new InputStream[this.valueCount];
        try {
            for (int i2 = 0; i2 < this.valueCount; ++i2) {
                ins[i2] = new FileInputStream(entry.getCleanFile(i2));
            }
        }
        catch (Throwable e2) {
            return null;
        }
        ++this.redundantOpCount;
        this.journalWriter.append("READ " + key + '\n');
        if (this.journalRebuildRequired()) {
            this.executorService.submit(this.cleanupCallable);
        }
        return new Snapshot(key, entry.sequenceNumber, ins);
    }

    public Editor edit(String key) throws IOException {
        return this.edit(key, -1L);
    }

    private synchronized Editor edit(String key, long expectedSequenceNumber) throws IOException {
        this.checkNotClosed();
        this.validateKey(key);
        Entry entry = this.lruEntries.get(key);
        if (expectedSequenceNumber != -1L && (entry == null || entry.sequenceNumber != expectedSequenceNumber)) {
            return null;
        }
        if (entry == null) {
            entry = new Entry(key);
            this.lruEntries.put(key, entry);
        } else if (entry.currentEditor != null) {
            return null;
        }
        Editor editor = new Editor(entry);
        entry.currentEditor = editor;
        this.journalWriter.write("DIRTY " + key + '\n');
        this.journalWriter.flush();
        return editor;
    }

    public File getDirectory() {
        return this.directory;
    }

    public long maxSize() {
        return this.maxSize;
    }

    public synchronized long size() {
        return this.size;
    }

    private synchronized void completeEdit(Editor editor, boolean success) throws IOException {
        int i2;
        Entry entry = editor.entry;
        if (entry.currentEditor != editor) {
            throw new IllegalStateException();
        }
        if (success && !entry.readable) {
            for (i2 = 0; i2 < this.valueCount; ++i2) {
                if (entry.getDirtyFile(i2).exists()) continue;
                editor.abort();
                return;
            }
        }
        for (i2 = 0; i2 < this.valueCount; ++i2) {
            File dirty = entry.getDirtyFile(i2);
            if (success) {
                long newLength;
                if (!dirty.exists()) continue;
                File clean = entry.getCleanFile(i2);
                dirty.renameTo(clean);
                long oldLength = entry.lengths[i2];
                ((Entry)entry).lengths[i2] = newLength = clean.length();
                this.size = this.size - oldLength + newLength;
                continue;
            }
            DiskLruCache.deleteIfExists(dirty);
        }
        ++this.redundantOpCount;
        entry.currentEditor = null;
        if (entry.readable | success) {
            entry.readable = true;
            this.journalWriter.write("CLEAN " + entry.key + entry.getLengths() + '\n');
            if (success) {
                entry.sequenceNumber = this.nextSequenceNumber++;
            }
        } else {
            this.lruEntries.remove(entry.key);
            this.journalWriter.write("REMOVE " + entry.key + '\n');
        }
        if (this.size > this.maxSize || this.journalRebuildRequired()) {
            this.executorService.submit(this.cleanupCallable);
        }
    }

    private boolean journalRebuildRequired() {
        int REDUNDANT_OP_COMPACT_THRESHOLD = 2000;
        return this.redundantOpCount >= 2000 && this.redundantOpCount >= this.lruEntries.size();
    }

    public synchronized boolean remove(String key) throws IOException {
        this.checkNotClosed();
        this.validateKey(key);
        Entry entry = this.lruEntries.get(key);
        if (entry == null || entry.currentEditor != null) {
            return false;
        }
        for (int i2 = 0; i2 < this.valueCount; ++i2) {
            File file = entry.getCleanFile(i2);
            if (!file.delete()) {
                throw new IOException("failed to delete " + file);
            }
            this.size -= entry.lengths[i2];
            ((Entry)entry).lengths[i2] = 0L;
        }
        ++this.redundantOpCount;
        this.journalWriter.append("REMOVE " + key + '\n');
        this.lruEntries.remove(key);
        if (this.journalRebuildRequired()) {
            this.executorService.submit(this.cleanupCallable);
        }
        return true;
    }

    public boolean isClosed() {
        return this.journalWriter == null;
    }

    private void checkNotClosed() {
        if (this.journalWriter == null) {
            throw new IllegalStateException("cache is closed");
        }
    }

    public synchronized void flush() throws IOException {
        this.checkNotClosed();
        this.trimToSize();
        this.journalWriter.flush();
    }

    @Override
    public synchronized void close() throws IOException {
        if (this.journalWriter == null) {
            return;
        }
        for (Entry entry : new ArrayList<Entry>(this.lruEntries.values())) {
            if (entry.currentEditor == null) continue;
            entry.currentEditor.abort();
        }
        this.trimToSize();
        this.journalWriter.close();
        this.journalWriter = null;
    }

    private void trimToSize() throws IOException {
        while (this.size > this.maxSize) {
            Map.Entry<String, Entry> toEvict = this.lruEntries.entrySet().iterator().next();
            this.remove(toEvict.getKey());
        }
    }

    public void delete() throws IOException {
        this.close();
        DiskLruCache.deleteContents(this.directory);
    }

    private void validateKey(String key) {
        if (key.contains(" ") || key.contains("\n") || key.contains("\r")) {
            throw new IllegalArgumentException("keys must not contain spaces or newlines: \"" + key + "\"");
        }
    }

    private static String inputStreamToString(InputStream in) throws IOException {
        return DiskLruCache.readFully(new InputStreamReader(in, UTF_8));
    }

    private final class Entry {
        private final String key;
        private final long[] lengths;
        private boolean readable;
        private Editor currentEditor;
        private long sequenceNumber;

        private Entry(String key) {
            this.key = key;
            this.lengths = new long[DiskLruCache.this.valueCount];
        }

        public String getLengths() throws IOException {
            StringBuilder result = new StringBuilder();
            for (long size : this.lengths) {
                result.append(' ').append(size);
            }
            return result.toString();
        }

        private void setLengths(String[] strings) throws IOException {
            if (strings.length != DiskLruCache.this.valueCount) {
                throw this.invalidLengths(strings);
            }
            try {
                for (int i2 = 0; i2 < strings.length; ++i2) {
                    this.lengths[i2] = Long.parseLong(strings[i2]);
                }
            }
            catch (NumberFormatException e2) {
                throw this.invalidLengths(strings);
            }
        }

        private IOException invalidLengths(String[] strings) throws IOException {
            throw new IOException("unexpected journal line: " + Arrays.toString(strings));
        }

        public File getCleanFile(int i2) {
            return new File(DiskLruCache.this.directory, this.key + "." + i2);
        }

        public File getDirtyFile(int i2) {
            return new File(DiskLruCache.this.directory, this.key + "." + i2 + ".tmp");
        }
    }

    public final class Editor {
        private final Entry entry;
        private boolean hasErrors;

        private Editor(Entry entry) {
            this.entry = entry;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public InputStream newInputStream(int index) throws IOException {
            DiskLruCache diskLruCache = DiskLruCache.this;
            synchronized (diskLruCache) {
                if (this.entry.currentEditor != this) {
                    throw new IllegalStateException();
                }
                if (!this.entry.readable) {
                    return null;
                }
                return new FileInputStream(this.entry.getCleanFile(index));
            }
        }

        public String getString(int index) throws IOException {
            InputStream in = this.newInputStream(index);
            return in != null ? DiskLruCache.inputStreamToString(in) : null;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public OutputStream newOutputStream(int index) throws IOException {
            DiskLruCache diskLruCache = DiskLruCache.this;
            synchronized (diskLruCache) {
                if (this.entry.currentEditor != this) {
                    throw new IllegalStateException();
                }
                return new FaultHidingOutputStream(new FileOutputStream(this.entry.getDirtyFile(index)));
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void set(int index, String value) throws IOException {
            OutputStreamWriter writer = null;
            try {
                writer = new OutputStreamWriter(this.newOutputStream(index), UTF_8);
                writer.write(value);
            }
            catch (Throwable throwable) {
                DiskLruCache.closeQuietly(writer);
                throw throwable;
            }
            DiskLruCache.closeQuietly(writer);
        }

        public void commit() throws IOException {
            if (this.hasErrors) {
                DiskLruCache.this.completeEdit(this, false);
                DiskLruCache.this.remove(this.entry.key);
            } else {
                DiskLruCache.this.completeEdit(this, true);
            }
        }

        public void abort() throws IOException {
            DiskLruCache.this.completeEdit(this, false);
        }

        private class FaultHidingOutputStream
        extends FilterOutputStream {
            private FaultHidingOutputStream(OutputStream out) {
                super(out);
            }

            @Override
            public void write(int oneByte) {
                try {
                    this.out.write(oneByte);
                }
                catch (Throwable e2) {
                    Editor.this.hasErrors = true;
                }
            }

            @Override
            public void write(byte[] buffer, int offset, int length) {
                try {
                    this.out.write(buffer, offset, length);
                }
                catch (Throwable e2) {
                    Editor.this.hasErrors = true;
                }
            }

            @Override
            public void close() {
                try {
                    this.out.close();
                }
                catch (Throwable e2) {
                    Editor.this.hasErrors = true;
                }
            }

            @Override
            public void flush() {
                try {
                    this.out.flush();
                }
                catch (Throwable e2) {
                    Editor.this.hasErrors = true;
                }
            }
        }
    }

    public final class Snapshot
    implements Closeable {
        private final String key;
        private final long sequenceNumber;
        private final InputStream[] ins;

        private Snapshot(String key, long sequenceNumber, InputStream[] ins) {
            this.key = key;
            this.sequenceNumber = sequenceNumber;
            this.ins = ins;
        }

        public Editor edit() throws IOException {
            return DiskLruCache.this.edit(this.key, this.sequenceNumber);
        }

        public InputStream getInputStream(int index) {
            return this.ins[index];
        }

        public String getString(int index) throws IOException {
            return DiskLruCache.inputStreamToString(this.getInputStream(index));
        }

        @Override
        public void close() {
            for (InputStream in : this.ins) {
                DiskLruCache.closeQuietly(in);
            }
        }
    }
}

