package com.cloud.hisavana.sdk.internal.base;

import android.support.annotation.NonNull;

import com.cloud.hisavana.sdk.BuildConfig;
import com.cloud.hisavana.sdk.ad.base.AdxDefault;
import com.cloud.hisavana.sdk.data.bean.response.AdsDTO;
import com.cloud.hisavana.sdk.api.listener.TaRequest;
import com.cloud.hisavana.sdk.common.bean.TaNativeInfo;
import com.cloud.hisavana.sdk.common.callback.InternalAdListener;
import com.cloud.hisavana.sdk.common.constant.TaErrorCode;
import com.cloud.hisavana.sdk.common.tracking.DownUpPointBean;
import com.cloud.hisavana.sdk.common.util.AdLogUtil;
import com.cloud.hisavana.sdk.common.util.Preconditions;
import com.cloud.hisavana.sdk.common.util.RunTimer;
import com.cloud.hisavana.sdk.internal.detach.AdInfoTransfer;
import com.transsion.core.utils.NetUtil;

import java.util.ArrayList;
import java.util.List;

/**
 * Created by jianbing.yu on 2019/3/29.
 * <p>
 * 插屏 Banner native
 */
public abstract class BaseAd {

    protected String mPlacementId;
    protected boolean isDefaultAd = false;
    /**
     * 超时相关
     */
    private RunTimer runTimer = new RunTimer();

    protected boolean isTimeout = false; // 超时标志 --> 即使后面请求回来了 也不执行后面的业务逻辑了
    protected TaRequest mTaRequest = new TaRequest.TaRequestBuild().build();
    protected String mRid = "";

    public static final int LOAD = 1;
    public static final int RESPONSE = 2;
    public static final int FINISHED = 3;
    protected int mStep = LOAD;

    public String mSdkVersion = "";

    protected boolean mLoaded = false; // load成功
    protected boolean mShowed = false; // show成功
    public boolean mClicked = false; // click成功

    protected boolean oneByOne = false;
    protected String mRequestId;

    // =============================================================================================


    /**
     * 这个 监听器 设置给了 Adx 去请求广告 --> 拿到广告之后回调
     */
    protected InternalAdListener mInternalListener = new InternalAdListener() {

        @Override
        public void onAdResponse(List<AdsDTO> adBeanList) {
            if (isTimeout) {
                AdLogUtil.LOG.e("Request time out");
                return;
            }
            if (adBeanList == null && adBeanList.size() <= 0) {
                AdLogUtil.LOG.e("adList is empty");
                return;
            }
            onAdResponseStub(adBeanList);
            //Adx Tracker, load ad success
            if (adBeanList.get(0) != null) {
                mRid = adBeanList.get(0).rid;
                mStep = RESPONSE;
            }
        }

        @Override
        public void onSplashCacheGet(List<AdsDTO> adBeans) {
            if (isTimeout) {
                AdLogUtil.LOG.e("Request time out");
                return;
            }
            onSplashCacheGetStub(adBeans);
        }

        @Override
        public void onAdLoaded() {
            if (isTimeout) {
                AdLogUtil.LOG.e("Request time out");
                return;
            }
            if (runTimer != null) {
                runTimer.resetTimerTask();
            }
            onAdLoadedStub();
            mStep = FINISHED;
            mLoaded = true;

            if (mTaRequest != null && mTaRequest.getListener() != null) {
                AdLogUtil.LOG.d("onAdLoaded");
                mTaRequest.getListener().onAdLoaded();
            }
        }

        @Override
        public void onAdLoaded(List<TaNativeInfo> nativeInfos) {
            if (isTimeout) {
                AdLogUtil.LOG.d("Request time out");
                return;
            }
            if (runTimer != null) {
                runTimer.resetTimerTask();
            }
            onAdLoadedStub(nativeInfos);

            StringBuilder icon = new StringBuilder();
            StringBuilder image = new StringBuilder();
            for (TaNativeInfo nativeInfo : nativeInfos) {
                if (nativeInfo != null) {
                    icon.append(nativeInfo.getIconImage() != null && nativeInfo.getIconImage().isCached() ? "1" : "0");
                    image.append(nativeInfo.getImage() != null && nativeInfo.getImage().isCached() ? "1" : "0");
                    String mime = nativeInfo.getImage() != null ? nativeInfo.getImage().getMime() + "" : "1";
                    AdLogUtil.LOG.d("Track_native_fill=" + mime);
                }
            }
            mStep = FINISHED;
            mLoaded = true;

            if (mTaRequest != null && mTaRequest.getListener() != null && nativeInfos != null) {
                AdLogUtil.LOG.d("onAdLoaded TadNativeInfos size is:=" + nativeInfos.size());
                List<TaNativeInfo> adInfos = AdInfoTransfer.getAdInfo(nativeInfos);
                if (adInfos != null && nativeInfos.size() > 0) {
                    mTaRequest.getListener().onAdLoaded(adInfos);
                }
            }
        }

        @Override
        public void onError(TaErrorCode adError) {
            if (isTimeout) {
                AdLogUtil.LOG.e("Request time out");
                return;
            }
            if (runTimer != null) {
                runTimer.resetTimerTask();
            }
            onErrorStub(adError);
            mStep = FINISHED;

            if (mTaRequest != null && mTaRequest.getListener() != null) {
                AdLogUtil.LOG.e("adError：=" + adError.getErrorMessage());
                mTaRequest.getListener().onError(adError);
            }
        }

        @Override
        public void onAdClicked(DownUpPointBean pointBean) {
            onAdClickStub();
            mClicked = true;
            if (mTaRequest != null && mTaRequest.getListener() != null) {
                AdLogUtil.LOG.d("onAdClicked");
                mTaRequest.getListener().onAdClicked(pointBean);
            }
        }

        @Override
        public void onAdChoiceClick() {
            mClicked = true;
            if (mTaRequest != null && mTaRequest.getListener() != null) {
                AdLogUtil.LOG.d("onAdClicked");
                mTaRequest.getListener().onAdChoiceClick();
            }
        }

        @Override
        public void onAdClosed() {
            onAdClosedStub();
            if (mTaRequest != null && mTaRequest.getListener() != null) {
                AdLogUtil.LOG.d("onAdClosed");
                mTaRequest.getListener().onAdClosed();
            }
        }

        @Override
        public void onAdShow() {
            onAdShowStub();

            if (mTaRequest != null && mTaRequest.getListener() != null && !mShowed) {
                mShowed = true;
                AdLogUtil.LOG.d("onAdShow");
                mTaRequest.getListener().onAdShow();
            }
        }

        @Override
        public void onTimeOut() {
            onAdTimeoutStub();
        }

        @Override
        public void onMediaDownloaded(TaNativeInfo adInfo) {
            if (mTaRequest != null && mTaRequest.getListener() != null) {
                AdLogUtil.LOG.d("onMediaDownloaded");
                mTaRequest.getListener().onMediaDownloaded(adInfo);
            }
        }
    };


    /**
     * 定义一套回调 给子类使用
     * 网络请求回来之后 回调给子类
     */
    protected void onAdClickStub() {
    }

    protected void onAdLoadedStub() {
    }

    protected void onAdLoadedStub(List<TaNativeInfo> nativeInfos) {
    }

    protected void onAdClosedStub() {
    }

    protected void onAdTimeoutStub() {
        isTimeout = true;
        mStep = FINISHED;

        if (mTaRequest != null && mTaRequest.getListener() != null) {
            AdLogUtil.LOG.e("onTimeOut");
            mTaRequest.getListener().onTimeOut();
        }
    }

    protected void onSplashCacheGetStub(List<AdsDTO> adBeans) {
    }

    protected void onErrorStub(TaErrorCode adError) {
    }

    protected void onAdShowStub() {
    }

    protected void onAdResponseStub(List<AdsDTO> adBeanList) {

    }


    // =============================================================================================


    public BaseAd(int adType, String placementId) {
        this.mPlacementId = placementId;
        mSdkVersion = BuildConfig.VERSION_NAME;
    }


    // =============================================================================================


    public abstract void setPlacementId(String placementId);

    /**
     * 加载广告 各自子类实现加载 --> Adx 加载
     *
     * @return
     */
    protected abstract boolean loadPolyAd();

    //获取广告类型
    public abstract int getAdType();

    public String placementId() {
        return mPlacementId;
    }

    public void setDefaultAd(boolean defaultAd) {
        isDefaultAd = defaultAd;
    }

    public InternalAdListener adListener() {
        return mInternalListener;
    }

    public void setAdRequest(@NonNull TaRequest adRequest) {
        this.mTaRequest = adRequest;
    }

    // =============================================================================================

    public void loadAd(String requestId) {
        this.mRequestId = requestId;
        Preconditions.runOnMainThread(new Preconditions.Callback() {
            @Override
            public void onRun() {
                if (isDefaultAd) {
                    //请求打底广告
                    loadDefaultAd();
                } else {
                    loadAdInternal();
                }
            }
        });
    }

    public void loadAdInternal() {

        if (!NetUtil.checkNetworkState()) {
            if (mInternalListener != null) {
                mInternalListener.onError(TaErrorCode.NETWORK_ERROR);
            }
            return;
        }

        if (loadPolyAd()) {
            resetFlag();
            mStep = LOAD;
            runTimerTask();
        }
    }

    /**
     * 加载打底广告
     */
    private void loadDefaultAd() {
        // 从缓存中获取打底广告数据
        List<AdsDTO> adItems = AdxDefault.getDefaultAdItem(mPlacementId, true);
        if (mTaRequest != null && mTaRequest.getListener() != null) {
            if (adItems != null && adItems.size() > 0 && adItems.get(0) != null
                    //判断广告类型是否一样
                    && adItems.get(0).getCodeSeatType() == getAdType()) {
                AdLogUtil.LOG.d("onAdDefaultLoaded");
                List<AdsDTO> adBeanList = new ArrayList<>();
                adBeanList.add(adItems.get(0));
                onAdResponseStub(adBeanList);
            } else {
                AdLogUtil.LOG.d("onAdDefaultLoadedError");
                mTaRequest.getListener().onError(new TaErrorCode(100, "load default ad error"));
            }
        }
    }


    // =============================================================================================


    public void setOneByOne(boolean oneByOne) {
        this.oneByOne = oneByOne;
    }

    protected void onLoaded() {
        resetTimerTask();
        mStep = FINISHED;
        mLoaded = true;
    }

    private void resetFlag() {
        isTimeout = false;
        mLoaded = false;
        mShowed = false;
        mClicked = false;
    }

    /**
     * 启动定时器
     */
    protected void runTimerTask() {
        if (runTimer != null && mTaRequest != null) {
            int scheduleTime = mTaRequest.getScheduleTime();
            isTimeout = false;
            runTimer.resetTimerTask();
            runTimer.setTimeOutCallback(timeOutCallback);
            runTimer.setScheduleTime(scheduleTime);
            runTimer.runTimerTask();
        }
    }

    protected void resetTimerTask() {
        if (runTimer != null) {
            runTimer.resetTimerTask();
        }
    }

    /**
     * 超时回调
     */
    private RunTimer.TimeOutCallback timeOutCallback = new RunTimer.TimeOutCallback() {
        @Override
        public void isTimeOut() {
            if (mInternalListener != null) {
                mInternalListener.onTimeOut();
            }
        }
    };

    public void destroy() {
        if (mStep == LOAD || mStep == RESPONSE) {
            AdLogUtil.LOG.d("request is being cancel,current step is:" + mStep);
            mStep = FINISHED;
        }
        resetTimerTask();
        resetFlag();
        mTaRequest = null;
    }

}
