package com.cloud.hisavana.sdk.common.tranmeasure;

import android.support.annotation.NonNull;

import com.cloud.hisavana.sdk.common.util.AdLogUtil;

import java.util.ArrayList;
import java.util.List;

/**
 * 将每一个View的监控抽象成 Session
 * 当前就是Session的管理类
 */
public class MeasureSessionManager {

    /**
     * 所有正在监控中的 Session
     */
    private @NonNull
    List<MeasureSession> mRunningSessions = new ArrayList<>();

    /**
     * 最多同时监控10个
     */
    private final int MAX_SIZE = 10;


    // =============================================================================================


    private static MeasureSessionManager instance;

    public static MeasureSessionManager getInstance() {
        if (instance == null) {
            synchronized (MeasureSessionManager.class) {
                if (instance == null) {
                    instance = new MeasureSessionManager();
                }
            }
        }
        return instance;
    }


    // =============================================================================================


    /**
     * 获取正在 监控的 Session
     */
    protected List<MeasureSession> getRunningSessions() {
        return mRunningSessions;
    }

    /**
     * 获取MeasureSession,MeasureSession的View和Native的相关信息分开。
     */
    public <T> MeasureSession getMeasureSession(T key) {
        for (MeasureSession session : mRunningSessions) {
            if (session != null && session.isHit(key)) {
                AdLogUtil.measureLog("MeasureSessionManager have existed session");
                return session;
            }
        }
        AdLogUtil.measureLog("MeasureSession create new session");
        return new MeasureSession(key);
    }

    /**
     * 释放资源 --> 不在监控
     *
     * @param key 数据对象 唯一标识
     */
    public <T> void destroySession(T key) {
        if (null == key) {
            return;
        }

        // 便利所有正在监控的Session --> finish
        for (MeasureSession session : mRunningSessions) {
            if (session != null && session.isHit(key)) {
                AdLogUtil.measureLog("got existed session");
                session.finish();
                break;
            }
        }
    }

    /**
     * 监控 --> 测量当前监控的View是否符合要求
     */
    protected void measure() {
        // 便利所有的
        for (MeasureSession session : mRunningSessions) {
            if (session != null) {
                session.measure();
            }
        }
    }

    /**
     * 注册一个 需要监控的 Session
     */
    protected void registerSession(MeasureSession session) {
        AdLogUtil.measureLog("registerSession(注册一个 需要监控的 Session)");

        if (session != null && !mRunningSessions.contains(session)) {
            mRunningSessions.add(session);
            // 如果超过了最大值 --> 直接抛弃第一个
            if (mRunningSessions.size() > MAX_SIZE) {
                mRunningSessions.remove(0);
            }

            // 如果没有正在监控的Session --> 开启监控
            if (hasRunningSession()) {
                Monitor.getInstance().start();
            }
        }
    }

    /**
     * 注销监控
     */
    protected void unregisterSession(MeasureSession session) {
        AdLogUtil.measureLog("unregisterSession(注销监控)");

        if (session != null) {
            mRunningSessions.remove(session);
        }

        // 如果没有正在监控的Session --> 关闭监控
        if (!hasRunningSession()) {
            Monitor.getInstance().end();
        }
    }

    /**
     * 是否还有需要监控的Session了
     */
    private boolean hasRunningSession() {
        return mRunningSessions.size() > 0;
    }
}
