package com.cloud.hisavana.sdk.common.util;

import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.support.annotation.NonNull;
import android.text.TextUtils;
import android.widget.ImageView;

import com.cloud.hisavana.sdk.common.bean.AdImage;
import com.cloud.hisavana.sdk.common.constant.Constants;
import com.cloud.hisavana.sdk.common.constant.TaErrorCode;
import com.cloud.hisavana.sdk.common.http.DownLoadRequest;
import com.cloud.hisavana.sdk.common.http.listener.DrawableResponseListener;
import com.cloud.hisavana.sdk.data.bean.response.AdsDTO;
import com.transsion.core.pool.TranssionPoolManager;

import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * 广告图片下载工具类
 */
public class ImageDownloadHelper {

    /**
     * 加载单张图片
     *
     * @param url 图片url
     */
    public static void downloadImage(@NonNull final String url, @NonNull int isPreCache, AdsDTO adsDTO, int m_type,
                                     final ImageListener listener) {
        DownLoadRequest downLoadRequest = new DownLoadRequest().setPreCache(isPreCache).
                setListener(new DrawableResponseListener() {
                    @Override
                    public void onRequestSuccess(int statusCode, byte[] data, AdImage mediaBean) {
                        AdLogUtil.LOG.d("image cache success:" + url);
                        if (listener != null) listener.onImagesCached(data);
                    }

                    @Override
                    protected void onRequestError(TaErrorCode adError) {
                        AdLogUtil.LOG.w("image load fail,error code:" + adError.getErrorCode() + ",error message:" + adError.getErrorMessage());
                        if (listener != null) listener.onImagesFailedToCache(adError);
                    }
                }).setUrl(url).setAdsDTO(adsDTO, m_type);
        downLoadRequest.netRequestPreExecute();
    }

    /**
     * 加载单张图片
     *
     * @param url       图片url
     * @param imageView 待设置的View
     */
    public static void loadImageView(@NonNull final String url, @NonNull final ImageView imageView, AdsDTO adsDTO, int m_type, final ImageListener listener) {
        DownLoadRequest downLoadRequest = new DownLoadRequest().setListener(new DrawableResponseListener() {
            @Override
            public void onRequestSuccess(int statusCode, byte[] data,  AdImage mediaBean) {
                AdLogUtil.LOG.d("image load success " + url);
                if (mediaBean == null) {
                    return;
                }
                mediaBean.attachView(imageView);
                // 缓存成功
                if (listener != null) listener.onImagesCached(data);
            }

            @Override
            protected void onRequestError(TaErrorCode adError) {
                AdLogUtil.LOG.w("image load fail,code:" + adError.getErrorCode() + ",message:" + adError.getErrorMessage());
                if (listener != null) listener.onImagesFailedToCache(adError);
            }
        }).setUrl(url).setAdsDTO(adsDTO, m_type);
        downLoadRequest.netRequestPreExecute();
    }


    /**
     * 预缓存多张图片
     *
     * @param imageUrls     图片url列表
     * @param imageListener 回调接口
     */
    public static void preCacheSplashImages(@NonNull final List<String> imageUrls, AdsDTO adsDTO, int m_type, @NonNull final ImageListener imageListener) {
        final AtomicInteger imageCounter = new AtomicInteger(imageUrls.size());
        final AtomicBoolean anyFailures = new AtomicBoolean(false);

        for (final String url : imageUrls) {
            if (!TextUtils.isEmpty(url)) {
                AdLogUtil.LOG.d("request url:" + url);

                DownLoadRequest downLoadRequest = new DownLoadRequest().setPreCache(Constants.ImageLoadType.IMAGE_DISK_INCLUDED).setListener(new DrawableResponseListener() {
                    @Override
                    public void onRequestSuccess(int statusCode, byte[] data,  AdImage mediaBean) {
                        AdLogUtil.LOG.d("image cache success");
                        final int count = imageCounter.decrementAndGet();
                        if (count == 0 && !anyFailures.get() && imageListener != null) {
                            // 缓存成功
                            imageListener.onImagesCached(data);
                        }
                    }

                    @Override
                    protected void onRequestError(TaErrorCode adError) {
                        AdLogUtil.LOG.w("image cache fail,code:" + adError.getErrorCode() + ",message:" + adError.getErrorMessage());
                        boolean anyPreviousErrors = anyFailures.getAndSet(true);
                        imageCounter.decrementAndGet();
                        if (!anyPreviousErrors && imageListener != null) {
                            // 缓存失败
                            imageListener.onImagesFailedToCache(adError);
                        }
                    }
                });

                downLoadRequest.setUrl(url).setAdsDTO(adsDTO, m_type)
                        .netRequestPreExecute();
            }
        }
    }

    public static void preCacheNativeImageList(@NonNull final List<AdImage> images,
                                               final ImageListener imageListener,
                                               final boolean unified) {
        final AtomicInteger imageCounter = new AtomicInteger(images.size());
        if (unified) {
            for (AdImage image : images) {
                preCacheNativeImage(image, new ImageListener() {
                    @Override
                    public void onImagesCached(byte[] data) {
                        final int count = imageCounter.decrementAndGet();
                        if (count == 0 && imageListener != null) {
                            imageListener.onImagesCached(data);
                        }
                    }

                    @Override
                    public void onImagesFailedToCache(TaErrorCode errorCode) {
                        final int count = imageCounter.decrementAndGet();
                        if (count == 0 && imageListener != null) {
                            imageListener.onImagesFailedToCache(errorCode);
                        }
                    }
                });
            }
        } else {
            final boolean[] stop = {false};
            for (AdImage image : images) {
                if (stop[0]) {
                    break;
                }
                preCacheNativeImage(image, new ImageListener() {
                    @Override
                    public void onImagesCached(byte[] data) {
                        final int count = imageCounter.decrementAndGet();
                        if (count == 0 && imageListener != null) {
                            imageListener.onImagesCached(data);
                        }
                    }

                    @Override
                    public void onImagesFailedToCache(TaErrorCode errorCode) {
                        stop[0] = true;
                        if (imageListener != null) {
                            imageListener.onImagesFailedToCache(errorCode);
                        }
                    }
                });
            }
        }
    }

    /**
     * 图片预加载
     *
     * @param image    图片url信息
     * @param listener 回调接口
     */
    public static void preCacheNativeImage(@NonNull final AdImage image, final ImageListener listener) {
        if (image == null) {
            AdLogUtil.LOG.d("image url is null");
            return;
        }
        final String url = image.getImgUrl();
        AdLogUtil.LOG.d("开始缓存图片，request url:" + url);
        DownLoadRequest downLoadRequest = new DownLoadRequest().setPreCache(image.preCache)
                .setListener(new DrawableResponseListener() {
                    @Override
                    public void onRequestSuccess(int statusCode, byte[] data,  AdImage mediaBean) {
                        AdLogUtil.LOG.d("图片缓存成功，image cache success");
                        if (image.preCache == Constants.ImageLoadType.IMAGE_PARSE && mediaBean != null) {
                            if (image.getMime() != Constants.MIME.VIDEO) {
                                image.setMime(mediaBean.getMime());
                            }
                        }
                        image.setCached(true);
                        if (image.preCache == Constants.ImageLoadType.IMAGE_RAM_INCLUDED) {
                            if (mediaBean == null) {
                                return;
                            }
                            //一期，gif当作图片处理
                            if (mediaBean.getMime() == Constants.MIME.IMG || mediaBean.getMime() == Constants.MIME.GIF) {
                                image.setDrawable(mediaBean.getDrawable());
                            }
                        }
                        if (listener != null) {
                            listener.onImagesCached(data);
                        }
                    }

                    @Override
                    protected void onRequestError(TaErrorCode adError) {
                        AdLogUtil.LOG.w("图片缓存失败，url:" + image.getImgUrl() + ", code:" + adError.getErrorCode() +
                                ",message:" + adError.getErrorMessage());
                        if (listener != null) {
                            listener.onImagesFailedToCache(new TaErrorCode(adError.getErrorCode(), "cache image fail, url:" + image.getImgUrl() + ", " + adError.getErrorMessage()));
                        }
                    }
                })
                .setUrl(url).setAdsDTO(image.adsDTO, image.getmType());
        downLoadRequest.netRequestPreExecute();
    }

    /**
     * 图片下载回调接口
     */
    public interface ImageListener {
        /**
         * 图片缓存完成
         */
        void onImagesCached(byte[] data);

        /**
         * 图片缓存失败
         *
         * @param errorCode 错误码
         */
        void onImagesFailedToCache(TaErrorCode errorCode);
    }
}
