package com.cloud.hisavana.sdk.common.tracking;

import android.net.Uri;
import android.os.Build;
import android.text.TextUtils;
import android.util.Log;

import com.cloud.hisavana.sdk.BuildConfig;
import com.cloud.hisavana.sdk.api.config.SspAd;
import com.cloud.hisavana.sdk.common.constant.Constants;
import com.cloud.hisavana.sdk.common.constant.TaErrorCode;
import com.cloud.hisavana.sdk.common.util.AESUtils;
import com.cloud.hisavana.sdk.common.util.AdLogUtil;
import com.cloud.hisavana.sdk.common.util.DeviceUtil;
import com.cloud.hisavana.sdk.common.util.GPSTracker;
import com.cloud.hisavana.sdk.common.util.MitNetUtil;
import com.cloud.hisavana.sdk.common.util.Preconditions;
import com.cloud.hisavana.sdk.common.util.UserAgentUtil;
import com.cloud.hisavana.sdk.data.control.AdxPreferencesHelper;
import com.transsion.core.deviceinfo.DeviceInfo;
import com.transsion.core.pool.TranssionPoolManager;
import com.transsion.core.utils.AppUtil;
import com.transsion.http.HttpClient;
import com.transsion.http.impl.StringCallback;

import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

/**
 * 落地页跟踪
 */
public class TrackingManager {

    /**
     * 超时时间
     */
    private static final int TIMEOUT_IN_MILLIONS = 15 * 1000;


    // =============================================================================================


    /**
     * 当广告被点击后，触发此字段中的 clickTrackingUrls
     */
    public static void clickTrackingUrlsExposure(List<String> urls, final String clickid) {
        if (urls == null || urls.size() < 1) {
            AdLogUtil.trackLog("clickTrackingUrls || showTrackingUrls --> urls==null || urls.size()=<1 --> return ");
            return;
        }

        // 发起多个请求
        for (final String url : urls) {
            if (TextUtils.isEmpty(url)) {
                continue;
            }

            TranssionPoolManager.getInstance().addTask(new Runnable() {
                @Override
                public void run() {
                    sendRequestToServer(processUrl2(url, clickid), null);
                }
            });
        }
    }


    // =============================================================================================


    /**
     * deepLinkUrl  应用深链接地址
     * clickUrl     点击链接地址
     */
    public static void deepLinkUrlExposure(String url, DownUpPointBean pointBean, String clickid, TrackingCallBack callBack) {
        if (null == url || TextUtils.isEmpty(url) || null == pointBean) {
            AdLogUtil.trackLog("deepLinkUrlExposure --> null == url || null == pointBean");
            return;
        }

        // 发起请求
        sendRequestToServer(processUrl(url, pointBean, clickid), callBack);
    }


    // =============================================================================================


    /**
     * 发起网络请求
     *
     * @param url
     */
    private static void sendRequestToServer(String url, final TrackingCallBack callBack) {
        AdLogUtil.trackLog("sendRequestToServer --> url = " + url);

        if (url == null) {
            return;
        }
        HttpClient.get()
                .log(SspAd.isDebug())
                .connectTimeout(TIMEOUT_IN_MILLIONS)
                .readTimeout(TIMEOUT_IN_MILLIONS)
                .url(url)
                .addHeader("User-Agent", UserAgentUtil.getUserAgent())
                .build()
                .execute(new StringCallback(true) {
                             @Override
                             public void onSuccess(final int statusCode, final String response) {
                                 if (callBack == null) {
                                     return;
                                 }

                                 Preconditions.runOnMainThread(new Preconditions.Callback() {
                                     @Override
                                     public void onRun() {
                                         if (!TextUtils.isEmpty(response)) {
                                             callBack.onServerRequestSuccess(statusCode, response);
                                         } else {
                                             callBack.onServerRequestFailure(TaErrorCode.HTTP_RESPONSE_ERROR.getErrorCode(),
                                                     TaErrorCode.HTTP_RESPONSE_ERROR.getErrorMessage(), null);
                                         }
                                     }
                                 });

                             }

                             @Override
                             public void onFailure(final int statusCode, final String response, final Throwable throwable) {
                                 AdLogUtil.trackLog("sendRequestToServer --> onFailure --> statusCode = " + statusCode + " response = " + response);
                                 if (callBack != null) {
                                     Preconditions.runOnMainThread(new Preconditions.Callback() {
                                         @Override
                                         public void onRun() {
                                             callBack.onServerRequestFailure(statusCode, response, throwable);
                                         }
                                     });
                                 }
                             }
                         }
                );
    }

    /**
     * 将结果 导出
     */
    public interface TrackingCallBack {

        /**
         * 请求失败
         */
        void onServerRequestFailure(int statusCode, String responseBody, Throwable throwable);

        /**
         * 请求成功
         */
        void onServerRequestSuccess(int statusCode, String response);

    }


    // =============================================================================================


    /**
     * 处理 url
     * <p>
     * xd=_X-DOWN_&yd=_Y-DOWN_&xu=_X-UP_&yu=_Y-UP_&ai=_APPID_&pn=_PACKAGE-NAME_&ve=_VERSION_&sv=_SDK-VERSION_&ot=_OS-TYPE_&ov=_OS-VERSION_&nc=_NET-CONN-TYPE_&op=_OPERATOR-TYPE_&ga=_GAID_&dt=_DEVICE-TYPE_&br=_BRAND_&mo=_MODEL_&ma=_MAKER_&la=_LATITUDE_&lo=_LONGITUDE_&iw=_IMAGE-WIDTH_&ih=_IMAGE-HEIGHT_
     */
    public static String processUrl(String url, DownUpPointBean pointBean, String clickid) {

        if (TextUtils.isEmpty(url) || null == pointBean) {
            return url;
        }

        // 组装Url
        AdLogUtil.trackLog("TrackingManager --> processUrl --> 处理前 url = " + url);
        AdLogUtil.trackLog("TrackingManager --> processUrl --> pointBean = " + pointBean.toString());

        StringBuilder sb = new StringBuilder();
        sb.append("xd=" + pointBean.getDownX()); // X-DOWN         点击坐标（按）     String
        sb.append("&yd=" + pointBean.getDownY()); // Y-DOWN         点击坐标（按）     String
        sb.append("&xu=" + pointBean.getUpX()); // X-UP           点击坐标（弹）     String
        sb.append("&yu=" + pointBean.getUpY()); // Y-UP           点击坐标（弹）     String
        sb.append("&ai=" + SspAd.AppId); //  APPID          点击的应用id     String

        sb.append("&pn=" + AppUtil.getPkgName()); //  PACKAGE-NAME   点击的app包名    String
        sb.append("&ve=" + AppUtil.getVersionName()); // VERSION        点击的app版本号   String
        sb.append("&sv=" + BuildConfig.VERSION_NAME); // SDK-VERSION    点击的sdk版本号   String
        sb.append("&ot=" + 1); // OS-TYPE        操作系统类型      String //1、Android；2、iOS
        sb.append("&ov=" + DeviceUtil.getSystemVersion()); // OS-VERSION     操作系统版本号     String

        sb.append("&nc=" + MitNetUtil.getNetType()); //  NET-CONN-TYPE  网络连接类型      String
        sb.append("&op=" + DeviceUtil.getMNC() + DeviceUtil.getMCC()); // OPERATOR-TYPE  运营商类型       String
        sb.append("&ga=" + DeviceUtil.getGAId()); // GAID           点击时的gaid    String
        sb.append("&dt=" + (DeviceInfo.isPad() ? 2 : 1)); // DEVICE-TYPE    设备类型        String
        sb.append("&br=" + Build.BRAND); // BRAND          机型品牌        String
        sb.append("&mo=" + Build.MODEL); // MODEL          客户端手机型号     String
        sb.append("&ma=" + Build.MANUFACTURER); // MAKER          设备制造商        String

        sb.append("&la=" + GPSTracker.getLatitude()); // LATITUDE       点击时的纬度      String
        sb.append("&lo=" + GPSTracker.getLongitude()); // LONGITUDE      点击时的经度      String

        sb.append("&iw=" + pointBean.getImageW()); // IMAGE-WIDTH    点击时素材宽      String
        sb.append("&ih=" + pointBean.getImageH()); // IMAGE-HEIGHT   点击时素材高      String

        // 双11版本新增 clickid --> ci
        sb.append("&ci=" + clickid);

        AdLogUtil.trackLog("埋点字段 --> sb=" + sb.toString());
        url = url + "&p=" + AESUtils.encrypt(sb.toString());
        AdLogUtil.trackLog("TrackingManager --> processUrl --> 处理后 url = " + url);

        return url;
    }

    /**
     * 第三方归因服务
     * <p>
     * 回调链接宏替换参数
     * __CLICK_ID__
     * __CLICK_IP__
     * <p>
     * callback_url 里面的
     * __CLICK_ID__
     * __CLICK_TS__
     */
    private static String processUrl2(String url, String clickId) {

        if (TextUtils.isEmpty(url)) {
            return url;
        }

        // 下面 是一个例子
        //https://app.adjust.com/k80uf2p?campaign=__CAMPAIGN_NAME__%20(4110)
        //&adgroup=eagllwin
        //&creative=__CREATIVE_NAME__%20(60245)
        //&click_id=__CLICK_ID__&gps_adid_lower_md5=66e7d3fa59bf768cdf94a6f661fa9365
        //&android_id_lower_md5=__ANDROID_ID_LOWER_MD5__
        //&ip_address=__CLICK_IP__
        //&install_callback=http%3A%2F%2Fthirdback.test.eagllwin.com%2Feagllwin%2Fascribe%2Fv1%2Fthirdback%2Fascribe%2Fcallback%3Fclick_id%3D__CLICK_ID__%26click_ts%3D__CLICK_TS__%26callbackparam%3DeyJyZXF1ZXN0SWQiOiI4NjRhZDA3OS04YjcyLTRiYjctYmZlYy1lNWVhMWE2OGUyNWIiLCJhZHZlcnRpc2VySWQiOiIxMDYwIiwicGxhbklkIjoiNDExMCIsImFkZ3JvdXBJZCI6IjQyMzIiLCJjcmVhdGl2ZUlkIjoiNjAyNDUiLCJnYWlkIjoiZjBjYTcyZDEtYjM3Mi00M2VjLTk2NzAtOTRiN2NhOGY3YWY5IiwiYXBwSWQiOiJoZXpoIiwiY29kZVR5cGUiOiIzIiwiY2FsbGJhY2tUeXBlIjoiY2xpY2sifQ%3D%3D%26source%3Dadjust%26conv_time%3D%7Bcreated_at%7D%26event_type%3D2

//        url = "https://app.adjust.com/k80uf2p?campaign=__CAMPAIGN_NAME__%20(4110)&adgroup=eagllwin&creative=__CREATIVE_NAME__%20(60245)&click_id=__CLICK_ID__&gps_adid_lower_md5=66e7d3fa59bf768cdf94a6f661fa9365&android_id_lower_md5=__ANDROID_ID_LOWER_MD5__&ip_address=__CLICK_IP__&install_callback=http%3A%2F%2Fthirdback.test.eagllwin.com%2Feagllwin%2Fascribe%2Fv1%2Fthirdback%2Fascribe%2Fcallback%3Fclick_id%3D__CLICK_ID__%26click_ts%3D__CLICK_TS__%26callbackparam%3DeyJyZXF1ZXN0SWQiOiI4NjRhZDA3OS04YjcyLTRiYjctYmZlYy1lNWVhMWE2OGUyNWIiLCJhZHZlcnRpc2VySWQiOiIxMDYwIiwicGxhbklkIjoiNDExMCIsImFkZ3JvdXBJZCI6IjQyMzIiLCJjcmVhdGl2ZUlkIjoiNjAyNDUiLCJnYWlkIjoiZjBjYTcyZDEtYjM3Mi00M2VjLTk2NzAtOTRiN2NhOGY3YWY5IiwiYXBwSWQiOiJoZXpoIiwiY29kZVR5cGUiOiIzIiwiY2FsbGJhY2tUeXBlIjoiY2xpY2sifQ%3D%3D%26source%3Dadjust%26conv_time%3D%7Bcreated_at%7D%26event_type%3D2";
//        url = "www.baidu.com";
//        url = "https://www.baidu.com";
//        url = "https://www.baidu.com?????a=1&b=2&&&&c=3"; // https://www.baidu.com?a=1&b=2&=&=&=&c=3
//        url = "https://app.adjust.com/k80uf2p?campaign=__CAMPAIGN_NAME__%20(4110)&adgroup=eagllwin&creative=__CREATIVE_NAME__%20(60245)&click_id=__CLICK_ID__&gps_adid_lower_md5=66e7d3fa59bf768cdf94a6f661&install_callback=http%3A%2F%2Fthirdback.test.eagllwin.com%2Feagllwin%2Fascribe%2Fv1%2Fthirdback%2Fascribe%2Fcallback%3Fclick_id%3D__CLICK_ID__%26click_ts%3D__CLICK_TS__%26callbackparam%3DeyJyZX";
//        url = "jkasdghjksdvbkbkjasndkja23474iwrwisdbfkjsb3iuyioh";
//        url = "qidohcoh?a;dscj;j????asldjkl&bnkjans&a=hushkach&najnsc=nxjksnjk&s=cnalnsn&s=nlksncln38497897&q=97986776&*656434#$%^&====="; // 无法解析  结果 --> qidohcoh?bnkjans=&a=hushkach&najnsc=nxjksnjk&s=cnalnsn&s=nlksncln38497897&q=97986776&*656434=#$%^&=====
//        url = "";
//        url = "18734927852738094759823749752347";

        return replace(url, clickId);
    }

    /**
     * 宏替换
     *
     * @param url
     * @return
     */
    private static String replace(String url, String clickId) {

        try {

            AdLogUtil.LOG.d("TrackingManager --> processUrl2 --> 准备处理 url = " + url + " clickId = " + clickId);
            Uri uri = Uri.parse(url);

            // 解析不了 直接返回
            if (null == uri) {
                return url;
            }

            // 用来保存替换之后的 key-value
            List<InnerClass> newParams = new ArrayList<>();

            // 获取uri所有的 key的name
            Set<String> queryParameterNames = uri.getQueryParameterNames();

            // 如果是空的就不解析了
            if (!queryParameterNames.isEmpty()) {

                for (String name : queryParameterNames) {

                    if (TextUtils.isEmpty(name)) {
                        continue;
                    }

                    // 通过name获取值 这里获取到的是一个list ，所以需要遍历操作 , 没有搜索到 返回 Collections.emptyList();
                    List<String> values = uri.getQueryParameters(name);

                    //遍历所有的值找到需要替换的元素
                    for (String vs : values) {

                        if (TextUtils.equals(name, "install_callback") || TextUtils.equals(name, "callback")) {
                            String replace = replace(vs, clickId);
                            // 将处理过的key value 重新保存  内部的
                            newParams.add(new InnerClass(name, replace));
                        } else {
                            // 需要替换的
                            String newVs = vs;
                            if (TextUtils.equals(vs, "__CLICK_ID__")) {
                                newVs = clickId;
                            } else if (TextUtils.equals(vs, "__CLICK_IP__")) {
                                newVs = DeviceUtil.getIp();
                            } else if (TextUtils.equals(vs, "__ANDROID_ID_LOWER_MD5__")) {
                                newVs = DeviceUtil.getAndroidIDMd5ToLowerCase();
                            } else if (TextUtils.equals(vs, "__CLICK_TS__")) {
                                newVs = String.valueOf(System.currentTimeMillis() + AdxPreferencesHelper.getInstance().getLong(Constants.SERVICES_TIME_DIFFERENCE_VALUE));
                            }
                            // 将处理过的key value 重新保存
                            newParams.add(new InnerClass(name, newVs));
                        }
                    }
                }

                // 重新拼接参数
                Uri.Builder builder = uri.buildUpon().clearQuery();
                if (null != builder) {
                    for (InnerClass innerClass : newParams) {
                        builder.appendQueryParameter(innerClass.getKey(), URLDecoder.decode(innerClass.getValue(), "UTF-8"));
                    }
                    AdLogUtil.LOG.d("TrackingManager --> processUrl2 --> 返回结果 " + builder.toString());
                    return builder.toString();
                }
            }

        } catch (Exception e) {
            AdLogUtil.LOG.d("TrackingManager --> processUrl2 --> error " + Log.getStackTraceString(e));
        }

        // 处理失败 返回原数据
        AdLogUtil.LOG.d("TrackingManager --> processUrl2 --> 无法处理直接返回 url = " + url);
        return url;
    }

    /**
     * 保存 key value 的对象
     */
    private static class InnerClass {

        private String key;
        private String value;

        public InnerClass(String key, String value) {
            this.key = key;
            this.value = value;
        }

        public String getKey() {
            return key;
        }

        public void setKey(String key) {
            this.key = key;
        }

        public String getValue() {
            return value;
        }

        public void setValue(String value) {
            this.value = value;
        }
    }

}
