package com.cloud.hisavana.sdk.offline;

import static com.cloud.hisavana.sdk.common.constant.Constants.GET_OFF_LINE_AD_TIME;
import static com.cloud.hisavana.sdk.common.constant.Constants.HOST.OFF_LINE_URL;
import static com.cloud.hisavana.sdk.common.constant.Constants.HOST.X_SHARE_OFF_LINE_URL;
import static com.cloud.hisavana.sdk.common.constant.Constants.OFF_LINE_ENABLE;
import static com.cloud.hisavana.sdk.common.constant.Constants.OFF_LINE_REQUEST_TIME_INTERVAL;
import static com.cloud.hisavana.sdk.common.constant.Constants.OFF_LINE_SHOW_COUNT_LIMIT;
import static com.cloud.hisavana.sdk.common.util.AdLogUtil.OFF_LINE_TAG;

import android.support.annotation.Keep;
import android.support.annotation.NonNull;
import android.text.TextUtils;

import com.cloud.hisavana.sdk.api.config.SspAd;
import com.cloud.hisavana.sdk.common.athena.AthenaTracker;
import com.cloud.hisavana.sdk.common.constant.Constants;
import com.cloud.hisavana.sdk.common.constant.TaErrorCode;
import com.cloud.hisavana.sdk.common.gsonutil.GsonUtil;
import com.cloud.hisavana.sdk.common.util.AdLogUtil;
import com.cloud.hisavana.sdk.common.util.DeviceUtil;
import com.cloud.hisavana.sdk.common.util.ImageDownloadHelper;
import com.cloud.hisavana.sdk.common.util.MitNetUtil;
import com.cloud.hisavana.sdk.common.util.Preconditions;
import com.cloud.hisavana.sdk.data.bean.request.AdxImpBean;
import com.cloud.hisavana.sdk.data.bean.response.AdsDTO;
import com.cloud.hisavana.sdk.data.control.AdxPreferencesHelper;
import com.transsion.core.CoreUtil;
import com.transsion.core.pool.TranssionPoolManager;
import com.transsion.http.HttpClient;
import com.transsion.http.ImageURL;
import com.transsion.http.RequestCall;
import com.transsion.http.cache.IDiskCache;
import com.transsion.http.impl.StringCallback;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * Created by xuhui.fang ON 2022/4/18
 */
public class OffLineManager {
    private static final long DEFAULT_INTERVAL_TIME = 60 * 1000;
    private static final String OFF_LINE_AD_FILE_NAME = "off_line_ad.json";
    private static final String OFF_LINE_FILE_DIR = "/offlineAd";
    private AtomicBoolean isLoading = new AtomicBoolean(false);
    private OffLineData offLineData;
    private String requestId;
    @Keep
    private OffLineLoadListener listener;
    private int splashAdCounts = -1;
    private int nativeAdCounts = -1;
    private int interstitialAdCounts = -1;
    private int maxShowTimes = -1;
    private long intervalTime;
    private long lastTime;

    private static class SingletonHolder {
        private static OffLineManager instance = new OffLineManager();
    }

    public static OffLineManager getInstance() {
        return SingletonHolder.instance;
    }

    public OffLineManager setListener(OffLineLoadListener offLineLoadListener) {
        this.listener = offLineLoadListener;
        return this;
    }

    /**
     * 返回有效离线广告数量
     */
    public int getOfflineAdCounts(int adType) {
        switch (adType) {
            case Constants.AdType.SPLASH_NATIVE: //splash
                if (splashAdCounts < 0) {
                    countOfflineAd();
                }
                return Math.max(splashAdCounts, 0);
            case Constants.AdType.NATIVE: //native
                if (nativeAdCounts < 0) {
                    countOfflineAd();
                }
                return Math.max(nativeAdCounts, 0);
            case Constants.AdType.INTERSTITIAL: //interstitial
                if (interstitialAdCounts < 0) {
                    countOfflineAd();
                }
                return Math.max(interstitialAdCounts, 0);
            default:
                return 0;
        }
    }

    /**
     * 判断是否可以请求离线广告
     *
     * @return
     */
    private boolean isCanGetOffLineAd() {
        if (intervalTime == 0) {
            intervalTime = AdxPreferencesHelper.getInstance().getLong(OFF_LINE_REQUEST_TIME_INTERVAL, DEFAULT_INTERVAL_TIME);
        }
        if (lastTime == 0) {
            lastTime = AdxPreferencesHelper.getInstance().getLong(GET_OFF_LINE_AD_TIME, 0);
        }
        AdLogUtil.Log().d(OFF_LINE_TAG, "当前间隔时间==" + intervalTime);
        AdLogUtil.Log().d(OFF_LINE_TAG, "请求的间隔时间==" + (System.currentTimeMillis() - lastTime));
        if ((System.currentTimeMillis() - lastTime) < intervalTime) {
            return false;
        }
        return true;
    }

    /**
     * 将离线广告的云控json写入本地
     * 后期该用数据库的方式
     *
     * @param
     */
    private void writeOffLineAd(final String json) {
        if (TextUtils.isEmpty(json)) {
            return;
        }
        AdLogUtil.Log().d(OFF_LINE_TAG, "要写入的广告数据 ====" + (SspAd.isDebug() ? json : ""));
        TranssionPoolManager.getInstance().addTask(new Runnable() {
            @Override
            public void run() {
                String temporaryName = System.currentTimeMillis() + ".json";
                File offlineDir = new File(CoreUtil.getContext().getFilesDir() + OFF_LINE_FILE_DIR);
                File temporaryFile = new File(offlineDir, temporaryName);
                File file = new File(offlineDir, OFF_LINE_AD_FILE_NAME);
                if (!offlineDir.exists()) offlineDir.mkdir();
                FileWriter fileWriter;
                BufferedWriter bw = null;
                try {
                    if (!temporaryFile.exists()) temporaryFile.createNewFile();
                    fileWriter = new FileWriter(temporaryFile);
                    bw = new BufferedWriter(fileWriter);
                    bw.write(json);
                    bw.flush();
                    bw.close();
                    if (file.exists()) file.delete();
                    temporaryFile.renameTo(file);
                    temporaryFile.delete();
                    AdLogUtil.Log().d(OFF_LINE_TAG, "offline ad write json ====" + (SspAd.isDebug() ? json : ""));
                } catch (IOException e) {
                    e.printStackTrace();
                } finally {
                    try {
                        if (bw != null) {
                            bw.close();
                        }
                    } catch (IOException e) {
                        e.printStackTrace();
                    }
                }
            }
        });
    }

    /**
     * 读取本地的离线广告配置
     *
     * @return
     */
    private OffLineData readOffLineAd() {
        File file = new File(CoreUtil.getContext().getFilesDir() + OFF_LINE_FILE_DIR, OFF_LINE_AD_FILE_NAME);
        if (!file.exists()) return null;
        FileReader fileReader = null;
        try {
            fileReader = new FileReader(file);
            char[] bt = new char[1024];
            StringBuffer sb = new StringBuffer();
            int len = 0;
            while ((len = fileReader.read(bt)) != -1) {
                sb.append(bt, 0, len);
            }
            if (sb.length() == 0) return null;
            AdLogUtil.Log().d(OFF_LINE_TAG, "offline ad read json ====" + (SspAd.isDebug() ? sb.toString() : ""));
            OffLineData offLineData = GsonUtil.fromJson(sb.toString(), OffLineData.class);
            fileReader.close();
            return offLineData;
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        } finally {
            try {
                if (fileReader != null) {
                    fileReader.close();
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * 请求离线广告
     */
    public void requestOffLineAd(final boolean isRunForeground, final int requestType) {
        //  AthenaTracker.trackOffLinePreRequest(requestTs, requestType, requestId);
        if (!MitNetUtil.isNetWorkAvailable() || isLoading.get()) {
            AdLogUtil.Log().d(OFF_LINE_TAG, "无网络或正在请求当中");
            return;
        }
        final long requestTs = System.currentTimeMillis();
        requestId = DeviceUtil.getUUID();
        AdLogUtil.Log().d(OFF_LINE_TAG, "offline ad request ---requestOffLineAd()");
        isLoading.set(true);
        if (!isCanGetOffLineAd()) {
            if (offLineData == null) {
                offLineData = readOffLineAd();
            }
            loadMaterial(isRunForeground);
            AdLogUtil.Log().d(OFF_LINE_TAG, "请求间隔时间未达到直接去请求素材");
            return;
        }
        AthenaTracker.trackOffLineSspRequest(requestTs, requestType, requestId);
        AdLogUtil.Log().d(OFF_LINE_TAG, "请求间隔时间已经达到，请求云控数据");
        HttpClient.get().log(SspAd.isDebug())
                .sslSocketFactory(null)
                .connectTimeout(Constants.HTTP_TIME_OUT.TIMEOUT_IN_MILLIONS)
                .readTimeout(Constants.HTTP_TIME_OUT.READ_TIMEOUT_IN_MILLIONS)
                .addHeader("x-tr-signature", "h5")
                //.addHeader("cloudControlVersion",cloudControlVersion)
                .url(OFF_LINE_URL)
                .addHeader("Accept-Timezone", "UTC")
                .build()
                .execute(new StringCallback(true) {
                    @Override
                    public void onFailure(final int statusCode, final String message, final Throwable throwable) {
                        Preconditions.runOnMainThread(new Preconditions.Callback() {
                            @Override
                            public void onRun() {
                                AthenaTracker.trackOffLineSspReturn(requestTs, requestType, requestId,
                                        null, statusCode + "", message);
                                AdLogUtil.Log().d(OFF_LINE_TAG, "offline ad request ---onFailure == throwable:" + throwable.getMessage());
                                if (null != listener) {
                                    listener.complete();
                                }
                                isLoading.set(false);
                            }
                        });
                    }

                    @Override
                    public void onSuccess(int statusCode, final String response) {
                        Preconditions.runOnMainThread(new Preconditions.Callback() {
                            @Override
                            public void onRun() {
                                AdLogUtil.Log().d(OFF_LINE_TAG, "offline ad request ---onSuccess ==" + response);
                                if (TextUtils.isEmpty(response)) {
                                    isLoading.set(false);
                                    if (null != listener) {
                                        listener.complete();
                                    }
                                    AthenaTracker.trackOffLineSspReturn(requestTs, requestType, requestId, null, null, null);
                                    return;
                                }
                                try {
                                    OffLineResponseBody responseBody = GsonUtil.fromJson(response, OffLineResponseBody.class);
                                    lastTime = System.currentTimeMillis();
                                    AdxPreferencesHelper.getInstance().putLong(GET_OFF_LINE_AD_TIME, lastTime);
                                    AdxPreferencesHelper.getInstance().putBoolean(OFF_LINE_ENABLE, responseBody.getData().getEnable());
                                    maxShowTimes = responseBody.getData().getShowCountLimit();
                                    AdxPreferencesHelper.getInstance().putInt(OFF_LINE_SHOW_COUNT_LIMIT, maxShowTimes);
                                    intervalTime = responseBody.getData().getRequestInterval() * 60 * 1000;
                                    AdxPreferencesHelper.getInstance().putLong(OFF_LINE_REQUEST_TIME_INTERVAL, intervalTime);
                                    //   int localVersion = AdxPreferencesHelper.getInstance().getInt(OFF_LINE_CONFIG_VERSION);
                                    //   AdLogUtil.Log().d(OFF_LINE_TAG, "当前版本号 == " + localVersion + " 云控版本：" + responseBody.getData().getConfigVersion());
//                                    if (responseBody.getData().getConfigVersion() <= localVersion) {
//                                        if (offLineData == null) {
//                                            offLineData = readOffLineAd();
//                                        }
//                                        loadMaterial(offLineData, isRunForeground);
//                                        return;
//                                    }
                                    //  AdxPreferencesHelper.getInstance().putInt(OFF_LINE_CONFIG_VERSION, responseBody.getData().getConfigVersion());
                                    if (responseBody.getData() == null || !responseBody.getData().getEnable()) {
                                        isLoading.set(false);
                                        if (null != listener) {
                                            listener.complete();
                                        }
                                        AthenaTracker.trackOffLineSspReturn(requestTs, requestType, requestId,
                                                null, TaErrorCode.RESPONSE_AD_IS_EMPTY.getErrorCode() + "",
                                                TaErrorCode.RESPONSE_AD_IS_EMPTY.getErrorMessage());
                                        return;
                                    }
                                    offLineData = responseBody.getData();
                                    AdLogUtil.Log().d(OFF_LINE_TAG, "当前的线程 --- " + Thread.currentThread().getName());

                                    trackAdSspReturn(requestTs, requestType);
                                    updateData(isRunForeground, requestTs, requestType);
                                } catch (Exception e) {
                                    AdLogUtil.Log().d(OFF_LINE_TAG, e.getMessage());
                                    isLoading.set(false);
                                    if (null != listener) {
                                        listener.complete();
                                    }
                                }
                            }
                        });
                    }
                });
    }

    private void trackAdSspReturn(long requestTs, int requestType) {
        if (offLineData == null) {
            return;
        }
        List<AdsDTO> adsDTOs = new ArrayList<>();
        if (offLineData.getSplash() != null) {
            adsDTOs.addAll(offLineData.getSplash());
        }
        if (offLineData.getNativeAd() != null) {
            adsDTOs.addAll(offLineData.getNativeAd());
        }
        if (offLineData.getInterstitial() != null) {
            adsDTOs.addAll(offLineData.getInterstitial());
        }
        ArrayList<Integer> ids = new ArrayList<>();
        for (AdsDTO adsDTO : adsDTOs) {
            if (null != adsDTO) {
                ids.add(adsDTO.getId());
            }
        }
        if (ids.isEmpty()) {
            AthenaTracker.trackOffLineSspReturn(requestTs, requestType,
                    requestId, null, TaErrorCode.RESPONSE_AD_IS_EMPTY.getErrorCode() + "",
                    TaErrorCode.RESPONSE_AD_IS_EMPTY.getErrorMessage());
        } else {
            AthenaTracker.trackOffLineSspReturn(requestTs, requestType,
                    requestId, ids, null, null);
        }
    }

    /**
     * 数据更新
     */
    private void updateData(boolean isRunForeground, long requestTs, int requestType) {
        if (offLineData == null) {
            isLoading.set(false);
            if (listener != null) {
                listener.complete();
            }
            return;
        }
        controlData(offLineData.getSplash(), requestTs, requestType);
        controlData(offLineData.getNativeAd(), requestTs, requestType);
        controlData(offLineData.getInterstitial(), requestTs, requestType);
        writeOffLineAd(GsonUtil.toJson(offLineData));
        loadMaterial(isRunForeground);
    }


    /**
     * 对数据进行处理
     *
     * @param newList
     */
    private void controlData(List<AdsDTO> newList, long requestTs, int requestType) {
        if (null == newList || newList.isEmpty()) return;
        Iterator<AdsDTO> newIterator = newList.iterator();
        while (newIterator.hasNext()) {
            AdsDTO adsDTO = newIterator.next();
            adsDTO.setOffLineAd(true);
            AdxImpBean impBean = adsDTO.getImpBeanRequest();
            if (impBean == null) {
                impBean = new AdxImpBean();
            }
            impBean.adt = adsDTO.getCodeSeatType();
            impBean.pmid = "";
            impBean.requestId = requestId;
            impBean.requestType = requestType;
            impBean.requestTs = requestTs;
            impBean.isOffLineAd = true;
            adsDTO.setImpBeanRequest(impBean);
            if (adsDTO.getCodeSeatType().intValue() == Constants.AdType.SPLASH_NATIVE) {
                if (adsDTO.getNativeObject() != null) {
                    adsDTO.getNativeObject().setLogo(null);
                }
            }

            if (null == adsDTO || null == adsDTO.getNativeObject() || null == adsDTO.getNativeObject().getMainImages()
                    || adsDTO.getNativeObject().getMainImages().isEmpty() || TextUtils.isEmpty(adsDTO.getNativeObject().getMainImages().get(0).getUrl())) {
                newIterator.remove();
                continue;
            }
        }
    }

    /**
     * 加载物料
     */
    private void loadMaterial(boolean isRunForeground) {
        if (offLineData == null) {
            isLoading.set(false);
            if (null != listener) listener.complete();
            return;
        }
        AdLogUtil.Log().d(OFF_LINE_TAG, "loadMaterial 开始请求物料");
        final List<AdsDTO> adsDTOs = new ArrayList<>();
        if (offLineData.getSplash() != null) {
            adsDTOs.addAll(offLineData.getSplash());
        }
        if (offLineData.getNativeAd() != null) {
            adsDTOs.addAll(offLineData.getNativeAd());
        }
        if (offLineData.getInterstitial() != null) {
            adsDTOs.addAll(offLineData.getInterstitial());
        }
        if (adsDTOs.isEmpty()) {
            isLoading.set(false);
            if (null != listener) {
                listener.complete();
            }
            return;
        }
        AdLogUtil.Log().d(OFF_LINE_TAG, "下载的物料size == " + adsDTOs.size());
        DownLoadOffLineAd temp = null;
        DownLoadOffLineAd head = null;
        for (AdsDTO impl : adsDTOs) {
            impl.setOffLineAd(true);
            DownLoadOffLineAd next = new DownLoadOffLineAd(impl);
            if (temp == null) {
                head = next;
            } else {
                temp.nextExecutor(next);
            }
            temp = next;
        }
        if (head == null) {
            isLoading.set(false);
            if (listener != null) {
                listener.complete();
            }
            return;
        }
        adSrcLoad(head, isRunForeground, new DownLoadOffLineAd.LoadListener() {
            @Override
            public void loadComplete() {
                countOfflineAd();
                // writeOffLineAd(offLineData);
                AdLogUtil.Log().d(OFF_LINE_TAG, "全部资源下载完成");
                //adChoice图片只需要下载一次就好
                if(!adsDTOs.isEmpty() && TextUtils.isEmpty(getCacheFilePath(adsDTOs.get(0).getAdChoiceImageUrl()))){
                    ImageDownloadHelper.downloadImage(adsDTOs.get(0).getAdChoiceImageUrl(),
                            Constants.ImageLoadType.IMAGE_DISK_INCLUDED, adsDTOs.get(0),
                            AthenaTracker.M_TYPE_AD_CHOICE_VIEW, null);
                }
                for (AdsDTO adsDTO : adsDTOs) {
                    if (adsDTO.getCodeSeatType().intValue() == Constants.AdType.INTERSTITIAL
                            || adsDTO.getCodeSeatType().intValue() == Constants.AdType.NATIVE) {
                        if (!TextUtils.isEmpty(getCacheFilePath(adsDTO.getLogoUrl()))) {
                            continue;
                        }
                        ImageDownloadHelper.downloadImage(adsDTO.getLogoUrl(),
                                Constants.ImageLoadType.IMAGE_DISK_INCLUDED, adsDTO,
                                AthenaTracker.M_TYPE_LOGO, null);
                    }
                }
                isLoading.set(false);
                if (listener != null) {
                    listener.complete();
                }
            }
        });
    }

    public void adSrcLoad(@NonNull DownLoadOffLineAd downLoadOffLineAd, boolean isRunForeground, DownLoadOffLineAd.LoadListener loadListener) {

        downLoadOffLineAd.setLoadListener(loadListener);
        if (isRunForeground) {
            downLoadOffLineAd.loadOrder(downLoadOffLineAd);
        } else {
            downLoadOffLineAd.loadTogether(downLoadOffLineAd);
        }

    }

    /**
     * 获取splash的离线广告
     */
    public void getOffLineAd(int type, AdDataListener listener) {
        if (!AdxPreferencesHelper.getInstance().getBoolean(OFF_LINE_ENABLE, false)) {
            if (listener != null) {
                listener.getAdDataFail();
            }
            return;
        }
        AdLogUtil.Log().d(OFF_LINE_TAG, "当前离线广告功能为true");
        if (offLineData == null) {
            offLineData = readOffLineAd();
        }
        if (null == offLineData) {
            AdLogUtil.Log().d(OFF_LINE_TAG, "本地的离线广告为空");
            if (listener != null) {
                listener.getAdDataFail();
            }
            return;
        }
        List<AdsDTO> adsDTOs;
        switch (type) {
            case Constants.AdType.SPLASH_NATIVE: //splash
                adsDTOs = offLineData.getSplash();
                break;
            case Constants.AdType.NATIVE: //native
                adsDTOs = offLineData.getNativeAd();
                break;
            case Constants.AdType.INTERSTITIAL: //interstitial
                adsDTOs = offLineData.getInterstitial();
                break;
            default:
                adsDTOs = new ArrayList<>();
                break;
        }
        if (null == adsDTOs || adsDTOs.isEmpty()) {
            if (listener != null) {
                listener.getAdDataFail();
            }
            return;
        }
        AdLogUtil.Log().d(OFF_LINE_TAG, "离线广告数据：" + (SspAd.isDebug() ? adsDTOs : ""));
        if (maxShowTimes < 0) {
            maxShowTimes = AdxPreferencesHelper.getInstance().getInt(OFF_LINE_SHOW_COUNT_LIMIT);
        }
        int count = adsDTOs.size();
        AdsDTO resultAd = null;
        for (int i = 0; i < count; i++) {
            if (checkAdAvailable(adsDTOs.get(i))) {
                resultAd = adsDTOs.get(i);
                break;
            }
        }
        if (resultAd == null) {
            if (listener != null) {
                listener.getAdDataFail();
            }
            return;
        }
        if (listener != null) {
            AdsDTO newResult = GsonUtil.fromJson(GsonUtil.toJson(resultAd), AdsDTO.class);
            listener.getAdData(newResult);
            AdLogUtil.Log().d(OFF_LINE_TAG, "离线广告数据：" + (SspAd.isDebug() ? resultAd : ""));
        }
    }

    /**
     * 更改理想广告的展示次数
     *
     * @param adsDTO 当前展示的广告
     */
    public void updateShowCount(AdsDTO adsDTO) {
        if (adsDTO == null || offLineData == null) {
            return;
        }
        List<AdsDTO> adsDTOList;
        switch (adsDTO.getCodeSeatType()) {
            case Constants.AdType.SPLASH_NATIVE: //splash
                adsDTOList = offLineData.getSplash();
                break;
            case Constants.AdType.NATIVE: //native
                adsDTOList = offLineData.getNativeAd();
                break;
            case Constants.AdType.INTERSTITIAL: //interstitial
                adsDTOList = offLineData.getInterstitial();
                break;
            default:
                adsDTOList = new ArrayList<>();
                break;
        }
        if (adsDTOList == null || adsDTOList.isEmpty()) {
            return;
        }
        if (!adsDTOList.contains(adsDTO)) {
            return;
        }
        int showOffLineCount = adsDTO.getShowOffLineCount();
        adsDTO.setShowOffLineCount(showOffLineCount + 1);
        countOfflineAd();
        //将广告放到队列末尾
        adsDTOList.remove(adsDTO);
        adsDTOList.add(adsDTO);
        writeOffLineAd(GsonUtil.toJson(offLineData));
    }

    /**
     * 通过url来获取本地缓存的file
     *
     * @param url
     * @return
     */
    public String getCacheFilePath(String url) {
        if (TextUtils.isEmpty(url)) {
            return "";
        }
        RequestCall requestImgCall = HttpClient.image(CoreUtil.getContext())         //必须方法体，需要传入context
                .useOffLineCache(true)
                .cache(true)
                .url(url)           //必须方法体，传入图片url
                .build();
        IDiskCache iDiskCacheImg = requestImgCall.getDiskCache();
        ImageURL imageURL = new ImageURL(url);
        File file = iDiskCacheImg.get(imageURL);
        if (file != null && file.exists()) {
            return file.getAbsolutePath();
        }
        return "";
    }


    public interface OffLineLoadListener {
        void complete();
    }

    public interface AdDataListener {
        void getAdData(AdsDTO adsDTO);

        void getAdDataFail();

    }

    /**
     * 计算可用离线广告数量
     */
    private void countOfflineAd() {
        if (offLineData == null) {
            offLineData = readOffLineAd();
        }
        if (offLineData == null) {
            nativeAdCounts = 0;
            splashAdCounts = 0;
            interstitialAdCounts = 0;
            return;
        }
        List<AdsDTO> interstitial = offLineData.getInterstitial();
        List<AdsDTO> splash = offLineData.getSplash();
        List<AdsDTO> nativeAd = offLineData.getNativeAd();
        interstitialAdCounts = countListAd(interstitial);
        splashAdCounts = countListAd(splash);
        nativeAdCounts = countListAd(nativeAd);
    }

    private int countListAd(List<AdsDTO> adsDTOList) {
        if (adsDTOList == null || adsDTOList.size() == 0) {
            return 0;
        }
        int listCount = 0;
        for (AdsDTO adsDTO : adsDTOList) {
            if (checkAdAvailable(adsDTO)) {
                ++listCount;
            }
        }
        return listCount;
    }

    public boolean checkAdAvailable(AdsDTO adsDTO) {
        if (adsDTO == null) {
            return false;
        }
        String imgUrl = "";
        if (adsDTO.getNativeObject() != null && adsDTO.getNativeObject().getMainImages() != null &&
                !adsDTO.getNativeObject().getMainImages().isEmpty()) {
            imgUrl = adsDTO.getNativeObject().getMainImages().get(0).getUrl();
        }
        String clickUrl = adsDTO.getClickUrl();
        String imgFilePath = getCacheFilePath(imgUrl);
        if (TextUtils.isEmpty(imgFilePath)) {
            return false;
        }
        String h5FilePath = getCacheFilePath(clickUrl);
        if ((adsDTO.getStoreDeeplink() == null || adsDTO.getStoreDeeplink().isEmpty()) && TextUtils.isEmpty(h5FilePath)) {
            return false;
        }
        if (maxShowTimes <= 0) {
            maxShowTimes = AdxPreferencesHelper.getInstance().getInt(OFF_LINE_SHOW_COUNT_LIMIT);
        }
        return adsDTO.getShowOffLineCount() < maxShowTimes;
    }
}
