package com.cloud.hisavana.sdk.common.widget.config;

import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.text.TextUtils;

import com.cloud.hisavana.sdk.BuildConfig;
import com.cloud.hisavana.sdk.common.util.AdLogUtil;
import com.cloud.hisavana.sdk.common.widget.net.HttpManager;
import com.transsion.core.CoreUtil;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

/**
 * {"code":0,"message":null,"data":{"webview":[{"url":"https://ipage.test.eagllwin.com/static/vendor.d9a6ab97.js","md5":"d9a6ab97"}]}}
 */
public class ConfigManager {
    private volatile static ConfigManager mInstance;
    private static Map<String, Class<? extends AbstractConfig>> mClassMap = new HashMap();

    private static final String HOST_RELEASE = "https://cloud-config-api.shalltry.com";
    private static final String HOST_DEBUG = "https://cloud-config-api-test.shalltry.com";

    public interface OnConfigCallback {
        void onCallback(ConfigData configData);

        void onFail(int code, String error);

        abstract class ConfigCallbackImpl implements OnConfigCallback {
            @Override
            public void onFail(int code, String error) {
                AdLogUtil.Log().d("webview--","ConfigManager onFail code = " + code + " error = " + error);
            }
        }
    }

    private ConfigManager() {
    }

    public static ConfigManager getInstance() {
        if (mInstance == null) {
            synchronized (ConfigManager.class) {
                if (mInstance == null) {
                    mInstance = new ConfigManager();
                }
            }
        }
        return mInstance;
    }

    /**
     * 将自己的信息注册到manager中，可以在manager拿到数据之后就收到自己对应的实验数据
     *
     * @param feature
     * @param clazz
     */
    public void register(String feature, Class<? extends AbstractConfig> clazz) {
        if (mClassMap != null && !mClassMap.containsKey(feature)) {
            mClassMap.put(feature, clazz);
        }
    }

    public void init(final Context context, String key, final OnConfigCallback callback) {
        try {
            String url = getHost() + "/common/cloudconfig/v1/consumer-not-login/configkeyvalue/query/get";
            JSONObject paramsObj = new JSONObject();
            paramsObj.put("key", key);
            Map<String, String> headers = new HashMap<>();
            headers.put("Accept-Timezone", "UTC");
            HttpManager.post(url, headers, paramsObj, new RequestCallbackImpl(callback));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private String getHost() {
        if (BuildConfig.DEBUG) {
            return HOST_DEBUG;
        }
        return HOST_RELEASE;
    }

    class RequestCallbackImpl implements HttpManager.OnRequestCallback {
        private final OnConfigCallback callback;

        public RequestCallbackImpl(OnConfigCallback callback) {
            this.callback = callback;
        }

        @Override
        public void onSuccess(String response) {
            AdLogUtil.Log().d("webview--","ConfigManager init =" + response);
            ConfigCacheManager.set(CoreUtil.getContext(), response);
            final ConfigData configData = ConfigParser.parser(response);
            notifyObserver(configData);
            if (callback != null && configData != null) {
                new Handler(Looper.getMainLooper()).post(new Runnable() {
                    @Override
                    public void run() {
                        callback.onCallback(configData);
                    }
                });
            }
        }

        @Override
        public void onFail(int code, String message) {
            AdLogUtil.Log().d("webview--","ConfigManager init fail code =" + code + " error = " + message);
            if (callback != null) {
                callback.onFail(code, message);
            }
        }
    }

    private void notifyObserver(ConfigData configData) {
        if (mClassMap != null && mClassMap.size() > 0 && configData != null) {
            dispatchToEachObserver(mClassMap, configData.getDataObj());
        }
    }

    private void dispatchToEachObserver(Map<String, Class<? extends AbstractConfig>> observerMap, JSONObject dataObj) {
        if (observerMap == null || observerMap.isEmpty() || dataObj == null) {
            return;
        }
        Iterator<String> iterator = mClassMap.keySet().iterator();
        while (iterator.hasNext()) {
            String feature = iterator.next();
            if (TextUtils.isEmpty(feature)) {
                continue;
            }

            boolean has = dataObj.has(feature);
            if (!has) {
                continue;
            }
            //这里强制要求配置的每一个key对应的value为jsonarray
            if (!(dataObj.opt(feature) instanceof JSONArray)) {
                throw new IllegalArgumentException("feature value must be JSONArray");
            }
            JSONArray jsonArray = dataObj.optJSONArray(feature);
            if (jsonArray == null || jsonArray.length() == 0) {
                continue;
            }
            Class<? extends AbstractConfig> aClass = mClassMap.get(feature);
            if (aClass == null) {
                continue;
            }
            try {
                AbstractConfig config = aClass.newInstance();
                config.onLoaded(jsonArray);
            } catch (Exception e) {
                e.printStackTrace();
                AdLogUtil.Log().d("webview--","ConfigManager dispatchToEachObserver error = " + e.toString());
            }
        }
    }

    public static JSONArray getByFeature(String feature) {
        String config = ConfigCacheManager.get(CoreUtil.getContext());
        if (TextUtils.isEmpty(config)) {
            return null;
        }
        ConfigData configData = ConfigParser.parser(config);
        if (configData == null) {
            return null;
        }
        JSONObject dataObj = configData.getDataObj();
        if (dataObj == null || !dataObj.has(feature)) {
            return null;
        }
        return dataObj.optJSONArray(feature);
    }
}
