package com.cloud.hisavana.sdk.common.widget.webview;

import android.os.MessageQueue;
import android.text.TextUtils;

import com.cloud.hisavana.sdk.BuildConfig;
import com.cloud.hisavana.sdk.common.util.AdLogUtil;
import com.transsion.core.pool.TranssionPoolManager;
import com.transsion.http.HttpClient;
import com.transsion.http.impl.HttpCallbackImpl;

import java.util.concurrent.atomic.AtomicBoolean;

import static com.transsion.core.CoreUtil.getContext;


public class IdleDownloadTask implements MessageQueue.IdleHandler {

    private final WebUrlData mWebData;
    private OnDownloadCallback mCallback;

    public interface OnDownloadCallback {
        void onComplete(String url);

        void onFail(int code, String error);
    }

    public IdleDownloadTask(WebUrlData webData) {
        this.mWebData = webData;
    }

    public IdleDownloadTask(WebUrlData webData, OnDownloadCallback callback) {
        this.mWebData = webData;
        this.mCallback = callback;
    }

    @Override
    public boolean queueIdle() {
        download(mWebData);
        return false;
    }

    private AtomicBoolean called = new AtomicBoolean(false);

    private void onSuccessFinal(String url) {
        if (mCallback != null && called.compareAndSet(false, true)) {
            mCallback.onComplete(url);
        }
    }

    private void download(final WebUrlData webData) {
        if (webData == null || TextUtils.isEmpty(webData.getUrl())) {
            return;
        }
        if (WebAssetsManager.checkWebAssetExist(webData.getUrl())) {
            return;
        }
        final long start = System.currentTimeMillis();
        HttpClient.download(getContext())
                .log(false)
                .pathname("xxxx")
                .connectTimeout(20 * 1000)
                .readTimeout(20 * 1000)
                .url(webData.getUrl())
                .build()
                .execute(new HttpCallbackImpl(true) {
                    @Override
                    public void onSuccess(int statusCode, final byte[] data) {
                        TranssionPoolManager.getInstance().addTask(new Runnable() {
                            @Override
                            public void run() {
                                if (webData != null) {
                                    String assetPath = WebAssetsManager.saveAssetToFile(data, webData.getUrl());
                                    AdLogUtil.Log().d("webview--", "IdleDownloadTask save assetPath = " + assetPath + " time = " + (System.currentTimeMillis() - start));
                                }
                                onSuccessFinal(webData.getUrl());
                            }
                        });
                    }

                    @Override
                    public void onFailure(int statusCode, byte[] bytes, Throwable throwable) {
                        AdLogUtil.Log().d("webview--", "IdleDownloadTask download fail url = " + webData.getUrl());
                        if (mCallback != null && throwable != null) {
                            mCallback.onFail(statusCode, throwable.getMessage());
                        }
                    }

                    @Override
                    public void onSuccess(int statusCode, byte[] responseBody, final String filePath) {
                        AdLogUtil.Log().d("webview--", "IdleDownloadTask download success url = " + webData.getUrl() + " path = " + filePath);
                        onSuccessFinal(webData.getUrl());
                    }
                });
    }
}
