package com.cloud.hisavana.sdk.database;

import static com.cloud.hisavana.sdk.common.constant.Constants.CONTENT_ADS_DATA;
import static com.cloud.hisavana.sdk.common.constant.Constants.CONTENT_CLOUD_CONFIG;

import android.content.ContentProvider;
import android.content.ContentValues;
import android.content.UriMatcher;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.net.Uri;
import android.os.Handler;
import android.text.TextUtils;
import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.cloud.hisavana.sdk.common.AdLogUtil;
import com.cloud.hisavana.sdk.common.athena.AthenaTracker;
import com.cloud.hisavana.sdk.common.constant.Constants;
import com.cloud.hisavana.sdk.common.util.TimeUtil;
import com.cloud.hisavana.sdk.data.bean.response.AdsDTO;
import com.cloud.hisavana.sdk.data.bean.response.ConfigCodeSeatDTO;
import com.cloud.sdk.commonutil.control.AdxPreferencesHelper;
import com.cloud.sdk.commonutil.gsonutil.GsonUtil;
import com.google.gson.reflect.TypeToken;
import com.transsion.core.CoreUtil;

import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.List;

public class HisavanaContentProvider extends ContentProvider {
    private static final String TAG = "HisavanaContentProvider";
    private static final String CONFIG_PATH = "config";
    private static final String AD_PATH = "ad_data";
    private static final int CONFIG_SUCCESS = 100;
    private static final int AD_SUCCESS = 200;
    private static final int DELAY_INIT_TIME = 5000;
    private String authorities;
    private UriMatcher uriMatcher;
    private SQLiteDatabase database;

    private void initUriMatcher() {
        if (uriMatcher == null) {
            authorities = getContext().getPackageName() + ".HisavanaContentProvider";
            uriMatcher = new UriMatcher(UriMatcher.NO_MATCH);
            uriMatcher.addURI(authorities, CONFIG_PATH, CONFIG_SUCCESS);
            uriMatcher.addURI(authorities, AD_PATH, AD_SUCCESS);
        }
    }

    @Override
    public boolean onCreate() {
        new Handler().postDelayed(new Runnable() {
            @Override
            public void run() {
                try {
                    CoreUtil.init(getContext());
                    int initType = AdxPreferencesHelper.getInstance().getInt(Constants.IS_DEBUG, Constants.InitType.NOT_INIT);
                    if (initType != Constants.InitType.NOT_INIT) {
                        AthenaTracker.init(initType == Constants.InitType.DEBUG, getContext());
                    }
                } catch (Exception exception) {
                    AdLogUtil.Log().e(TAG, Log.getStackTraceString(exception));
                }
            }
        }, DELAY_INIT_TIME);
        initUriMatcher();
        return false;
    }

    @Nullable
    @Override
    public Cursor query(@NonNull Uri uri, @Nullable String[] projection, @Nullable String selection, @Nullable String[] selectionArgs, @Nullable String sortOrder) {
        openDB();
        Cursor cursor = null;
        try {
            cursor = database.rawQuery(selection, null);
        } catch (Exception e) {
            AdLogUtil.Log().e(TAG, Log.getStackTraceString(e));
        }
        return cursor;
    }

    @Nullable
    @Override
    public String getType(@NonNull Uri uri) {
        return null;
    }

    @Nullable
    @Override
    public Uri insert(@NonNull Uri uri, @Nullable ContentValues values) {
        AdLogUtil.Log().d(TAG, "insert " + uri + " " + uriMatcher.match(uri));
        boolean isSuccess = false;
        String path = "";
        try {
            openDB();
            if (uriMatcher.match(uri) == CONFIG_SUCCESS) {
                path = CONFIG_PATH;
                //云控插入
                if (values != null) {
                    String content = values.getAsString(CONTENT_CLOUD_CONFIG);
                    if (!TextUtils.isEmpty(content)) {
                        Type type = new TypeToken<List<ConfigCodeSeatDTO>>() {
                        }.getType();
                        List<ConfigCodeSeatDTO> list = GsonUtil.fromJson(content, type);
                        isSuccess = clearAndInsertConfig(list);
                    }
                }
            } else if (uriMatcher.match(uri) == AD_SUCCESS) {
                path = AD_PATH;
                //广告数据插入
                if (values != null) {
                    String content = values.getAsString(CONTENT_ADS_DATA);
                    if (!TextUtils.isEmpty(content)) {
                        Type type = new TypeToken<List<AdsDTO>>() {
                        }.getType();
                        List<AdsDTO> adsDTOList = GsonUtil.fromJson(content, type);
                        isSuccess = insertOrUpdateAds(adsDTOList);
                    }
                }
            } else {
                AdLogUtil.Log().d(TAG, "insert,uri is wrong");
            }
        } catch (Exception e) {
            AdLogUtil.Log().e(TAG, Log.getStackTraceString(e));
        }
        return Uri.parse("content://" + authorities + "/" + path + "/" + isSuccess);
    }

    @Override
    public int delete(@NonNull Uri uri, @Nullable String selection, @Nullable String[] selectionArgs) {
        AdLogUtil.Log().d(TAG, "delete ");
        try {
            if (uriMatcher.match(uri) == AD_SUCCESS) {
                if (selectionArgs != null && selectionArgs.length > 0) {
                    StringBuilder param = new StringBuilder("(");
                    for (int i = 0; i < selectionArgs.length; i++) {
                        if (i == selectionArgs.length - 1) {
                            param.append("'").append(selectionArgs[i]).append("'");
                        } else {
                            param.append("'").append(selectionArgs[i]).append("',");
                        }
                    }
                    param.append(")");
                    String sql = "DELETE FROM " + HisavanaDatabaseHelper.AD_DATA_TABLE + " WHERE " + HisavanaDatabaseHelper._ID + " in " + param;
                    openDB();
                    database.execSQL(sql);
                }
            }
        } catch (Exception exception) {
            AdLogUtil.Log().e(TAG, "delete " + Log.getStackTraceString(exception));
        }
        return 0;
    }

    @Override
    public int update(@NonNull Uri uri, @Nullable ContentValues values, @Nullable String selection, @Nullable String[] selectionArgs) {
        AdLogUtil.Log().d(TAG, "update " + values);
        try {
            if (uriMatcher.match(uri) == CONFIG_SUCCESS) {
                //config更新
                String json = null;
                if (values != null) {
                    json = values.getAsString(CONTENT_CLOUD_CONFIG);
                }
                if (!TextUtils.isEmpty(json)) {
                    Type type = new TypeToken<List<ConfigCodeSeatDTO>>() {
                    }.getType();
                    List<ConfigCodeSeatDTO> list = GsonUtil.fromJson(json, type);
                    if (list != null && !list.isEmpty()) {
                        boolean result = updateConfig(list);
                        return result ? 1 : 0;
                    }
                }
            } else if (uriMatcher.match(uri) == AD_SUCCESS) {
                //广告数据更新
                String json = null;
                if (values != null) {
                    json = values.getAsString(CONTENT_ADS_DATA);
                }
                if (!TextUtils.isEmpty(json)) {
                    Type type = new TypeToken<List<AdsDTO>>() {
                    }.getType();
                    List<AdsDTO> list = GsonUtil.fromJson(json, type);
                    if (list != null) {
                        boolean result = updateAdsList(list);
                        return result ? 1 : 0;
                    }
                }
            }
        } catch (Exception exception) {
            AdLogUtil.Log().e(TAG, "update " + Log.getStackTraceString(exception));
        }
        return 0;
    }


    private boolean insertOrUpdateAds(List<AdsDTO> adsDTOList) {
        if (adsDTOList == null || adsDTOList.isEmpty()) {
            AdLogUtil.Log().d(TAG, "insertOrUpdateAds list is null or empty");
            return false;
        }
        int adsLen = adsDTOList.size();
        String[] ids = new String[adsLen];
        for (int i = 0; i < adsLen; i++) {
            AdsDTO ad = adsDTOList.get(i);
            if (ad != null) {
                ids[i] = String.valueOf(ad.getAdCreativeId());
            }
        }
        Cursor cursor = queryAdsByAdIds(ids);
        List<AdsDTO> queryList = new ArrayList<>();
        List<AdsDTO> updateList = new ArrayList<>();
        AdsDTO temp = null;
        while (cursor != null && cursor.moveToNext()) {
            int jsonIndex = cursor.getColumnIndex(HisavanaDatabaseHelper.AD_BEAN);
            if (jsonIndex >= 0) {
                String json = cursor.getString(jsonIndex);
                if (!TextUtils.isEmpty(json)) {
                    try {
                        temp = GsonUtil.fromJson(json, AdsDTO.class);
                        queryList.add(temp);
                    } catch (GsonUtil.GsonParseException e) {
                        e.printStackTrace();
                    }
                }
            }
        }
        if (cursor != null) {
            cursor.close();
        }
        boolean insertSuccess = false;
        boolean updateSuccess = true;
        if (!queryList.isEmpty()) {
            for (int i = adsLen - 1; i >= 0; i--) {
                temp = adsDTOList.get(i);
                for (int j = 0; j < queryList.size(); j++) {
                    AdsDTO update = queryList.get(j);
                    if (temp.getAdCreativeId().equals(update.getAdCreativeId()) &&
                            TextUtils.equals(temp.getCodeSeatId(), update.getCodeSeatId())) {
                        temp.setShowDate(update.getShowDate());
                        temp.setShowNum(update.getShowNum());
                        temp.setTableId(update.getTableId());
                        adsDTOList.remove(temp);
                        updateList.add(temp);
                        break;
                    }
                }
            }
            updateSuccess = updateAdsList(updateList);
        }
        insertSuccess = insertAdsList(adsDTOList);
        return insertSuccess && updateSuccess;
    }

    private Cursor queryAdsByCodeSeatId(String configId) {
        openDB();
        if (configId == null) {
            return null;
        }
        String sql = "SELECT * FROM " + HisavanaDatabaseHelper.AD_DATA_TABLE + " WHERE " + HisavanaDatabaseHelper.CODE_SEAT_ID + "=" + configId;
        return database.rawQuery(sql, null);
    }

    private Cursor queryAdsByAdIds(String[] adIds) {
        Cursor cursor = null;
        String sql = "";
        if (adIds == null || adIds.length == 0) {
            sql = "SELECT * FROM " + HisavanaDatabaseHelper.AD_DATA_TABLE;
        } else {
            StringBuilder param = new StringBuilder("(");
            for (int i = 0; i < adIds.length; i++) {
                if (i == adIds.length - 1) {
                    param.append("'").append(adIds[i]).append("'");
                } else {
                    param.append("'").append(adIds[i]).append("',");
                }
            }
            param.append(")");
            sql = "select * from " + HisavanaDatabaseHelper.AD_DATA_TABLE + " where " + HisavanaDatabaseHelper.AD_ID + " IN " + param;
        }
        try {
            openDB();
            AdLogUtil.Log().d(TAG, "getCursorForProvider sql " + sql);
            cursor = database.rawQuery(sql, null);
        } catch (Exception e) {
            AdLogUtil.Log().e(TAG, "getCursorForProvider e " + Log.getStackTraceString(e));
        }
        return cursor;
    }

    private boolean insertAdsList(List<AdsDTO> insertList) {
        if (insertList == null || insertList.isEmpty()) {
            return false;
        }
        openDB();
        try {
            database.beginTransaction();
            long current = TimeUtil.getZeroClockTimestamp(System.currentTimeMillis());
            for (AdsDTO adsDTO : insertList) {
                adsDTO.setShowDate(current);
                String json = GsonUtil.toJson(adsDTO);
                ContentValues contentValues = new ContentValues();
                contentValues.put(HisavanaDatabaseHelper.AD_ID, String.valueOf(adsDTO.getAdCreativeId()));
                contentValues.put(HisavanaDatabaseHelper.CODE_SEAT_ID, adsDTO.getCodeSeatId());
                contentValues.put(HisavanaDatabaseHelper.FILE_PATH, adsDTO.getFilePath());
                contentValues.put(HisavanaDatabaseHelper.PRICE, adsDTO.getFirstPrice());
                contentValues.put(HisavanaDatabaseHelper.AD_BEAN, json);
                database.insert(HisavanaDatabaseHelper.AD_DATA_TABLE, null, contentValues);
            }
            database.setTransactionSuccessful();
        } catch (Exception exception) {
            AdLogUtil.Log().e(TAG, "insert Config fail " + Log.getStackTraceString(exception));
            return false;
        } finally {
            database.endTransaction();
        }
        return true;
    }

    private boolean updateAdsList(List<AdsDTO> updateList) {
        if (updateList == null || updateList.isEmpty()) {
            AdLogUtil.Log().d(TAG, "updateAdsList list is empty ");
            return false;
        }
        AdLogUtil.Log().d(TAG, "updateAdsList size " + updateList.size());
        openDB();
        try {
            database.beginTransaction();
            for (AdsDTO adsDTO : updateList) {
                String json = GsonUtil.toJson(adsDTO);
                ContentValues contentValues = new ContentValues();
                contentValues.put(HisavanaDatabaseHelper.AD_BEAN, json);
                String tableId = String.valueOf(adsDTO.getTableId());
                int result = database.update(HisavanaDatabaseHelper.AD_DATA_TABLE, contentValues, HisavanaDatabaseHelper._ID + " =?", new String[]{tableId});
                AdLogUtil.Log().d(TAG, "updateAdsList result " + result);
            }
            database.setTransactionSuccessful();
        } catch (Exception exception) {
            AdLogUtil.Log().e(TAG, "insert Config fail " + Log.getStackTraceString(exception));
            return false;
        } finally {
            database.endTransaction();
        }
        return true;
    }

    /**
     * content provider 获取云控配置
     *
     * @param selectionArgs 代码位id数组
     * @return cursor
     */
    private Cursor getCursorForConfig(String[] selectionArgs) {
        Cursor cursor = null;
        String sql = "";
        if (selectionArgs == null || selectionArgs.length == 0) {
            sql = "select * from " + HisavanaDatabaseHelper.CLOUD_TABLE;
        } else {
            StringBuilder param = new StringBuilder("(");
            for (int i = 0; i < selectionArgs.length; i++) {
                AdLogUtil.Log().d(TAG, "getCursorForProvider ***** " + selectionArgs[i]);
                if (i == selectionArgs.length - 1) {
                    param.append("'").append(selectionArgs[i]).append("'");
                } else {
                    param.append("'").append(selectionArgs[i]).append("',");
                }
            }
            param.append(")");
            sql = "select * from " + HisavanaDatabaseHelper.CLOUD_TABLE + " where codeSeatId IN " + param;
        }
        try {
            openDB();
            AdLogUtil.Log().d(TAG, "getCursorForProvider sql " + sql);
            cursor = database.rawQuery(sql, null);
        } catch (Exception e) {
            AdLogUtil.Log().e(TAG, "getCursorForProvider e " + Log.getStackTraceString(e));
        }
        return cursor;
    }

    private boolean clearAndInsertConfig(List<ConfigCodeSeatDTO> insertList) {
        AdLogUtil.Log().d(TAG, "insertOrUpdateConfig " + insertList);
        boolean result = false;
        openDB();
        database.execSQL("delete from " + HisavanaDatabaseHelper.CLOUD_TABLE + " where 1=1");
        result = insertConfig(insertList);
        return result;
    }

    private boolean insertConfig(List<ConfigCodeSeatDTO> insertList) {
        if (insertList == null || insertList.isEmpty()) {
            return false;
        }
        openDB();
        try {
            database.beginTransaction();
            for (ConfigCodeSeatDTO config : insertList) {
                String json = GsonUtil.toJson(config);
                ContentValues contentValues = new ContentValues();
                contentValues.put(HisavanaDatabaseHelper.CODE_SEAT_ID, config.getCodeSeatId());
                contentValues.put(HisavanaDatabaseHelper.CODE_SEAT_BEAN, json);
                database.insert(HisavanaDatabaseHelper.CLOUD_TABLE, null, contentValues);
            }
            database.setTransactionSuccessful();
        } catch (Exception exception) {
            AdLogUtil.Log().e(TAG, "insert Config fail " + Log.getStackTraceString(exception));
            return false;
        } finally {
            database.endTransaction();
        }
        return true;
    }

    private boolean updateConfig(List<ConfigCodeSeatDTO> updateList) {
        if (updateList == null || updateList.isEmpty()) {
            return false;
        }
        try {
            openDB();
            database.beginTransaction();
            for (ConfigCodeSeatDTO config : updateList) {
                String json = GsonUtil.toJson(config);
                ContentValues contentValues = new ContentValues();
                contentValues.put(HisavanaDatabaseHelper.CODE_SEAT_ID, config.getCodeSeatId());
                contentValues.put(HisavanaDatabaseHelper.CODE_SEAT_BEAN, json);
                int result = database.update(HisavanaDatabaseHelper.CLOUD_TABLE, contentValues, HisavanaDatabaseHelper.CODE_SEAT_ID + "=?", new String[]{config.getCodeSeatId()});
                AdLogUtil.Log().d(TAG, "updateConfig result " + result);
            }
            database.setTransactionSuccessful();
        } catch (Exception exception) {
            AdLogUtil.Log().e(TAG, "update config fail" + Log.getStackTraceString(exception));
            return false;
        } finally {
            database.endTransaction();
        }
        return true;
    }

    /**
     * 初始化 数据库
     */
    private synchronized SQLiteDatabase openDB() {
        if (database == null || !database.isOpen()) {
            try {
                HisavanaDatabaseHelper dbHelper = new HisavanaDatabaseHelper(CoreUtil.getContext());
                database = dbHelper.getWritableDatabase();
            } catch (Exception ex) {
                AdLogUtil.Log().e(TAG, "openDB ex " + Log.getStackTraceString(ex));
            }
        }
        return database;
    }
}
