package com.cloud.hisavana.sdk.manager;

import static com.cloud.hisavana.sdk.common.constant.Constants.CONTENT_CLOUD_CONFIG;
import static com.cloud.hisavana.sdk.database.HisavanaDatabaseHelper.CLOUD_TABLE;

import android.content.ContentResolver;
import android.content.ContentValues;
import android.database.Cursor;
import android.net.Uri;
import android.text.TextUtils;
import android.util.Log;

import com.cloud.hisavana.sdk.api.config.AdManager;
import com.cloud.hisavana.sdk.common.AdLogUtil;
import com.cloud.hisavana.sdk.common.constant.Constants;
import com.cloud.hisavana.sdk.common.util.TimeUtil;
import com.cloud.hisavana.sdk.data.bean.response.ConfigCodeSeatDTO;
import com.cloud.hisavana.sdk.database.HisavanaDatabaseHelper;
import com.cloud.sdk.commonutil.gsonutil.GsonUtil;
import com.cloud.sdk.commonutil.util.Preconditions;
import com.transsion.core.CoreUtil;
import com.transsion.core.pool.TranssionPoolManager;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

public class ConfigProviderManager {
    private static final String TAG = "CloudProviderManager";
    private final Uri configUri;
    private static volatile ConfigProviderManager sInstance;
    private Map<String, ConfigCodeSeatDTO> configCache;

    private ConfigProviderManager() {
        configUri = Uri.parse("content://" + CoreUtil.getContext().getPackageName() + ".HisavanaContentProvider/config");
    }

    public static ConfigProviderManager getInstance() {
        if (sInstance == null) {
            synchronized (HisavanaDatabaseHelper.class) {
                if (sInstance == null) {
                    sInstance = new ConfigProviderManager();
                }
            }
        }
        return sInstance;
    }

    /**
     * 云控数据插入 建议子线程调用
     *
     * @param codeSeats 广告位列表
     * @return 插入结果
     */
    public boolean insertAtSubThread(List<ConfigCodeSeatDTO> codeSeats) {
        boolean result = false;
        if (codeSeats == null || codeSeats.isEmpty()) {
            return result;
        }
        if (configCache == null) {
            getAllConfigAtSubThread();
        }
        Collection<ConfigCodeSeatDTO> oldConfigs = configCache.values();
        long current = System.currentTimeMillis();
        List<ConfigCodeSeatDTO> tempList = new ArrayList<>();
        for (ConfigCodeSeatDTO config : codeSeats) {
            for (ConfigCodeSeatDTO old : oldConfigs) {
                if (TextUtils.equals(config.getCodeSeatId(), old.getCodeSeatId())) {
                    long dateTime = Long.parseLong(old.getDate());
                    if ((current - dateTime) >= Constants.DAY_TIME_MS) {
                        config.setCurrentShowTimes(0);
                        config.setDate(String.valueOf(TimeUtil.getZeroClockTimestamp(current)));
                    } else {
                        config.setDate(old.getDate());
                        config.setCurrentShowTimes(old.getCurrentShowTimes());
                    }
                    config.setShowIndex(old.getShowIndex());
                    config.setLastRequestAdTime(old.getLastRequestAdTime());
                    config.setAdRequestFail(old.isAdRequestFail());
                    tempList.add(config);
                    break;
                }
            }
        }
        if (!tempList.isEmpty()) {
            for (ConfigCodeSeatDTO config : tempList) {
                if (config != null && configCache != null) {
                    configCache.put(config.getCodeSeatId(), config);
                }
            }
        }
        try {
            ContentResolver contentResolver = CoreUtil.getContext().getContentResolver();
            ContentValues contentValues = new ContentValues();
            contentValues.put(Constants.CONTENT_CLOUD_CONFIG, GsonUtil.toJson(codeSeats));
            Uri uri = contentResolver.insert(configUri, contentValues);
            result = uri.toString().contains("true");
        } catch (Exception exception) {
            AdLogUtil.Log().e(TAG, Log.getStackTraceString(exception));
            result = false;
        }
        return result;
    }

    /**
     * 该方法可以在主线程调用  云控数据会通过listener在主线程回调给调用方
     *
     * @param codeSeatId 云控id
     * @param listener   云控回调
     */
    public void getConfigById(final String codeSeatId, final ConfigCachedListener listener) {
        AdLogUtil.Log().d(TAG, "getConfigById " + codeSeatId);
        if (configCache == null || configCache.isEmpty()) {
            WorkThread.getInstance().post(new Runnable() {
                @Override
                public void run() {
                    getAllConfigAtSubThread();
                    Preconditions.runOnMainThread(new Preconditions.Callback() {
                        @Override
                        public void onRun() {
                            if (configCache != null && listener != null) {
                                ConfigCodeSeatDTO config = configCache.get(codeSeatId);
                                resetConfigShowTime(config, null);
                                AdLogUtil.Log().d(TAG, "getConfigById cache  " + (AdManager.isDebug() ? "" : configCache));
                                listener.onConfigLoad(config);
                            }
                        }
                    });
                }

            });
        } else {
            Preconditions.runOnMainThread(new Preconditions.Callback() {
                @Override
                public void onRun() {
                    if (configCache != null && listener != null) {
                        ConfigCodeSeatDTO config = configCache.get(codeSeatId);
                        resetConfigShowTime(config, null);
                        AdLogUtil.Log().d(TAG, "getConfigById cache  " + (AdManager.isDebug() ? "" : configCache));
                        listener.onConfigLoad(config);
                    }
                }
            });
        }

    }

    /**
     * 云控展示次数+1
     * 更新云控到数据库中
     *
     * @param codeSeatId 云控id
     */
    public void updateConfigShowTimes(String codeSeatId) {
        AdLogUtil.Log().d(TAG, "updateConfigData ");
        if (TextUtils.isEmpty(codeSeatId)) {
            return;
        }
        if (configCache == null) {
            return;
        }
        ConfigCodeSeatDTO configDTO = configCache.get(codeSeatId);
        if (configDTO == null) {
            return;
        }
        configDTO.setCurrentShowTimes(configDTO.getCurrentShowTimes() + 1);
        configDTO.setShowIndex(configDTO.getTempIndex());
        updateConfigData(configDTO);
    }

    /**
     * 获取所有云控数据 可能会耗时 建议子线程调用
     *
     * @return 所有云控数据
     */
    public Map<String, ConfigCodeSeatDTO> getAllConfigAtSubThread() {
        AdLogUtil.Log().d(TAG, "getAllConfig ");
        if (configCache == null || configCache.isEmpty()) {
            configCache = new ConcurrentHashMap<>();
            Cursor cursor = null;
            ContentResolver contentResolver = CoreUtil.getContext().getContentResolver();
            String selection = "select * from " + CLOUD_TABLE;
            cursor = contentResolver.query(configUri, null, selection, null, null);
            if (cursor != null) {
                try {
                    while (cursor.moveToNext()) {
                        int codeSeatBeanIndex = cursor.getColumnIndex(HisavanaDatabaseHelper.CODE_SEAT_BEAN);
                        String json = cursor.getString(codeSeatBeanIndex);
                        ConfigCodeSeatDTO temp = GsonUtil.fromJson(json, ConfigCodeSeatDTO.class);
                        if (temp == null) {
                            continue;
                        }
                        int currentShowTimesIndex = cursor.getColumnIndex(HisavanaDatabaseHelper.CURRENT_CODESEAT_SHOW_TIMES);
                        if (currentShowTimesIndex >= 0) {
                            int currentShowTimes = cursor.getInt(currentShowTimesIndex);
                            temp.setCurrentShowTimes(currentShowTimes);
                        }
                        int dateTimeIndex = cursor.getColumnIndex(HisavanaDatabaseHelper.DATETIME);
                        if (dateTimeIndex >= 0) {
                            String date = cursor.getString(dateTimeIndex);
                            temp.setDate(date);
                        }
                        int currentIndexIndex = cursor.getColumnIndex(HisavanaDatabaseHelper.CURRENT_INDEX);
                        if (currentIndexIndex >= 0) {
                            int current = cursor.getInt(currentIndexIndex);
                            temp.setShowIndex(current);
                        }
                        configCache.put(temp.getCodeSeatId(), temp);
                    }
                } catch (Exception exception) {
                    AdLogUtil.Log().e(TAG, Log.getStackTraceString(exception));
                } finally {
                    cursor.close();
                }
            }
            resetConfigListShowTime(configCache.values());
        }
        AdLogUtil.Log().d(TAG, "getAllConfig " + configCache.size());
        return configCache;
    }

    /**
     * 重置云控第二天展示次数为0
     *
     * @param configCodeSeatDTOS 云控列表
     */
    private void resetConfigListShowTime(Collection<ConfigCodeSeatDTO> configCodeSeatDTOS) {
        if (configCodeSeatDTOS == null || configCodeSeatDTOS.size() <= 0) {
            return;
        }
        try {
            List<ConfigCodeSeatDTO> updateList = new ArrayList<>();
            for (ConfigCodeSeatDTO config : configCodeSeatDTOS) {
                resetConfigShowTime(config, updateList);
            }
            if (!updateList.isEmpty()) {
                ContentResolver contentResolver = CoreUtil.getContext().getContentResolver();
                ContentValues contentValues = new ContentValues();
                contentValues.put(CONTENT_CLOUD_CONFIG, GsonUtil.toJson(updateList));
                contentResolver.update(configUri, contentValues, null, null);
            }
        } catch (Exception e) {
            AdLogUtil.Log().e(TAG, "resetCodeSeatConfigTimes fail");
        }
    }

    /**
     * 重置单个云控每日展现次数
     * @param config 云控
     * @param updateList 更新列表 可为空
     */
    private void resetConfigShowTime(ConfigCodeSeatDTO config, List<ConfigCodeSeatDTO> updateList) {
        if (config == null) {
            return;
        }
        long time = Long.parseLong(config.getDate());
        long current = System.currentTimeMillis();
        if (current - time > Constants.DAY_TIME_MS) {
            config.setDate(String.valueOf(TimeUtil.getZeroClockTimestamp(current)));
            if (updateList != null) {
                updateList.add(config);
            }
        }
    }

    /**
     * 更新云控数据到数据库
     *
     * @param codeSeatDTO 云控数据
     */
    public void updateConfigData(ConfigCodeSeatDTO codeSeatDTO) {
        if (codeSeatDTO == null) {
            return;
        }
        final ContentValues contentValues = new ContentValues();
        final List<ConfigCodeSeatDTO> configs = new ArrayList<>();
        configs.add(codeSeatDTO);
        contentValues.put(Constants.CONTENT_CLOUD_CONFIG, GsonUtil.toJson(configs));
        WorkThread.getInstance().post(new Runnable() {
            @Override
            public void run() {
                try {
                    ContentResolver contentResolver = CoreUtil.getContext().getContentResolver();
                    int result = contentResolver.update(configUri, contentValues, null, null);
                    AdLogUtil.Log().d(TAG, "updateConfigData result " + result);
                } catch (Exception exception) {
                    AdLogUtil.Log().d(TAG, "updateConfigData " + Log.getStackTraceString(exception));
                }
            }
        });

    }

    public interface ConfigCachedListener {
        void onConfigLoad(ConfigCodeSeatDTO codeSeatDTO);
    }
}
