package com.cloud.hisavana.sdk.manager;

import static com.cloud.hisavana.sdk.common.constant.Constants.CONTENT_ADS_DATA;

import android.content.ContentResolver;
import android.content.ContentValues;
import android.database.Cursor;
import android.net.Uri;
import android.text.TextUtils;
import android.util.Log;

import com.cloud.hisavana.sdk.common.AdLogUtil;
import com.cloud.hisavana.sdk.common.constant.Constants;
import com.cloud.hisavana.sdk.common.util.OfflineAdExpiredUtil;
import com.cloud.hisavana.sdk.common.util.TimeUtil;
import com.cloud.hisavana.sdk.data.bean.response.AdsDTO;
import com.cloud.hisavana.sdk.data.bean.response.ConfigCodeSeatDTO;
import com.cloud.hisavana.sdk.database.HisavanaDatabaseHelper;
import com.cloud.sdk.commonutil.gsonutil.GsonUtil;
import com.cloud.sdk.commonutil.util.Preconditions;
import com.transsion.core.CoreUtil;
import com.transsion.core.pool.TranssionPoolManager;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

public class OfflineProviderManager {
    private static final String TAG = "OfflineProviderManager";
    private final Uri adDataUri;
    private static volatile OfflineProviderManager sInstance;

    private OfflineProviderManager() {
        adDataUri = Uri.parse("content://" + CoreUtil.getContext().getPackageName() + ".HisavanaContentProvider/ad_data");
    }

    public static OfflineProviderManager getInstance() {
        if (sInstance == null) {
            synchronized (HisavanaDatabaseHelper.class) {
                if (sInstance == null) {
                    sInstance = new OfflineProviderManager();
                }
            }
        }
        return sInstance;
    }

    /**
     * 数据库插入广告列表
     * 建议子线程调用
     *
     * @param adsDTOList 广告列表
     * @return 插入成功或失败
     */
    public boolean insertAdsListAtSubThread(List<AdsDTO> adsDTOList) {
        AdLogUtil.Log().d(TAG, "insertAdsList");
        boolean result = false;
        if (adsDTOList == null || adsDTOList.isEmpty()) {
            return result;
        }
        List<AdsDTO> tempList = new ArrayList<>(adsDTOList);
        try {
            Iterator<AdsDTO> iterator = tempList.listIterator();
            //判断广告是否过期
            while (iterator.hasNext()) {
                if (OfflineAdExpiredUtil.idOfflineAdEnd(iterator.next())) {
                    iterator.remove();
                }
            }
            ContentResolver contentResolver = CoreUtil.getContext().getContentResolver();
            ContentValues contentValues = new ContentValues();
            contentValues.put(Constants.CONTENT_ADS_DATA, GsonUtil.toJson(tempList));
            Uri uri = contentResolver.insert(adDataUri, contentValues);
            if (uri != null) {
                result = uri.toString().endsWith("true");
            }
        } catch (Exception exception) {
            AdLogUtil.Log().e(TAG, Log.getStackTraceString(exception));
            result = false;
        }
        return result;
    }

    /**
     * 获取广告by广告位id
     *
     * @param codeSeatId 广告位id
     * @return 广告列表
     */
    public List<AdsDTO> getAdListByCodeSeatIdAtSubThread(String codeSeatId) {
        if (TextUtils.isEmpty(codeSeatId)) {
            AdLogUtil.Log().d(TAG, "getAdByCodeSeatId code seat id is null");
            return null;
        }
        ContentResolver contentResolver = CoreUtil.getContext().getContentResolver();
        String sql = "SELECT * FROM " + HisavanaDatabaseHelper.AD_DATA_TABLE +
                " WHERE " + HisavanaDatabaseHelper.CODE_SEAT_ID + " = '" + codeSeatId + "' ORDER by " + HisavanaDatabaseHelper.PRICE + " DESC;";
        Cursor cursor = null;
        List<AdsDTO> result = new ArrayList<>();
        try {
            cursor = contentResolver.query(adDataUri, null, sql, null, null);
            if (cursor == null) {
                return null;
            }
            while (cursor.moveToNext()) {
                int adBeanIndex = cursor.getColumnIndex(HisavanaDatabaseHelper.AD_BEAN);
                int tableIdIndex = cursor.getColumnIndex(HisavanaDatabaseHelper._ID);
                if (adBeanIndex >= 0) {
                    String json = cursor.getString(adBeanIndex);
                    if (!TextUtils.isEmpty(json)) {
                        AdsDTO adsDTO = GsonUtil.fromJson(json, AdsDTO.class);
                        result.add(adsDTO);
                        if (tableIdIndex >= 0) {
                            int tableId = cursor.getInt(tableIdIndex);
                            adsDTO.setTableId(tableId);
                        }
                    }
                }
            }
            if (!result.isEmpty()) {
                //广告列表不为空 判断广告当天展示次数是否到了第二天
                long current = System.currentTimeMillis();
                List<AdsDTO> updateList = new ArrayList<>();
                for (AdsDTO ads : result) {
                    judgeDateShowTimes(ads, current, updateList);
                }
                AdLogUtil.Log().d(TAG, "getAdListByCodeSeatId updateList " + updateList);
                if (!updateList.isEmpty()) {
                    //更新列表不为空 则将数据更新到数据库
                    ContentResolver resolver = CoreUtil.getContext().getContentResolver();
                    ContentValues contentValues = new ContentValues();
                    String json = GsonUtil.toJson(updateList);
                    contentValues.put(CONTENT_ADS_DATA, json);
                    resolver.update(adDataUri, contentValues, null, null);
                }
            }
        } catch (Exception e) {
            AdLogUtil.Log().e(TAG, Log.getStackTraceString(e));
            result.clear();
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
        return result;
    }

    /**
     * 获取广告
     * 循环遍历删除过期广告
     *
     * @param codeSeatId 广告位id
     * @return 广告
     */
    public void getAdByCodeSeatId(final String codeSeatId, final LoadAdListener listener) {
        AdLogUtil.Log().d(TAG,"getAdByCodeSeatId "+codeSeatId);
        if (TextUtils.isEmpty(codeSeatId)) {
            return;
        }
        WorkThread.getInstance().post(new Runnable() {
            @Override
            public void run() {
                try {
                    ConfigCodeSeatDTO codeSeatDTO = ConfigProviderManager.getInstance().getAllConfigAtSubThread().get(codeSeatId);
                    if (codeSeatDTO == null) {
                        return;
                    }
                    List<AdsDTO> adsList = getAdListByCodeSeatIdAtSubThread(codeSeatId);
                    if (adsList != null && !adsList.isEmpty()) {
                        int index = codeSeatDTO.getShowIndex();
                        int len = adsList.size();
                        List<String> ids = new ArrayList<>();
                        AdsDTO temp = null;
                        AdsDTO ad = null;
                        for (int i = 0; i < len; i++) {
                            ad = adsList.get(i);
                            if (ad == null){
                                continue;
                            }
                            if (OfflineAdExpiredUtil.idOfflineAdEnd(ad)) {
                                ids.add(String.valueOf(ad.getTableId()));
                            }
                            if (temp == null) {
                                index = index >= len ? 0 : index;
                                AdsDTO result = adsList.get(index);
                                if (OfflineAdExpiredUtil.idOfflineAdExpired(result)) {
                                    temp = result;
                                }
                                index++;
                            }
                        }
                        ContentResolver contentResolver = CoreUtil.getContext().getContentResolver();
                        if (!ids.isEmpty()) {
                            String[] id = new String[ids.size()];
                            for (int i = 0; i < ids.size(); i++) {
                                id[i] = ids.get(i);
                            }
                            contentResolver.delete(adDataUri, null, id);
                        }
                        codeSeatDTO.setTempIndex(index);
                        AdLogUtil.Log().d(TAG, "getAdByCodeSeatId  " + temp);
                        if (listener != null){
                            final AdsDTO finalTemp = temp;
                            Preconditions.runOnMainThread(new Preconditions.Callback() {
                                @Override
                                public void onRun() {
                                    listener.onAdLoadFromDisk(finalTemp);
                                }
                            });
                        }
                        return;
                    }
                } catch (Exception exception) {
                    AdLogUtil.Log().e(TAG, Log.getStackTraceString(exception));
                }
                if (listener != null){
                    Preconditions.runOnMainThread(new Preconditions.Callback() {
                        @Override
                        public void onRun() {
                            listener.onAdLoadFromDisk(null);
                        }
                    });
                }
            }
        });

    }

    /**
     * 更新广告新一天展示次数
     *
     * @param adsDTO  广告
     * @param current 当前时间
     */
    private void judgeDateShowTimes(AdsDTO adsDTO, long current, List<AdsDTO> updateList) {
        if (adsDTO != null && updateList != null) {
            long time = adsDTO.getShowDate();
            if ((current - time) >= Constants.DAY_TIME_MS) {
                adsDTO.setShowDate(TimeUtil.getZeroClockTimestamp(current));
                adsDTO.setShowNum(0);
                updateList.add(adsDTO);
            }
        }
    }

    /**
     * @param adsDTO 广告
     * @return 更新结果
     */
    public void updateShowTimes(AdsDTO adsDTO) {
        if (adsDTO == null) {
            return;
        }
        AdLogUtil.Log().d(TAG, "updateShowTimes getTableId " + adsDTO.getTableId() + " showNum " + adsDTO.getShowNum());
        List<AdsDTO> list = new ArrayList();
        list.add(adsDTO);
        final ContentValues contentValues = new ContentValues();
        contentValues.put(CONTENT_ADS_DATA, GsonUtil.toJson(list));
        WorkThread.getInstance().post(new Runnable() {
            @Override
            public void run() {
                ContentResolver contentResolver = CoreUtil.getContext().getContentResolver();
                try {
                    contentResolver.update(adDataUri, contentValues, null, null);
                }catch (Exception e){
                    AdLogUtil.Log().e(TAG,"updateShowTimes "+Log.getStackTraceString(e));
                }
            }
        });
    }

    /**
     * 判断广告解压文件是否有效
     *
     * @param path 文件路径
     * @return 文件是否有效 无效 false 有效 true
     */
    public boolean isFileUsefulJudgeAtSubThread(String path) {
        if (TextUtils.isEmpty(path)) {
            return false;
        }
        String sql = "SELECT * FROM " + HisavanaDatabaseHelper.AD_DATA_TABLE + " WHERE " + HisavanaDatabaseHelper.FILE_PATH + " = " + "'" + path + "'";
        try {
            Cursor cursor = CoreUtil.getContext().getContentResolver().query(adDataUri, null, sql, null, null);
            if (cursor == null) {
                return false;
            }
            boolean result = cursor.getCount() > 0;
            cursor.close();
            return result;
        } catch (Exception e) {
            AdLogUtil.Log().e(TAG, Log.getStackTraceString(e));
        }
        return false;
    }

    public interface LoadAdListener{
        void onAdLoadFromDisk(AdsDTO adsDTO);
    }
}
