package com.cloud.hisavana.sdk.offline;

import static com.cloud.hisavana.sdk.common.constant.Constants.ImageLoadType.IMAGE_DISK_INCLUDED;

import android.text.TextUtils;

import com.cloud.hisavana.sdk.ad.base.AdType;
import com.cloud.hisavana.sdk.api.config.AdManager;
import com.cloud.hisavana.sdk.common.AdLogUtil;
import com.cloud.hisavana.sdk.common.athena.AthenaTracker;
import com.cloud.hisavana.sdk.common.bean.AdImage;
import com.cloud.hisavana.sdk.common.constant.Constants;
import com.cloud.hisavana.sdk.common.constant.TaErrorCode;
import com.cloud.hisavana.sdk.common.http.DownLoadRequest;
import com.cloud.hisavana.sdk.common.http.listener.DrawableResponseListener;
import com.cloud.hisavana.sdk.common.util.FileUtil;
import com.cloud.hisavana.sdk.common.util.MD5Utils;
import com.cloud.hisavana.sdk.common.util.ZipUtil;
import com.cloud.hisavana.sdk.data.bean.request.AdxImpBean;
import com.cloud.hisavana.sdk.data.bean.response.AdsDTO;
import com.transsion.core.CoreUtil;

import java.io.File;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * Created  ON 2022/7/11
 *
 * @author :fangxuhui
 */
public class OfflineDownload {

    private final static String TAG = "OfflineDownload";
    public final static String H5_ZIP_FOLDER_NAME = "offline_zip";

    private DownLoadCallBack callBack;

    private static class SingletonInner {
        private static final OfflineDownload instance = new OfflineDownload();
    }

    public static OfflineDownload getInstance() {
        return SingletonInner.instance;
    }

    /**
     * 下载广告列表中的素材和离线包
     * 子线程调用
     *
     * @param adsDTOs
     * @param callBack
     */
    public void downLoad(final List<AdsDTO> adsDTOs, DownLoadCallBack callBack) {
        this.callBack = callBack;
        AdLogUtil.Log().d(TAG, "准备下载 ======");
        if (adsDTOs == null || adsDTOs.isEmpty()) {
            AdLogUtil.Log().d(TAG, "广告list为空");
            onComplete();
            return;
        }
        AdLogUtil.Log().d(TAG, "准备下载 ======" + adsDTOs.size());
        DownLoadOffLineAd temp = null;
        DownLoadOffLineAd head = null;
        for (AdsDTO impl : adsDTOs) {
            if (impl == null) {
                continue;
            }
            DownLoadOffLineAd next = new DownLoadOffLineAd(impl);
            if (temp == null) {
                head = next;
            } else {
                temp.nextExecutor(next);
            }
            temp = next;
        }
        if (head == null) {
            AdLogUtil.Log().d(TAG, "下载失败");
            onComplete();
            return;
        }
        AdLogUtil.Log().d(TAG, "开始下载广告 === " + (AdManager.isDebug() ? adsDTOs : ""));
        downLoadAd(head, new LoadMaterialCallback() {
            @Override
            public void loadFinish() {
                AdLogUtil.Log().d(TAG, "广告物料下载完成");
                loadAdChoiceAndLogo(adsDTOs);
                onComplete();
            }
        });
    }

    /**
     * 遍历下载广告的素材和H5
     *
     * @param downLoadOffLineAd
     */
    private void downLoadAd(final DownLoadOffLineAd downLoadOffLineAd, LoadMaterialCallback loadMaterialCallback) {
        if (downLoadOffLineAd == null) {
            loadMaterialCallback.loadFinish();
            return;
        }
        final AdsDTO adsDTO = downLoadOffLineAd.getAdsDTO();
        String imgUrl = adsDTO.getAdImgUrl();
        int type = AthenaTracker.M_TYPE_LARGE_IMG;
        if ((adsDTO.getImpBeanRequest() != null && adsDTO.getImpBeanRequest().adt == AdType.BANNER) &&
                (TextUtils.equals(adsDTO.getMaterialStyle(), Constants.BANNER_MATERIAL_STYLE_B20301)
                        || TextUtils.equals(adsDTO.getMaterialStyle(), Constants.BANNER_MATERIAL_STYLE_B20302)
                        || TextUtils.equals(adsDTO.getMaterialStyle(), Constants.BANNER_MATERIAL_STYLE_B20303))) {
            AdLogUtil.Log().d(TAG, "该广告为banner且需下载logo");
            imgUrl = adsDTO.getLogoUrl();
            type = AthenaTracker.M_TYPE_LOGO;
        }
        if (adsDTO == null || TextUtils.isEmpty(imgUrl)) {
            downLoadAd(downLoadOffLineAd.getNext(), loadMaterialCallback);
            return;
        }
        final AtomicInteger loadCount = new AtomicInteger(2);

        AdLogUtil.Log().d(TAG, "开始下载主图 ==" + imgUrl);
        downLoadMaterial(adsDTO, imgUrl,
                type, false, downLoadOffLineAd, loadCount, loadMaterialCallback);

        String h5Url = adsDTO.getOfflineH5Url();
        AdLogUtil.Log().d(TAG, "开始下载离线包 ==" + adsDTO.getOfflineH5Url() + " 是否zip" + adsDTO.isH5Zip());
        if (TextUtils.isEmpty(h5Url)) {
            downLoadNext(loadCount, downLoadOffLineAd, loadMaterialCallback);
        } else {
            downLoadMaterial(adsDTO, h5Url,
                    AthenaTracker.M_TYPE_AD_WEB, adsDTO.isH5Zip(), downLoadOffLineAd, loadCount, loadMaterialCallback);
        }
    }


    /**
     * 下载物料
     *
     * @param adsDTO
     * @param url
     * @param type
     * @param isFetch
     * @param downLoadOffLineAd
     * @param loadCount
     */
    private void downLoadMaterial(
            final AdsDTO adsDTO,
            final String url,
            int type,
            final boolean isFetch,
            final DownLoadOffLineAd downLoadOffLineAd,
            final AtomicInteger loadCount,
            final LoadMaterialCallback loadMaterialCallback) {
        if (TextUtils.isEmpty(url)) {
            downLoadNext(loadCount, downLoadOffLineAd, loadMaterialCallback);
            return;
        }
        DownLoadRequest downLoadRequest = new DownLoadRequest()
                .setAdsDTO(adsDTO, type)
                .setUrl(url)
                .setFetch(isFetch)
                .setUsePool(true)
                .setPreCache(IMAGE_DISK_INCLUDED)
                .setListener(new DrawableResponseListener() {
                    @Override
                    public void onRequestSuccess(int statusCode, byte[] data, AdImage mediaBean) {
                        AdLogUtil.Log().d(TAG, "下载成功 == " + url);
                        if (isFetch && mediaBean != null) {
                            upZip(mediaBean.getFilePath(), url);
                        }
                        downLoadNext(loadCount, downLoadOffLineAd, loadMaterialCallback);
                    }

                    @Override
                    protected void onRequestError(TaErrorCode adError) {
                        AdLogUtil.Log().d(TAG, "下载失败 == " + url);
                        downLoadNext(loadCount, downLoadOffLineAd, loadMaterialCallback);
                    }
                });
        downLoadRequest.netRequestPreExecute();
    }


    /**
     * 请求下一个广告
     *
     * @param loadCount
     * @param downLoadOffLineAd
     */
    private void downLoadNext(AtomicInteger loadCount, DownLoadOffLineAd downLoadOffLineAd, LoadMaterialCallback loadMaterialCallback) {
        if (loadCount != null) {
            loadCount.decrementAndGet();
        }
        if (loadCount == null || loadCount.get() <= 0) {
            downLoadAd(downLoadOffLineAd == null ? null : downLoadOffLineAd.getNext(), loadMaterialCallback);
        }
    }

    /**
     * 解压文件
     *
     * @param zipPath
     * @param urlPath
     */
    private void upZip(final String zipPath, final String urlPath) {
        String outPutPath = FileUtil.getAppFilePath(CoreUtil.getContext())
                + File.separator + H5_ZIP_FOLDER_NAME
                + File.separator + MD5Utils.toMd5(urlPath);
        File file = new File(outPutPath);
        if (!file.exists()) {
            AdLogUtil.Log().d(TAG, "解压路径 ==" + outPutPath);
            ZipUtil.decompressZipFolder(zipPath, outPutPath);
        } else {
            AdLogUtil.Log().d(TAG, "解压文件已存在 ==" + outPutPath);
        }
    }

    /**
     * 下载logo和adChoice
     *
     * @param adsDTOs
     */
    private void loadAdChoiceAndLogo(List<AdsDTO> adsDTOs) {
        for (AdsDTO adsDTO : adsDTOs) {
            if (adsDTO == null) {
                continue;
            }
            if (!DownLoadRequest.IS_HAD_AD_CHOICE && adsDTO.getAdChoiceImageUrl() != null) {
                DownLoadRequest downLoadRequest = new DownLoadRequest()
                        .setAdsDTO(adsDTO, AthenaTracker.M_TYPE_AD_CHOICE_VIEW)
                        .setPreCache(IMAGE_DISK_INCLUDED)
                        .setUrl(adsDTO.getAdChoiceImageUrl());
                downLoadRequest.netRequestPreExecute();
            }

            String url = adsDTO.getLogoUrl();
            int type = AthenaTracker.M_TYPE_LOGO;
            if ((adsDTO.getImpBeanRequest() != null && adsDTO.getImpBeanRequest().adt == AdType.BANNER) &&
                    (TextUtils.equals(adsDTO.getMaterialStyle(), Constants.BANNER_MATERIAL_STYLE_B20301)
                            || TextUtils.equals(adsDTO.getMaterialStyle(), Constants.BANNER_MATERIAL_STYLE_B20302)
                            || TextUtils.equals(adsDTO.getMaterialStyle(), Constants.BANNER_MATERIAL_STYLE_B20303))) {

                return;
            }
            DownLoadRequest downLoadRequest = new DownLoadRequest()
                    .setAdsDTO(adsDTO, type)
                    .setUrl(url)
                    .setPreCache(IMAGE_DISK_INCLUDED);
            downLoadRequest.netRequestPreExecute();

        }
    }

    /**
     * 加载完成的回调
     */
    private void onComplete() {
        if (callBack != null) {
            callBack.onComplete();
        }
    }

    interface LoadMaterialCallback {
        void loadFinish();
    }

    public interface DownLoadCallBack {
        /**
         * 当前请求全部完成
         */
        void onComplete();

        void onCompleteAndTrack(AdxImpBean adxImpBean, List<AdsDTO> mAdBeans, TaErrorCode adError);
    }
}
