package com.cloud.hisavana.sdk.common.http;


import static com.cloud.hisavana.sdk.common.constant.Constants.ImageLoadType.IMAGE_DISK_INCLUDED;
import static com.cloud.hisavana.sdk.common.constant.Constants.ImageLoadType.IMAGE_RAM_INCLUDED;

import android.text.TextUtils;
import android.util.Log;
import android.widget.ImageView;

import androidx.annotation.NonNull;

import com.cloud.hisavana.sdk.common.AdLogUtil;
import com.cloud.hisavana.sdk.common.athena.AthenaTracker;
import com.cloud.hisavana.sdk.common.bean.AdImage;
import com.cloud.hisavana.sdk.common.constant.Constants;
import com.cloud.hisavana.sdk.common.constant.TaErrorCode;
import com.cloud.hisavana.sdk.common.http.listener.DrawableResponseListener;
import com.cloud.hisavana.sdk.data.bean.response.AdsDTO;
import com.cloud.sdk.commonutil.util.CommonLogUtil;
import com.transsion.core.CoreUtil;
import com.transsion.http.HttpClient;
import com.transsion.http.ImageURL;
import com.transsion.http.RequestCall;
import com.transsion.http.impl.HttpCallbackImpl;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;


/**
 * Created by peng.sun on 2017/6/19.
 */

public class DownLoadRequest extends RequestBase<DrawableResponseListener> {
    private static final String TAG = "DownLoadRequest";
    public static boolean IS_HAD_AD_CHOICE = false;
    private static final int HTTP_CACHE_SUCCESS = 250;
    private static final int HTTP_CACHE_FAIL = 256;

    /***资源大小*/
    long fileSize = 0L;
    /***cdn 请求头获取*/
    String cdn = "";
    /***请求开始时间*/
    long startTime = 0L;
    /***网路请求错误代码*/
    int errorCode = -1;
    /***当前下载方式*/
    int downloadWay = AthenaTracker.DOWNLOAD_WAY_HTTP_URL_CONNECTION;
    /*** 用于埋点*/
    private AdsDTO adsDTO;
    //  物料类型 1=logo 2=大图 int
    private int mType;
    /***1. 从网络获取；2. 从本地缓存获取*/
    private int requestType = 1;
    private boolean isFetch = false;
    private boolean isOffline = false;
    private boolean isUsePool = true;
    private String url = "";
    private int isPreCache = IMAGE_RAM_INCLUDED;

    public static void downloadImage(@NonNull final String url, AdsDTO adsDTO,
                                     int materialType, final DrawableResponseListener listener) {
        downloadImage(url, IMAGE_DISK_INCLUDED, adsDTO, materialType, false, false, listener);
    }

    public static void getImage(@NonNull final String url, AdsDTO adsDTO,
                                int materialType, final DrawableResponseListener listener) {
        downloadImage(url, IMAGE_RAM_INCLUDED, adsDTO, materialType, false, false, listener);
    }

    public static void downloadFile(@NonNull final String url, int isPreCache, AdsDTO adsDTO,
                                    int materialType, boolean isFetch, final DrawableResponseListener listener) {
        downloadImage(url, isPreCache, adsDTO, materialType, isFetch, false, listener);
    }


    /**
     * 下载文件
     *
     * @param url          文件url
     * @param isPreCache   下载模式 IMAGE_RAM_INCLUDED:下载后将数据读取到内存  IMAGE_DISK_INCLUDED:缓存文件到磁盘不读取到内存中
     * @param adsDTO       ad
     * @param materialType 物料类型
     * @param isFetch      是否返回文件本地地址
     * @param usePool      是否将回调放到请求线程
     * @param listener     回调监听
     */
    public static void downloadImage(@NonNull final String url, int isPreCache, AdsDTO adsDTO,
                                     int materialType, boolean isFetch, boolean usePool, final DrawableResponseListener listener) {
        DownLoadRequest downLoadRequest = new DownLoadRequest()
                .setPreCache(isPreCache)
                .setListener(listener)
                .setUrl(url)
                .setUsePool(usePool)
                .setAdsDTO(adsDTO, materialType)
                .setFetch(isFetch);
        downLoadRequest.netRequestPreExecute();
    }

    /**
     * 加载单张图片并展示到Imageview上
     *
     * @param url          图片url
     * @param imageView    待设置的View
     * @param adsDTO       广告
     * @param materialType 物料类型
     */
    public static void renderImageView(@NonNull final String url, @NonNull final ImageView imageView, AdsDTO adsDTO, int materialType) {
        getImage(url, adsDTO, materialType, new DrawableResponseListener() {

            @Override
            public void onRequestError(TaErrorCode adError) {
                AdLogUtil.Log().e(TAG, "loadImageView " + adError);
            }

            @Override
            public void onRequestSuccess(int statusCode, AdImage mediaBean) {
                if (mediaBean == null) {
                    return;
                }
                mediaBean.attachView(imageView);
            }
        });
    }

    private DownLoadRequest() {
    }

    public DownLoadRequest setAdsDTO(AdsDTO ads, int type) {
        this.adsDTO = ads;
        if (ads != null) {
            this.isOffline = ads.isOfflineAd();
        }
        this.mType = type;
        return this;
    }

    public DownLoadRequest setUrl(String url) {
        this.url = url;
        return this;
    }

    public DownLoadRequest setListener(DrawableResponseListener listener) {
        this.mListener = listener;
        return this;
    }

    public DownLoadRequest setPreCache(int preCache) {
        this.isPreCache = preCache;
        return this;
    }

    public DownLoadRequest setFetch(boolean isFectch) {
        this.isFetch = isFectch;
        return this;
    }

    public DownLoadRequest setUsePool(boolean usePool) {
        isUsePool = usePool;
        return this;
    }

    /**
     * 前期执行检查网络
     */
    @Override
    public void netRequestPreExecute() {
        if (TextUtils.isEmpty(url)) {
            AdLogUtil.Log().e(CommonLogUtil.TAG, "url is empty");
            return;
        }
        netRequestPosExecute();
    }

    @Override
    protected void netRequestPosExecute() {
        try {
            RequestCall requestCall = HttpClient.image(CoreUtil.getContext())
                    .log(false)
                    .cache(true)
                    .fetchFilePath(isFetch)
                    .useOffLineCache(isOffline)
                    .connectTimeout(TIMEOUT_IN_MILLIONS)
                    .readTimeout(20 * 1000)
                    .url(url)
                    .build();
            // 用于计算网络请求耗时
            startTime = System.currentTimeMillis();
            ImageURL imageURL = new ImageURL(url);
            String md5Url = imageURL.toMD5();
            AdLogUtil.Log().d(TAG, md5Url);
            requestCall.execute(new HttpCallbackImpl(isUsePool) {
                @Override
                public void onSuccess(int statusCode, byte[] data) {
                    AdLogUtil.Log().d(TAG, "DownLoadRequest --> onSuccess statusCode " + statusCode + " url " + url);
                    requestType = statusCode == HTTP_CACHE_SUCCESS ? 2 : 1;
                    if (mListener != null && mListener instanceof DrawableResponseListener) {
                        mListener.onServerRequestSuccess(isPreCache, statusCode, data, null);
                    }
                    if (null != data) {
                        fileSize = data.length;
                    }
                    if (mType == AthenaTracker.M_TYPE_AD_CHOICE_VIEW) {
                        IS_HAD_AD_CHOICE = true;
                    }
                    downloadTrack(AthenaTracker.M_STATUS_SUCCESS);
                }

                @Override
                public void onResponseHeader(Map<String, List<String>> map) {
                    super.onResponseHeader(map);
                    //AdLogUtil.Log().d(CommonLogUtil.TAG,"DownLoadRequest --> onResponseHeader");
                    // 如果是客户端读取的应该是x-response-cdn
                    // 如果是服务端应该是拿x-request-cdn
                    if (null != map && !map.isEmpty()) {
                        List<String> strings = map.get("x-response-cdn");
                        if (null != strings && !strings.isEmpty()) {
                            cdn = strings.get(0);
                        }
                    }
                }

                @Override
                public void onFailure(int statusCode, byte[] bytes, Throwable throwable) {
                    AdLogUtil.Log().d(CommonLogUtil.TAG, "DownLoadRequest --> onFailure statusCode " + statusCode + " e" + throwable + " url " + url);
                    requestType = statusCode == HTTP_CACHE_FAIL ? 2 : 1;
                    if (mListener != null) {
                        mListener.onServerRequestFailure(statusCode, bytes, throwable);
                    }
                    errorCode = statusCode;
                    downloadTrack(AthenaTracker.M_STATUS_ERROR);
                }

                @Override
                public void onSuccess(int statusCode, byte[] responseBody, final String filePath) {
                    AdLogUtil.Log().d(CommonLogUtil.TAG, "DownLoadRequest --> onSuccess statusCode " + statusCode + " filePath " + filePath + " url" + url);
                    requestType = statusCode == HTTP_CACHE_SUCCESS ? 2 : 1;
                    if (mListener != null && mListener instanceof DrawableResponseListener) {
                        mListener.onServerRequestSuccess(isPreCache, statusCode, responseBody, filePath);
                    }
                    if (null != responseBody) {
                        fileSize = responseBody.length;
                    }
                    if (mType == AthenaTracker.M_TYPE_AD_CHOICE_VIEW) {
                        IS_HAD_AD_CHOICE = true;
                    }
                    downloadTrack(AthenaTracker.M_STATUS_SUCCESS);
                }
            });
        } catch (Throwable e) {
            AdLogUtil.Log().e(Log.getStackTraceString(e));
            // 上报下载失败
            errorCode = Constants.DOWNLOAD_IMG_ERROR_CATCH_CODE;
            downloadTrack(AthenaTracker.M_STATUS_ERROR);
        }
    }


    // =============================================================================================


    /**
     * 开始下载 下载成功 下载失败 日志上报
     * info 广告详情
     * requestType 请求方式 1. 从网络获取；2. 从本地缓存获取
     * downloadWay 下载方式 1.HttpUrlConnection 2.Okhttp  int
     * mStatus     下载状态 1.开始下载 2.下载成功 3.下载失败
     * errorCode   错误信息
     * mType       资源类型
     * cdn         资源来源
     * timeConsuming 耗时
     * fileSize   文件大小
     */
    private void downloadTrack(int mStatus) {

        // 142 资源类型是 adchoice 不上报物料下载埋点事件
        if (mType == AthenaTracker.M_TYPE_AD_CHOICE_VIEW) {
            return;
        }

        AthenaTracker.trackImageDownload(adsDTO,
                requestType,
                downloadWay,
                mStatus,
                errorCode,
                mType,
                cdn,
                System.currentTimeMillis() - startTime,
                getSize(fileSize),
                isOffline ? 1 : 0);
    }

    /**
     * file_size 单位转换成 KB 上传
     *
     * @param byteSize byte size
     * @return
     */
    private int getSize(long byteSize) {
        int size = 0;
        try {
            BigDecimal sizeNew = new BigDecimal(byteSize);
            size = sizeNew.divide(BigDecimal.valueOf(1024), 0, BigDecimal.ROUND_HALF_UP).intValue();
        } catch (Exception e) {
            AdLogUtil.Log().d(CommonLogUtil.TAG, Log.getStackTraceString(e));
        }
        return size;
    }

    public interface CacheListener {
        void onSuccess();

        void onError(TaErrorCode adError);
    }
}
