package com.cloud.hisavana.sdk.common.util;

import static com.cloud.hisavana.sdk.common.constant.Constants.DAY_TIME_MS;

import android.text.TextUtils;

import com.cloud.hisavana.sdk.ad.base.AdType;
import com.cloud.hisavana.sdk.ad.base.PlatformUtil;
import com.cloud.hisavana.sdk.common.AdLogUtil;
import com.cloud.hisavana.sdk.common.constant.Constants;
import com.cloud.hisavana.sdk.data.bean.response.AdsDTO;
import com.cloud.hisavana.sdk.offline.OfflineDownload;
import com.transsion.core.CoreUtil;

import java.io.File;
import java.text.SimpleDateFormat;

/**
 * 1. 判断某条离线广告当天是否处于投放中；
 * 2. 判断某条路线广告是否处结束投放；
 */
public class OfflineAdExpiredUtil {

    public static final String TAG = "OfflineAdExpiredUtil";

    /**
     * 判断某条离线广告当天是否处于投放中；
     * 子线程调用，主线程可能会anr
     * @param adsDTO
     * @return true 投放中
     */
    public static boolean idOfflineAdExpired(AdsDTO adsDTO) {
        //广告过期，广告素材未下载好
        if (idOfflineAdEnd(adsDTO) || !isOfflineAdDownloaded(adsDTO)) {
            return false;
        }
        try {
            //离线广告开始投放日期
            int day = (int) ((System.currentTimeMillis() - adsDTO.getOfflineAdStartDateLong()) / DAY_TIME_MS);
            //今天投放中，并且未到单日展示次数上线
            return adsDTO.getOfflineAdLaunchDateStr().charAt(day) == '1' && !adsDTO.isReachedDisplayTimes();
        } catch (Exception e) {
            AdLogUtil.Log().e(TAG, e.getMessage());
        }
        return false;
    }

    /**
     * 广告素材下载成功
     * 广告落地页下载成功
     *
     * @param adsDTO
     * @return true 成功
     */
    public static boolean isOfflineAdDownloaded(AdsDTO adsDTO) {
        //判断广告素材下载成功
        if (adsDTO == null) return false;
        String imgUrl = adsDTO.getAdImgUrl();
        //banner部分模板判断logo
        if (adsDTO.getImpBeanRequest() != null
                && adsDTO.getImpBeanRequest().adt == AdType.BANNER
                && (TextUtils.equals(adsDTO.getMaterialStyle(), Constants.BANNER_MATERIAL_STYLE_B20301)
                || TextUtils.equals(adsDTO.getMaterialStyle(), Constants.BANNER_MATERIAL_STYLE_B20302)
                || TextUtils.equals(adsDTO.getMaterialStyle(), Constants.BANNER_MATERIAL_STYLE_B20303))) {
            imgUrl = adsDTO.getLogoUrl();
        }
        if (!PlatformUtil.isFileExist(imgUrl,adsDTO.isOfflineAd())) {
            AdLogUtil.Log().d(TAG, "adsDTO MainImages download fail");
            return false;
        }
        String h5Url = adsDTO.getOfflineH5Url();
        if(TextUtils.isEmpty(h5Url)){
            return true;
        }else{
            //zip包，判断是否解压成功
            boolean status;
            if (adsDTO.isH5Zip()) {
                String outPutPath = FileUtil.getAppFilePath(CoreUtil.getContext())
                        + File.separator + OfflineDownload.H5_ZIP_FOLDER_NAME
                        + File.separator + MD5Utils.toMd5(h5Url);
                File file = new File(outPutPath);
                status = file.exists();
                AdLogUtil.Log().d(TAG, "adsDTO offline zip download status: " + status);
            } else {
                //静态页面，判断静态页面下载成功
                status = !TextUtils.isEmpty(h5Url) && PlatformUtil.isFileExist(h5Url,adsDTO.isOfflineAd());
                AdLogUtil.Log().d(TAG, "adsDTO offline res download status: " + status);
            }
            return status;
        }
    }

    /**
     * 判断某条路线广告是否处结束投放；
     * @param adsDTO
     * @return true结束投放, false未结束投放
     */
    public static boolean idOfflineAdEnd(AdsDTO adsDTO) {
        if (adsDTO == null || adsDTO.getOfflineAdStartDate() == null || adsDTO.getOfflineAdLaunchDate() == null) {
            return true;
        }
        try {
            AdLogUtil.Log().d(TAG, "idOfflineAdEnd  " + adsDTO.getId());
            //离线广告开始投放日期
            SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
            long offlineAdStartDate = format.parse(adsDTO.getOfflineAdStartDate()).getTime();
            adsDTO.setOfflineAdStartDateLong(offlineAdStartDate);
            //离线广告状态
            String str = Long.toBinaryString(adsDTO.getOfflineAdLaunchDate());
            adsDTO.setOfflineAdLaunchDateStr(str);
            //取出最后一天投放的位置
            int lastIndexValid = str.lastIndexOf("1") + 1;
            //埋点需要，广告过期时间
            adsDTO.setOfflineAdExpireTime(offlineAdStartDate + DAY_TIME_MS * lastIndexValid);
            if (System.currentTimeMillis() <= adsDTO.getOfflineAdExpireTime()) {
                AdLogUtil.Log().d(TAG, "adsDTO idOfflineAdEnd false");
                return false;
            }
        } catch (Exception e) {
            AdLogUtil.Log().e(TAG, e.getMessage());
        }
        return true;
    }
}
