package com.cloud.hisavana.sdk.manager;

import android.text.TextUtils;
import android.util.Log;

import com.cloud.hisavana.sdk.ad.base.AdType;
import com.cloud.hisavana.sdk.ad.base.PostBody;
import com.cloud.hisavana.sdk.api.config.AdManager;
import com.cloud.hisavana.sdk.common.AdLogUtil;
import com.cloud.hisavana.sdk.common.athena.AthenaTracker;
import com.cloud.hisavana.sdk.common.constant.Constants;
import com.cloud.hisavana.sdk.common.constant.TaErrorCode;
import com.cloud.hisavana.sdk.common.http.AdServerRequest;
import com.cloud.hisavana.sdk.common.http.ServicesTimeUtil;
import com.cloud.hisavana.sdk.common.http.listener.CommonResponseListener;
import com.cloud.hisavana.sdk.common.util.FileUtil;
import com.cloud.hisavana.sdk.common.util.MD5Utils;
import com.cloud.hisavana.sdk.common.util.OfflineAdExpiredUtil;
import com.cloud.hisavana.sdk.common.util.TrackingUtil;
import com.cloud.hisavana.sdk.config.AdxServerConfig;
import com.cloud.hisavana.sdk.data.bean.request.AdxImpBean;
import com.cloud.hisavana.sdk.data.bean.response.AdResponseBody;
import com.cloud.hisavana.sdk.data.bean.response.AdsDTO;
import com.cloud.hisavana.sdk.data.bean.response.ConfigCodeSeatDTO;
import com.cloud.hisavana.sdk.offline.OfflineDownload;
import com.cloud.sdk.commonutil.util.DeviceUtil;
import com.transsion.core.CoreUtil;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

class OfflineAdManager {
    private static final String TAG = "OfflineAdManager";
    private static final long REQUEST_AD_INTERVAL = 60;
    private static volatile OfflineAdManager sInstance;
    private boolean isDownloading = false;
    private List<ConfigCodeSeatDTO> mRequestConfigList;
    private ConfigCodeSeatDTO mCurrentRequestConfig;
    private int mTriggerType;
    private static final int DOWNLOAD_MATERIAL_COUNT = 2;
    private final OfflineDownload.DownLoadCallBack mLoadCallBack = new OfflineDownload.DownLoadCallBack() {
        @Override
        public void onComplete() {
            AdLogUtil.Log().d(TAG, "onComplete");
            if (mRequestConfigList != null) {
                mRequestConfigList.remove(mCurrentRequestConfig);
                AdLogUtil.Log().d(TAG, "onComplete mConfigList.size() " + mRequestConfigList.size());
                if (!mRequestConfigList.isEmpty()) {
                    checkNextConfig();
                } else {
                    mRequestConfigList = null;
                    mCurrentRequestConfig = null;
                    clearZipFile();
                }
            }
        }

        @Override
        public void onCompleteAndTrack(AdxImpBean adxImpBean, List<AdsDTO> mAdBeans, TaErrorCode adError) {
            //埋点
            if (mAdBeans != null && mAdBeans.size() > 0) {
                AthenaTracker.trackSspReturn(mAdBeans, adError, adxImpBean);
            } else {
                AthenaTracker.trackSspReturn(null, adError, adxImpBean);
            }
            onComplete();
        }
    };

    private OfflineAdManager() {
    }

    public static OfflineAdManager getInstance() {
        if (sInstance == null) {
            synchronized (OfflineAdManager.class) {
                if (sInstance == null) {
                    sInstance = new OfflineAdManager();
                }
            }
        }
        return sInstance;
    }

    public void startDownloadAd(Collection<ConfigCodeSeatDTO> configList, int triggerType) {
        AdLogUtil.Log().d(TAG, "startDownloadAd");
        if (configList == null || configList.isEmpty() || isDownloading) {
            AdLogUtil.Log().d(TAG, "startDownloadAd fail,configList is empty or it is downloading");
            return;
        }
        mRequestConfigList = new ArrayList<>(configList);
        isDownloading = true;
        mTriggerType = triggerType;
        checkNextConfig();
    }

    /**
     * 判断广告下载还是素材下载
     *
     * @param codeSeatDTO 广告位云控数据
     */
    private void judgeRequestAd(ConfigCodeSeatDTO codeSeatDTO) {
        AdLogUtil.Log().d(TAG, "judgeRequestAd ");
        if (codeSeatDTO == null) {
            mLoadCallBack.onComplete();
            return;
        }
        AdLogUtil.Log().d(TAG, "judgeRequestAd code seat id " + codeSeatDTO.getCodeSeatId());
        mCurrentRequestConfig = codeSeatDTO;
        List<AdsDTO> adsDTOS = OfflineProviderManager.getInstance().getAdListByCodeSeatIdAtSubThread(codeSeatDTO.getCodeSeatId());
        long current = System.currentTimeMillis();
        //广告接口请求不到广告数据时，广告请求间隔为较长频控间隔（云控可配）。
        long interval = codeSeatDTO.isAdRequestFail() ? codeSeatDTO.getOfflineAdRequestTimeInterval() : codeSeatDTO.getOfflineAdRequestTimeIntervalNoAd();
        AdLogUtil.Log().d(TAG, "requestAds interval " + interval);
        interval = interval > 0 ? interval : REQUEST_AD_INTERVAL;
        if ((adsDTOS == null || adsDTOS.size() < codeSeatDTO.getOfflineAdCacheCount())
                && (current - codeSeatDTO.getLastRequestAdTime() > interval * 1000)) {
            String requestId = DeviceUtil.getUUID();
            requestAds(mCurrentRequestConfig, mTriggerType, requestId);
        } else {
            //广告数量已经达到或者超过云控配置最大缓存数 直接下载素材
            downloadMaterials(adsDTOS, codeSeatDTO);
        }
    }

    /**
     * 下载广告素材
     *
     * @param adsList 广告列表
     */
    private void downloadMaterials(List<AdsDTO> adsList, ConfigCodeSeatDTO codeSeatDTO) {
        if (adsList == null || adsList.isEmpty() || codeSeatDTO == null) {
            mLoadCallBack.onComplete();
            return;
        }
        List<AdsDTO> realDownloadList = new ArrayList<>();
        int downloadMaterialSize = codeSeatDTO.getOfflineAdRequestCount();
        downloadMaterialSize = downloadMaterialSize > 0 ? downloadMaterialSize : DOWNLOAD_MATERIAL_COUNT;
        for (AdsDTO ad : adsList) {
            if (!OfflineAdExpiredUtil.isOfflineAdDownloaded(ad)) {
                realDownloadList.add(ad);
                if (realDownloadList.size() >= downloadMaterialSize) {
                    break;
                }
            }
        }
        OfflineDownload.getInstance().downLoad(realDownloadList, mLoadCallBack);
    }

    /**
     * 网络请求广告
     *
     * @param config      云控
     * @param triggerType triggerType
     * @param requestId   requestId
     */
    private void requestAds(final ConfigCodeSeatDTO config, int triggerType, String requestId) {
        if (config == null) {
            mLoadCallBack.onComplete();
            return;
        }
        AdLogUtil.Log().d(TAG, "requestAds config id " + config.getCodeSeatId());
        final AdxImpBean impBean = new AdxImpBean();
        impBean.adt = config.getCodeSeatType();
        impBean.pmid = config.getCodeSeatId();
        impBean.mAdCount = config.getOfflineAdCacheCount();//使用最大缓存数量作为请求广告数量
        impBean.requestId = requestId;
        switch (triggerType) {
            case Constants.TriggerType.TRIGGER_COLD:
                impBean.requestType = Constants.RequestType.REQUEST_COLD;
                break;
            case Constants.TriggerType.TRIGGER_NET_CHANGE:
                impBean.requestType = Constants.RequestType.REQUEST_NET_CHANGE;
                break;
            case Constants.TriggerType.TRIGGER_AD_RETURN:
                impBean.requestType = Constants.RequestType.REQUEST_AD_RETURN;
                break;
        }
        impBean.offlineAd = true;
        impBean.triggerId = TrackingUtil.getTriggerId();
        AdServerRequest adServerRequest = new AdServerRequest().setListener(new CommonResponseListener<AdResponseBody>() {
            @Override
            protected void onRequestSuccess(int statusCode, AdResponseBody response) {
                // 保存服务器时间
                ServicesTimeUtil.saveServicesTime(response);
                if (response != null && response.getCode() == Constants.ResponseCode.BUSINESS_SUCCESS_CODE) {
                    setAdRequestResult(config, false);
                    if (response.getData() != null
                            && response.getData().getAds() != null
                            && response.getData().getAds().size() > 0 && response.getData().getOfflineAd()) {
                        final List<AdsDTO> adBeans = response.getData().getAds();
                        //服务端成功返回，广告不足，取长时间间隔
                        //服务端成功返回，广告充足，取短时间间隔
                        List<String> scales = response.getData().getScales();
                        if (!TextUtils.equals(response.getData().getCodeSeatId(), config.getCodeSeatId())) {
                            AdLogUtil.Log().d(TAG, "response pmid is diffrent with request's");
                            mLoadCallBack.onCompleteAndTrack(impBean, adBeans, TaErrorCode.RESPONSE_PMID_DIFFRENT_ERROR);
                            return;
                        }
                        if (!config.getCodeSeatType().equals(response.getData().getCodeSeatType())) {
                            AdLogUtil.Log().d(TAG, "response adt is diffrent with request's");
                            mLoadCallBack.onCompleteAndTrack(impBean, adBeans, TaErrorCode.RESPONSE_ADT_DIFFRENT_ERROR);
                            return;
                        }
                        Iterator<AdsDTO> it = adBeans.iterator();
                        final String adSeatType = response.getData().getAdSeatType();
                        while (it.hasNext()) {
                            final AdsDTO adBean = it.next();
                            if (adBean == null) {
                                AdLogUtil.Log().d(TAG, "ad is null or ad is not offline,remove from list");
                                it.remove();
                                continue;
                            }
                            if (adBean.getCodeSeatType() == AdType.NATIVE
                                    && (adBean.getNativeObject() == null
                                    || adBean.getNativeObject().getMainImages() == null
                                    || adBean.getNativeObject().getMainImages().size() == 0
                                    || TextUtils.isEmpty(adBean.getNativeObject().getMainImages().get(0).getUrl()))) {
                                AdLogUtil.Log().d(TAG, "Native ad's image is empty,remove from list");
                                it.remove();
                                continue;
                            }
                            adBean.setOfflineAd(true);
                            adBean.setFilePath(MD5Utils.toMd5(adBean.getOfflineH5Url()));
                            adBean.setFill_ts(System.currentTimeMillis());
                            adBean.setCacheTime(response.getData().getCacheTime());
                            adBean.setAbTest(response.getData().getAbTest());
                            adBean.setExtInfo(response.getData().getExtInfo());
                            adBean.setImpBeanRequest(impBean);
                            adBean.setAdSeatType(adSeatType);
                            if (!TextUtils.isEmpty(adBean.getAdm()) && scales != null) {
                                adBean.setScales(scales);
                            }
                        }
                        if (adBeans.size() > 0) {
                            AthenaTracker.trackSspReturn(adBeans, null, impBean);
                            //广告数据插入数据库
                            AdLogUtil.Log().d(TAG, "download ad before" + adBeans.size());
                            OfflineProviderManager.getInstance().insertAdsListAtSubThread(adBeans);
                            //下载广告素材
                            List<AdsDTO> list = OfflineProviderManager.getInstance().getAdListByCodeSeatIdAtSubThread(config.getCodeSeatId());
                            downloadMaterials(list, config);
                        } else {
                            mLoadCallBack.onCompleteAndTrack(impBean, null, TaErrorCode.SELF_AD_BE_FILTER);
                        }
                    } else {
                        AdLogUtil.Log().d(TAG, "ads list is empty");
                        mLoadCallBack.onCompleteAndTrack(impBean, null, TaErrorCode.RESPONSE_AD_IS_EMPTY);
                    }
                } else {
                    //服务端异常返回，取短时间间隔
                    setAdRequestResult(config, true);
                    AdLogUtil.Log().d(TAG, "业务错误 --> " + (response != null ?
                            ("error,response code is :" + response.getCode() + "," + "response msg is " + response.getMessage() + " ******")
                            : " error, response is null"));
                    final TaErrorCode adError;
                    //业务错误
                    if (null != response) {
                        AdLogUtil.Log().d(TAG, "loadAd() - error,response code is :" + response.getCode() + "," +
                                "response msg is " + response.getMessage());
                        adError = new TaErrorCode(response.getCode(), response.getMessage());
                    } else {
                        AdLogUtil.Log().d(TAG, "loadAd() - load ad error,null == response");
                        adError = new TaErrorCode(TaErrorCode.UNKNOWN_ERROR_CODE_1, "response is null");
                    }
                    mLoadCallBack.onCompleteAndTrack(impBean, null, adError);
                }
            }

            @Override
            public void onRequestError(final TaErrorCode adError) {
                //服务端异常返回，取短时间间隔
                setAdRequestResult(config, true);
                if (null != adError) {
                    AdLogUtil.Log().d(TAG, "load ad error adError=" + adError.getErrorMessage());
                }
                mLoadCallBack.onCompleteAndTrack(impBean, null, adError);
            }

        }).setPostBody(new AdServerRequest.IAdPostBody() {
            @Override
            public String getPostBody() {
                return PostBody.getAdPostBody(impBean);
            }
        })
                .setDebug(AdManager.isDebug())
                .setUrl(AdxServerConfig.getServerUrl() + AdxServerConfig.getServerApi())
                .setPlacementId(impBean.pmid)
                .setIsOfflineAd(true)
                .setAdxImpBean(impBean);
        if (adServerRequest != null) {
            adServerRequest.netRequestPreExecute();
        }
    }

    /**
     * 设置广告接口请求状态 使用长间隔还是短间隔
     *
     * @param config          云控
     * @param isAdRequestFail ture:请求间隔为短时间  false：请求间隔为长时间
     */
    private void setAdRequestResult(final ConfigCodeSeatDTO config, final boolean isAdRequestFail) {
        //更新云控 "上次请求广告时间"
        config.setLastRequestAdTime(System.currentTimeMillis());
        config.setAdRequestFail(isAdRequestFail);
        ConfigProviderManager.getInstance().updateConfigData(config);
    }

    /**
     * 判断处理下一个云控数据
     */
    private void checkNextConfig() {
        AdLogUtil.Log().d(TAG, "checkNextConfig ");
        if (mRequestConfigList != null && !mRequestConfigList.isEmpty()) {
            Iterator<ConfigCodeSeatDTO> iterator = mRequestConfigList.iterator();
            while (iterator.hasNext()) {
                ConfigCodeSeatDTO config = iterator.next();
                if (config.isOfflineAdEnable()) {
                    judgeRequestAd(config);
                    break;
                } else {
                    iterator.remove();
                }
            }
            if (mRequestConfigList != null && mRequestConfigList.isEmpty()) {
                mLoadCallBack.onComplete();
            }
        }
    }


    /**
     * 清理解压文件
     */
    private void clearZipFile() {
        AdLogUtil.Log().d(TAG, "clearZipFile，ad download logic is done.");
        String outPutPath = FileUtil.getAppFilePath(CoreUtil.getContext())
                + File.separator + OfflineDownload.H5_ZIP_FOLDER_NAME;
        File parent = new File(outPutPath);
        try {
            if (parent.exists() && parent.isDirectory()) {
                String[] files = parent.list();
                if (files != null && files.length > 0) {
                    AdLogUtil.Log().d(TAG, "clearZipFile files " + files.length);
                    for (String path : files) {
                        boolean fileUseful = OfflineProviderManager.getInstance().isFileUsefulJudgeAtSubThread(path);
                        if (!fileUseful) {
                            File file = new File(outPutPath + File.separator + path);
                            if (file.exists()) {
                                if (file.isDirectory()) {
                                    FileUtil.deleteFile(file);
                                } else {
                                    file.delete();
                                }
                            }
                        }
                    }
                }
            }
        } catch (Exception exception) {
            AdLogUtil.Log().e(TAG, Log.getStackTraceString(exception));
        }
        isDownloading = false;
    }
}
