@file:JvmName("ExecutorUtils")

package com.cloud.baobabsland.config.utils;

import android.os.Handler
import android.os.Looper
import java.util.concurrent.*
import java.util.concurrent.atomic.AtomicInteger
import kotlin.concurrent.thread

private val CPU_CORE = Runtime.getRuntime().availableProcessors()

private var ioA: AtomicInteger = AtomicInteger()
private val ioE: ExecutorService by lazy {
    ThreadPoolExecutor(
        1, CPU_CORE * 2,
        60L, TimeUnit.SECONDS,
        SynchronousQueue<Runnable>(),
        ThreadFactory {
            thread(
                name = "IO ${ioA.incrementAndGet()}",
                start = false,
                priority = Thread.NORM_PRIORITY,
                block = { it.run() })
        },
        ThreadPoolExecutor.DiscardPolicy()
    )
}

/**
 * 获取一个线程池，该线程池中的任务总是会启用新线程，并在新线程中执行。
 *
 * 该线程池适用于 I/O （读写文件、读写数据库、网络信息交互等）等任务的执行。
 *
 * 这个线程池核心线程数为 1 ，最大线程数上限为 CPU 核心数的 2 倍。
 */
fun io(): ExecutorService = ioE

private var computationA: AtomicInteger = AtomicInteger()
private val computationE: ExecutorService by lazy {
    ThreadPoolExecutor(
        CPU_CORE,
        CPU_CORE,
        0, TimeUnit.NANOSECONDS,
        LinkedBlockingQueue<Runnable>(),
        ThreadFactory {
            thread(
                name = "COMPUTATION ${computationA.incrementAndGet()}",
                start = false,
                priority = Thread.NORM_PRIORITY,
                block = { it.run() })
        },
        ThreadPoolExecutor.DiscardPolicy()
    )
}

/**
 * 计算所使用的线程池。
 *
 * 当需要 CPU 密集型计算，例如图形的计算。不要把 I/O 等具有较长时间阻塞的任务放在该线程池中执行。
 *
 * 这个线程池使用的固定的线程池，大小为 CPU 核数。
 */
fun computation(): ExecutorService = computationE

private var newA: AtomicInteger = AtomicInteger()
private val newE: ExecutorService by lazy {
    ThreadPoolExecutor(
        1, CPU_CORE,
        0L, TimeUnit.NANOSECONDS,
        SynchronousQueue<Runnable>(),
        ThreadFactory {
            thread(
                name = "NEW ${newA.incrementAndGet()}",
                start = false,
                priority = Thread.NORM_PRIORITY,
                block = { it.run() })
        }, ThreadPoolExecutor.DiscardPolicy()
    )
}

/**
 * 获取一个线程池，该线程池中的任务总是会启用新线程，并在新线程中执行。
 *
 * 这个线程池核心线程数为 1 ，，最大线程数上限为 CPU 核心数的 2 倍。
 */
fun new(): ExecutorService = newE

private val singleE: ExecutorService by lazy {
    ThreadPoolExecutor(
        1, 1,
        0L, TimeUnit.NANOSECONDS,
        LinkedBlockingQueue(),
        ThreadFactory {
            thread(
                name = "SINGLE",
                start = false,
                priority = Thread.MIN_PRIORITY,
                block = { it.run() })
        },
        ThreadPoolExecutor.DiscardPolicy()
    )
}

/**
 * 获取一个线程池，该线程池中的任务总是会同一个线程中执行。
 *
 * 这个线程池核心线程数为 1 ，最大线程数为 1 。
 */
fun single(): ExecutorService = singleE

private var scheduledA: AtomicInteger = AtomicInteger()
private val scheduledE: ScheduledExecutorService by lazy {
    ScheduledThreadPoolExecutor(
        1, ThreadFactory {
            thread(
                name = "SCHEDULED ${scheduledA.incrementAndGet()}",
                start = false,
                priority = Thread.MIN_PRIORITY,
                block = { it.run() })
        },
        ThreadPoolExecutor.DiscardPolicy()
    )
}


/**
 * 获取一个 [ScheduledExecutorService] ，用于执行重复任务和延迟任务等。
 *
 * 这个线程池核心线程数为 1 ，没有最大线程数上限。
 *
 * 由于没有最大线程上限，因此若遇到短时间内的大量任务的执行，可能会出现 OOM 的问题。
 */
fun scheduled(): ScheduledExecutorService = scheduledE

private val mainE: ExecutorService by lazy {
    object : AbstractExecutorService() {
        val handler: Handler = Handler(Looper.getMainLooper())

        @Throws(UnsupportedOperationException::class)
        override fun shutdown() = throw UnsupportedOperationException()

        @Throws(UnsupportedOperationException::class)
        override fun shutdownNow(): MutableList<Runnable> = throw UnsupportedOperationException()

        override fun isShutdown(): Boolean = false

        override fun awaitTermination(timeout: Long, unit: TimeUnit): Boolean = false

        override fun isTerminated(): Boolean = false

        override fun execute(command: Runnable) {
            handler.post(command)
        }
    }
}


/**
 * 获取一个线程池，该线程池中的任务总是会在主线程中执行。
 */
fun main(): ExecutorService = mainE