package com.cloud.baobabsland.download;

import android.content.Intent;
import android.text.TextUtils;
import android.util.Log;

import androidx.localbroadcastmanager.content.LocalBroadcastManager;

import com.cloud.baobabsland.bean.DownImageInfo;
import com.cloud.baobabsland.bean.response.ConfigDTO;
import com.cloud.baobabsland.constant.ErrorCode;
import com.cloud.baobabsland.constant.ServerConfig;
import com.cloud.baobabsland.worker.UnionWorker;
import com.cloud.baobabsland.athena.AthenaTracker;
import com.cloud.baobabsland.constant.Constants;
import com.cloud.baobabsland.http.DownLoadRequest;
import com.cloud.baobabsland.http.listener.DrawableResponseListener;
import com.cloud.baobabsland.utils.AdLogUtil;
import com.cloud.baobabsland.utils.CollectionUtils;
import com.cloud.baobabsland.utils.CommonLogUtil;
import com.cloud.baobabsland.utils.FileUtil;
import com.transsion.core.CoreUtil;
import com.transsion.core.pool.TranssionPoolManager;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * 图片下载工具类
 */
public class ImageDownloadHelper {

    public static void downLoadImages(final List<DownImageInfo> infoList) {
        List<DownImageInfo> imageUrls = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(infoList)) {
            imageUrls.addAll(infoList);
        } else {
            return;
        }

        NativeDownload head = null;
        NativeDownload temp = null;
        for (DownImageInfo info : imageUrls) {
            if (info != null) {
                if (head == null) {
                    head = new NativeDownload(info.imageUrl, info.imageType, info.feedCode);
                    temp = head;
                } else {
                    temp.next = new NativeDownload(info.imageUrl, info.imageType, info.feedCode);
                    temp = temp.next;
                }
            }
        }
        if (head != null) {
            head.startDownload();
        }
    }

    /**
     * 指定路径生成image图片
     *
     * @param downImagePath 下载到本地的image地址
     * @param urlPath       下载的图片原始地址
     */
    private static void createImageByPath(final String downImagePath, final String urlPath, final int imageType) {
        String imageFilePath;
        if (imageType == Constants.ImageType.FEED_DETAIL_IMAGE) {
            imageFilePath = Constants.DownPath.FEED_DETAIL_IMAGE_NAME;
        } else {
            imageFilePath = Constants.DownPath.FEED_LIST_IMAGE_NAME;
        }
        String outPutPath = FileUtil.getAppFilePath(CoreUtil.getContext())
                + File.separator + Constants.DownPath.OFFLINE_FOLDER_NAME
                + File.separator + imageFilePath;

        File file = new File(outPutPath);
        try {
            AdLogUtil.Log().d(CommonLogUtil.TAG, "image生成路径 == " + outPutPath);
            String filePath = file.getPath() + File.separator + FileUtil.getFileName(urlPath);
            File imagePath = new File(filePath);
            if (!FileUtil.createOrExistsFile(imagePath)) {
                AdLogUtil.Log().e("createImageByPath", "create file <" + imagePath + "> failed.");
            }
            TranssionPoolManager.getInstance().addTask(() -> {
                boolean copyImageResult = FileUtil.writeFileFromIS(imagePath, downImagePath);
                AdLogUtil.Log().i(CommonLogUtil.TAG, "-----------------copyImageResult = " + copyImageResult);
                if (copyImageResult) {
                    try {
                        //copy成功后，delete原文件
                        if(!new File(downImagePath).delete()) {
                            AdLogUtil.Log().e("downImagePath file delete fail!!");
                        }
                    } catch (Exception e) {
                        AdLogUtil.Log().e("downImagePath file delete fail!!  e -> " + Log.getStackTraceString(e));
                    }
                }
            });

        } catch (Exception e) {
            AdLogUtil.Log().e(Log.getStackTraceString(e));
        }
    }

    private static class NativeDownload {
        private NativeDownload next;
        private final String imageUrl;
        private AtomicInteger imageCounter;
        private int imageType;
        private String feedCode;

        public NativeDownload(String imageUrl, int imageType, String feedCode) {
            this.imageUrl = imageUrl;
            this.imageType = imageType;
            this.feedCode = feedCode;
        }

        private void startDownload() {
            if (TextUtils.isEmpty(imageUrl)) {
                downloadNext();
                return;
            }

            imageCounter = new AtomicInteger(1);
            preCacheNativeImage(imageUrl, imageType, feedCode);
        }

        public void calculateDownloadCount() {
            if (imageCounter.decrementAndGet() <= 0){
                downloadNext();
            }
        }

        private void downloadNext() {
            if (next != null) {
                next.startDownload();
            } else {
                if (imageType == Constants.ImageType.FEED_LIST_IMAGE) {
                    Intent intent = new Intent(Constants.CommonConstant.RECEIVER_ACTION);
                    LocalBroadcastManager.getInstance(CoreUtil.getContext()).sendBroadcast(intent);
                }
            }
        }


        /**
         * 图片预加载
         *
         * @param urlPath  图片url信息
         */
        private void preCacheNativeImage(String urlPath, int imageType, String feedCode) {
            if (urlPath.isEmpty()) {
                AdLogUtil.Log().d(CommonLogUtil.TAG, "image url is null");
                return;
            }
            if (!urlPath.contains("https") && !urlPath.contains("http")) {
                if (UnionWorker.getInstance(CoreUtil.getContext(), null).unionCache != null
                        && UnionWorker.getInstance(CoreUtil.getContext(), null).unionCache.getConfigDTO() != null) {
                    ConfigDTO configDTO = UnionWorker.getInstance(CoreUtil.getContext(), null).unionCache.getConfigDTO();
                    //oss的域名设置为云控返回的域名
                    if (configDTO.basicJson != null && !TextUtils.isEmpty(configDTO.basicJson.ossDomain)) {
                        urlPath = configDTO.basicJson.ossDomain + urlPath;
                    } else {
                        urlPath = ServerConfig.getOssUrl() + urlPath;
                    }
                }
            }

            final String downUrlPath = urlPath;
            AdLogUtil.Log().d(CommonLogUtil.TAG, "开始缓存图片，request downUrlPath: " + urlPath);
            DownLoadRequest downLoadRequest = new DownLoadRequest()
                    .setListener(new DrawableResponseListener() {
                        @Override
                        public void onRequestSuccess(int statusCode, byte[] data, String filePath) {
                            calculateDownloadCount();
                            AthenaTracker.trackDataByDownloadPic(feedCode, downUrlPath, 1);

                            AdLogUtil.Log().d(CommonLogUtil.TAG, "图片缓存成功，image cache success!! filePath = " + filePath + " , isExists = " + new File(filePath).exists());
                            createImageByPath(filePath, downUrlPath, imageType);
                        }

                        @Override
                        protected void onRequestError(ErrorCode adError) {
                            calculateDownloadCount();
                            AthenaTracker.trackDataByDownloadPic(feedCode, downUrlPath, 0);
                        }
                    })
                    .setUrl(downUrlPath);
            downLoadRequest.netRequestPreExecute(null, null);
        }
    }
}
