package com.cloud.baobabsland.download;

import android.util.Log;

import com.cloud.baobabsland.constant.ErrorCode;
import com.cloud.baobabsland.worker.UnionWorker;
import com.cloud.baobabsland.athena.AthenaTracker;
import com.cloud.baobabsland.constant.Constants;
import com.cloud.baobabsland.http.DownLoadRequest;
import com.cloud.baobabsland.http.listener.DrawableResponseListener;
import com.cloud.baobabsland.utils.AdLogUtil;
import com.cloud.baobabsland.utils.CommonLogUtil;
import com.cloud.baobabsland.utils.FileUtil;
import com.cloud.baobabsland.utils.FileUtils;
import com.cloud.baobabsland.utils.ZipUtil;
import com.transsion.core.CoreUtil;

import java.io.File;

public class OfflineDownload {

    private DownLoadCallBack callBack;

    private OfflineDownload() {
    }

    private static class SingletonInner {
        private static final OfflineDownload instance = new OfflineDownload();
    }

    public static OfflineDownload getInstance() {
        return SingletonInner.instance;
    }

    /**
     * 下载广告列表中的素材和离线包
     * 子线程调用
     *
     * @param callBack
     */
    public void downLoad(final String zipUrl, DownLoadCallBack callBack) {
        this.callBack = callBack;
        AdLogUtil.Log().d(CommonLogUtil.TAG, "准备下载 ====== zipUrl = " + zipUrl);
        downLoadMaterial(zipUrl, () -> {
            AdLogUtil.Log().d(CommonLogUtil.TAG, "Zip下载完成");
            onComplete();
        });
    }


    /**
     * 下载物料
     */
    private void downLoadMaterial(
            final String url,
            final LoadMaterialCallback loadMaterialCallback) {
        AthenaTracker.trackDataByDownLoadZip(url);

        DownLoadRequest downLoadRequest = new DownLoadRequest()
                .setUrl(url)
                .setFetch(true)
                .setUsePool(true)
                .setListener(new DrawableResponseListener() {
                    @Override
                    public void onRequestSuccess(int statusCode, byte[] data, String filePath) {
                        AdLogUtil.Log().d(CommonLogUtil.TAG, "下载成功 == onRequestSuccess -> filePath = " + filePath + " , url = " + url);
                        UnionWorker.UnionCache unionCache = UnionWorker.getInstance(CoreUtil.getContext(), null).unionCache;
                        if (unionCache != null
                                && unionCache.getConfigDTO() != null
                                && unionCache.getConfigDTO().basicJson != null) {
                            boolean isFileValid = FileUtils.Companion.isFileValid(
                                    unionCache.getConfigDTO().basicJson.h5FeedZipMd5,
                                    new File(filePath));
                            AdLogUtil.Log().i(CommonLogUtil.TAG, "check md5 result -> isFileValid = " + isFileValid);
                            AthenaTracker.trackDataByDownLoadZipRes(0, "success", isFileValid);
                            if (isFileValid) {
                                upZip(filePath);
                            } else {
                                if (!new File(filePath).delete()) {//md5不同，删除源文件
                                    AdLogUtil.Log().e("unzip() -> zipPath file delete fail!!");
                                }
                                AdLogUtil.Log().i(CommonLogUtil.TAG, "file md5 is not equal");
                            }
                        }
                    }

                    @Override
                    protected void onRequestError(ErrorCode error) {
                        if (error != null) {
                            AthenaTracker.trackDataByDownLoadZipRes(error.getErrorCode(), error.getErrorMessage(), false);
                            AdLogUtil.Log().e("下载失败 == code = " + error.getErrorCode() + " , message = " + error.getErrorMessage() + " , url = " + url);
                        }
                    }
                });

        downLoadRequest.netRequestPreExecute(null, null);
    }

    /**
     * 解压文件
     *
     * @param zipPath 下载的原始路径
     */
    private void upZip(final String zipPath) {
        String outPutPath = FileUtil.getAppFilePath(CoreUtil.getContext())
                + File.separator + Constants.DownPath.OFFLINE_FOLDER_NAME
                + File.separator + Constants.DownPath.H5_ZIP_FOLDER_NAME;
        AdLogUtil.Log().i(CommonLogUtil.TAG, " upZip() -> outPutPath = " + outPutPath);
        File file = new File(outPutPath);
        if (file.exists()) {
            if (FileUtil.deleteFile(file)) { //如果file已经存在先删除，在解压
                AdLogUtil.Log().d(CommonLogUtil.TAG, "解压路径 == " + outPutPath);
                decompressZipFolder(zipPath, outPutPath);
            } else {
                AdLogUtil.Log().e("unzip() -> outPutPath file delete fail!!");
            }
        } else {
            decompressZipFolder(zipPath, outPutPath);
        }
    }

    /**
     * 处理解压文件
     */
    private void decompressZipFolder(String zipPath, String outPutPath) {
        boolean unZipResult = ZipUtil.decompressZipFolder(zipPath, outPutPath);
        if (unZipResult) {
            AdLogUtil.Log().i(CommonLogUtil.TAG, "zipPath decompress success!!");
            try {
                //解压成功后，delete原文件
                if (!new File(zipPath).delete()) {
                    AdLogUtil.Log().e("unzip() -> zipPath file delete fail!!");
                }
            } catch (Exception e) {
                AdLogUtil.Log().e(Log.getStackTraceString(e));
            }
        }
    }

    /**
     * 加载完成的回调
     */
    private void onComplete() {
        if (callBack != null) {
            callBack.onComplete();
        }
    }

    interface LoadMaterialCallback {
        void loadFinish();
    }

    public interface DownLoadCallBack {
        /**
         * 当前请求全部完成
         */
        void onComplete();

        void onCompleteAndTrack();
    }
}
