package com.cloud.baobabsland.utils

import android.annotation.SuppressLint
import android.content.ContentResolver
import android.content.ContentValues
import android.content.Context
import android.content.res.AssetFileDescriptor
import android.net.Uri
import android.os.Build
import android.os.Environment
import android.provider.MediaStore
import android.util.Log
import androidx.annotation.RequiresApi
import androidx.core.content.FileProvider
import com.transsion.core.CoreUtil
import java.io.*

/**
 * 文件操作工具类
 */
class FileUtils private constructor() {
    companion object {
        //只读模式
        private const val MODE_READ_ONLY = "r"
        private val EXT_STORAGE_PATH = extStoragePath
        private val EXT_STORAGE_DIR = EXT_STORAGE_PATH + File.separator
        private val APP_EXT_STORAGE_PATH = EXT_STORAGE_DIR + "Android"
        private val EXT_DOWNLOADS_PATH = extDownloadsPath
        private val EXT_PICTURES_PATH = extPicturesPath
        private val EXT_DCIM_PATH = extDCIMPath

        /**
         * 根据文件路径获取文件
         *
         * @param filePath 文件路径
         * @return 文件
         */
        fun getFileByPath(filePath: String?): File? {
            if (filePath.isNullOrBlank()) return null
            return if (isSpace(filePath)) null else File(filePath)
        }

        /**
         * 判断文件是否存在
         *
         * @param file 文件
         * @return `true`: 存在<br></br>`false`: 不存在
         */
        fun isFileExists(file: File?): Boolean {
            if (file == null) {
                return false
            }
            return if (file.exists()) {
                true
            } else isFileExists(file.absolutePath)
        }

        /**
         * 判断文件是否存在
         *
         * @param filePath 文件路径
         * @return `true`: 存在<br></br>`false`: 不存在
         */
        private fun isFileExists(filePath: String?): Boolean {
            val file = getFileByPath(filePath) ?: return false
            return if (file.exists()) {
                true
            } else isFileExistsApi29(filePath)
        }

        /**
         * Android 10判断文件是否存在的方法
         *
         * @param filePath 文件路径
         * @return `true`: 存在<br></br>`false`: 不存在
         */
        private fun isFileExistsApi29(filePath: String?): Boolean {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
                var afd: AssetFileDescriptor? = null
                try {
                    val uri = Uri.parse(filePath)
                    afd = openAssetFileDescriptor(uri)
                    if (afd == null) {
                        return false
                    } else {
                        closeIOQuietly(afd)
                    }
                } catch (e: FileNotFoundException) {
                    return false
                } finally {
                    closeIOQuietly(afd)
                }
                return true
            }
            return false
        }

        /**
         * 获取文件输入流
         */
        fun getFileInputStream(file: File?): InputStream? {
            return try {
                FileInputStream(file)
            } catch (e: Exception) {
                AdLogUtil.Log().e(Log.getStackTraceString(e))
                null
            }
        }

        /**
         * 根据文件获取uri
         *
         * @param file 文件
         * @return
         */
        private fun getUriByFile(file: File?): Uri? {
            if (file == null) {
                return null
            }
            return if (isScopedStorageMode && isPublicPath(file)) {
                val filePath = file.absolutePath
                if (filePath.startsWith(EXT_DOWNLOADS_PATH)) {
                    getDownloadContentUri(CoreUtil.getContext(), file)
                } else if (filePath.startsWith(EXT_PICTURES_PATH) || filePath.startsWith(
                        EXT_DCIM_PATH
                    )
                ) {
                    getMediaContentUri(
                        CoreUtil.getContext(),
                        file
                    )
                } else {
                    getUriForFile(file)
                }
            } else {
                getUriForFile(file)
            }
        }

        /**
         * Return a content URI for a given file.
         *
         * @param file The file.
         * @return a content URI for a given file
         */
        private fun getUriForFile(file: File?): Uri? {
            if (file == null) {
                return null
            }
            return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                val authority = CoreUtil.getContext().packageName + ".updateFileProvider"
                FileProvider.getUriForFile(
                    CoreUtil.getContext(),
                    authority,
                    file
                )
            } else {
                Uri.fromFile(file)
            }
        }

        /**
         * 将媒体文件转化为资源定位符
         *
         * @param context
         * @param mediaFile 媒体文件
         * @return
         */
        @SuppressLint("Range")
        private fun getMediaContentUri(context: Context, mediaFile: File): Uri? {
            return try {
                val filePath = mediaFile.absolutePath
                val baseUri = MediaStore.Images.Media.EXTERNAL_CONTENT_URI
                val cursor = context.contentResolver.query(
                    baseUri,
                    arrayOf(MediaStore.Images.Media._ID),
                    MediaStore.Images.Media.DATA + "=? ",
                    arrayOf(filePath),
                    null
                )
                if (cursor != null && cursor.moveToFirst()) {
                    val id = cursor.getInt(cursor.getColumnIndex(MediaStore.MediaColumns._ID))
                    cursor.close()
                    Uri.withAppendedPath(baseUri, "" + id)
                } else {
                    if (mediaFile.exists()) {
                        val values = ContentValues()
                        values.put(MediaStore.Images.Media.DATA, filePath)
                        return context.contentResolver.insert(baseUri, values)
                    }
                    null
                }
            } catch (e: Exception) {
                e.printStackTrace()
                null
            }
        }

        /**
         * 是否是分区存储模式：在公共目录下file的api无效了
         *
         * @return 是否是分区存储模式
         */
        private val isScopedStorageMode: Boolean
            get() = Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q && !Environment.isExternalStorageLegacy()

        @SuppressLint("Range")
        @RequiresApi(api = Build.VERSION_CODES.Q)
        fun getDownloadContentUri(context: Context, file: File): Uri? {
            return try {
                val filePath = file.absolutePath
                val baseUri = MediaStore.Downloads.EXTERNAL_CONTENT_URI
                val cursor = context.contentResolver.query(
                    baseUri,
                    arrayOf(MediaStore.Downloads._ID),
                    MediaStore.Downloads.DATA + "=? ",
                    arrayOf(filePath),
                    null
                )
                if (cursor != null && cursor.moveToFirst()) {
                    val id = cursor.getInt(cursor.getColumnIndex(MediaStore.DownloadColumns._ID))
                    cursor.close()
                    Uri.withAppendedPath(baseUri, "" + id)
                } else {
                    if (file.exists()) {
                        val values = ContentValues()
                        values.put(MediaStore.Downloads.DATA, filePath)
                        return context.contentResolver.insert(baseUri, values)
                    }
                    null
                }
            } catch (e: Exception) {
                e.printStackTrace()
                null
            }
        }

        /**
         * 是否是公有目录
         *
         * @return 是否是公有目录
         */
        private fun isPublicPath(file: File?): Boolean {
            if (file == null) {
                return false
            }
            try {
                return isPublicPath(file.canonicalPath)
            } catch (e: IOException) {
                e.printStackTrace()
            }
            return false
        }

        /**
         * 是否是公有目录
         *
         * @return 是否是公有目录
         */
        private fun isPublicPath(filePath: String): Boolean {
            return if (isSpace(filePath)) {
                false
            } else filePath.startsWith(EXT_STORAGE_PATH) && !filePath.startsWith(
                APP_EXT_STORAGE_PATH
            )
        }

        private fun isSpace(s: String?): Boolean {
            if (s == null) {
                return true
            }
            var i = 0
            val len = s.length
            while (i < len) {
                if (!Character.isWhitespace(s[i])) {
                    return false
                }
                ++i
            }
            return true
        }

        /**
         * 安静关闭 IO
         *
         * @param closeables closeables
         */
        private fun closeIOQuietly(vararg closeables: Closeable?) {
            for (closeable in closeables) {
                if (closeable != null) {
                    try {
                        closeable.close()
                    } catch (ignored: IOException) {
                    }
                }
            }
        }

        /**
         * 从uri资源符中读取文件描述
         *
         * @param uri 文本资源符
         * @return AssetFileDescriptor
         */
        @Throws(FileNotFoundException::class)
        fun openAssetFileDescriptor(uri: Uri?): AssetFileDescriptor? {
            return contentResolver.openAssetFileDescriptor(
                uri!!, MODE_READ_ONLY
            )
        }

        private val contentResolver: ContentResolver
            get() = CoreUtil.getContext().contentResolver

        /**
         * 获取 Android 外置储存的根目录
         * <pre>path: /storage/emulated/0</pre>
         *
         * @return 外置储存根目录
         */
        private val extStoragePath: String
            get() = Environment.getExternalStorageDirectory().absolutePath

        /**
         * 获取下载目录
         * <pre>path: /storage/emulated/0/Download</pre>
         *
         * @return 下载目录
         */
        private val extDownloadsPath: String
            get() = Environment
                .getExternalStoragePublicDirectory(Environment.DIRECTORY_DOWNLOADS)
                .absolutePath

        /**
         * 获取图片目录
         * <pre>path: /storage/emulated/0/Pictures</pre>
         *
         * @return 图片目录
         */
        private val extPicturesPath: String
            get() = Environment
                .getExternalStoragePublicDirectory(Environment.DIRECTORY_PICTURES)
                .absolutePath

        /**
         * 获取相机拍摄的照片和视频的目录
         * <pre>path: /storage/emulated/0/DCIM</pre>
         *
         * @return 照片和视频目录
         */
        private val extDCIMPath: String
            get() = Environment
                .getExternalStoragePublicDirectory(Environment.DIRECTORY_DCIM)
                .absolutePath

        /**
         * 关闭流
         */
        @JvmStatic
        fun close(vararg closeables: Closeable?) {
            for (io in closeables) {
                if (io != null) {
                    try {
                        io.close()
                    } catch (e: IOException) {
                        e.printStackTrace()
                    }
                }
            }
        }

        /**
         * 下载文件是否有效-md5判断
         */
        fun isFileValid(originalMd5: String?, file: File?): Boolean {
            AdLogUtil.Log().i(CommonLogUtil.TAG, "down load zip  md5 = " + Md5Utils.getFileMD5(file) + " ， originalMd5 = " + originalMd5)
            return originalMd5?.isNotBlank() == true && originalMd5.equals(
                Md5Utils.getFileMD5(file),
                ignoreCase = true
            )
        }
    }
}