package com.cloud.baobabsland.worker

import android.content.Context
import android.text.TextUtils
import android.util.Log
import com.cloud.baobabsland.BuildConfig
import com.cloud.baobabsland.UnionSetting
import com.cloud.baobabsland.athena.AthenaTracker
import com.cloud.baobabsland.bean.DownImageInfo
import com.cloud.baobabsland.bean.response.*
import com.cloud.baobabsland.config.ConfigHelper.Companion.getConfigStoreInstance
import com.cloud.baobabsland.config.ConfigResultListener
import com.cloud.baobabsland.constant.Constants
import com.cloud.baobabsland.utils.*
import com.cloud.baobabsland.utils.gsonutil.GsonUtil
import com.google.gson.Gson
import com.google.gson.reflect.TypeToken
import com.transsion.core.CoreUtil
import java.io.File


object WorkerHelper {

    /**
     * 读取缓存的feedList数据
     */
    fun readOfflineFeedList(cacheFeedListFilePath: String?): List<FeedVO>? {
        if (TextUtils.isEmpty(cacheFeedListFilePath)) return emptyList()

        val offlineFeedList = CommonUtils.bufferRead(cacheFeedListFilePath?.let { File(it) })
        if (!TextUtils.isEmpty(offlineFeedList)) {
            try {
                val decryptResult = AESUtils.decrypt(offlineFeedList)
                AdLogUtil.Log().i(
                    CommonLogUtil.TAG,
                    "readOfflineFeedList() -> offlineFeedList data isn't null"
                )
                return GsonUtil.fromJson<List<FeedVO>>(
                    decryptResult,
                    object : TypeToken<List<FeedVO?>?>() {}.type
                )
            } catch (e: java.lang.Exception) {
                AdLogUtil.Log().e(Log.getStackTraceString(e))
            }
        } else {
            AdLogUtil.Log().i(CommonLogUtil.TAG, "readOfflineFeedList data is null")
        }
        return emptyList()
    }

    /**
     * 读取缓存的feedDetail数据
     */
    fun readOfflineFeedDetails(cacheFeedDetailFilePath: String?): MutableList<FeedDetailVO>? {
        if (TextUtils.isEmpty(cacheFeedDetailFilePath)) return mutableListOf()

        val offlineFeedDetails = CommonUtils.bufferRead(cacheFeedDetailFilePath?.let { File(it) })
        if (!TextUtils.isEmpty(offlineFeedDetails)) {
            try {
                val decryptResult = AESUtils.decrypt(offlineFeedDetails)
                AdLogUtil.Log().i(
                    CommonLogUtil.TAG,
                    "readOfflineFeedDetails() -> data isn't null"
                )
                return GsonUtil.fromJson<MutableList<FeedDetailVO>>(
                    decryptResult,
                    object : TypeToken<MutableList<FeedDetailVO?>?>() {}.type
                )
            } catch (e: java.lang.Exception) {
                AdLogUtil.Log().e(Log.getStackTraceString(e))
            }
        } else {
            AdLogUtil.Log().i(CommonLogUtil.TAG, "readOfflineFeedDetails data is null")
        }
        return mutableListOf()
    }

    /**
     * 读取缓存的实验数据
     */
    fun readExperimentList(cacheExperimentFilePath: String?): List<AbTestHitExpVO>? {
        if (TextUtils.isEmpty(cacheExperimentFilePath)) return emptyList()

        val experimentStr = CommonUtils.bufferRead(cacheExperimentFilePath?.let { File(it) })
        if (!TextUtils.isEmpty(experimentStr)) {
            try {
                val decryptResult = AESUtils.decrypt(experimentStr)
                AdLogUtil.Log()
                    .i(CommonLogUtil.TAG, "readExperimentList() -> data = $decryptResult")
                return GsonUtil.fromJson<List<AbTestHitExpVO>>(
                    decryptResult,
                    object : TypeToken<List<AbTestHitExpVO?>?>() {}.type
                )
            } catch (e: java.lang.Exception) {
                AdLogUtil.Log().e(Log.getStackTraceString(e))
            }
        } else {
            AdLogUtil.Log().i(CommonLogUtil.TAG, "readExperimentList data is null")
        }

        return emptyList()
    }


    /**
     * config数据缓存
     */
    fun writeConfig(cacheConfigFilePath: String?, configDTO: ConfigDTO?) {
        if (TextUtils.isEmpty(cacheConfigFilePath)) return

        if (configDTO != null) {
            try {
                val configDataStr = Gson().toJson(configDTO)
                AdLogUtil.Log().i(CommonLogUtil.TAG, "writeConfig() data")
                //写入本地文件
                CommonUtils.bufferSave(cacheConfigFilePath, AESUtils.encrypt(configDataStr))
            } catch (e: java.lang.Exception) {
                AdLogUtil.Log().e(Log.getStackTraceString(e))
            }
        } else {
            AdLogUtil.Log().i(CommonLogUtil.TAG, "writeConfig -> configDTO is null")
        }
    }

    /**
     * FeedList数据缓存
     */
    fun writeFeedList(cacheFeedListFilePath: String?, feedVOList: List<FeedVO?>?) {
        if (TextUtils.isEmpty(cacheFeedListFilePath)) return

        if (CollectionUtils.isNotEmpty(feedVOList)) {
            try {
                val feedListStr = Gson().toJson(feedVOList)
                AdLogUtil.Log().i(CommonLogUtil.TAG, "write feedListStr data")
                //写入本地文件
                CommonUtils.bufferSave(cacheFeedListFilePath, AESUtils.encrypt(feedListStr))
            } catch (e: Exception) {
                AdLogUtil.Log().e(Log.getStackTraceString(e))
            }
        } else {
            AdLogUtil.Log().i(CommonLogUtil.TAG, "writeFeedList -> feedVOList isEmpty")
        }
    }


    /**
     * FeedDetail数据缓存
     */
    fun writeFeedDetailList(
        cacheFeedDetailFilePath: String?,
        feedDetailVOList: List<FeedDetailVO?>?
    ) {
        if (TextUtils.isEmpty(cacheFeedDetailFilePath)) return

        if (CollectionUtils.isNotEmpty(feedDetailVOList)) {
            try {
                val feedDetailStr = Gson().toJson(feedDetailVOList)
                AdLogUtil.Log().i(
                    CommonLogUtil.TAG,
                    "feedDetailStr data is null = " + TextUtils.isEmpty(feedDetailStr)
                )
                //写入本地文件
                CommonUtils.bufferSave(cacheFeedDetailFilePath, AESUtils.encrypt(feedDetailStr))
            } catch (e: java.lang.Exception) {
                AdLogUtil.Log().e(Log.getStackTraceString(e))
            }
        } else {
            AdLogUtil.Log().i(CommonLogUtil.TAG, "writeFeedDetailList -> feedVOList isEmpty")
        }
    }

    /**
     * 实验数据缓存
     */
    fun writeExperimentList(
        cacheExperimentFilePath: String?,
        abTestHitExpList: List<AbTestHitExpVO?>?
    ) {
        if (TextUtils.isEmpty(cacheExperimentFilePath)) return

        if (CollectionUtils.isNotEmpty(abTestHitExpList)) {
            try {
                val experimentStr = Gson().toJson(abTestHitExpList)
                AdLogUtil.Log().i(
                    CommonLogUtil.TAG,
                    "experimentStr data is null = " + TextUtils.isEmpty(experimentStr)
                )
                //写入本地文件
                CommonUtils.bufferSave(cacheExperimentFilePath, AESUtils.encrypt(experimentStr))
            } catch (e: java.lang.Exception) {
                AdLogUtil.Log().e(Log.getStackTraceString(e))
            }
        } else {
            AdLogUtil.Log().i(CommonLogUtil.TAG, "writeExperimentList -> experimentStr isEmpty")
        }
    }

    /**
     * 删除缓存目录 -- 缓存开关=false
     */
    fun deleteOfflineData(): Boolean {
        try {
            AdLogUtil.Log().i(CommonLogUtil.TAG, "cloudConfig openCache is false")
            val deleteFilePath =
                FileUtil.getAppFilePath(CoreUtil.getContext()) + File.separator +
                        Constants.DownPath.OFFLINE_FOLDER_NAME + File.separator
            val deleteFile = File(deleteFilePath)
            return if (deleteFile.exists()) {
                if (!FileUtil.deleteFile(deleteFile)) {
                    AthenaTracker.deleteLocalData(false, 0)
                    AdLogUtil.Log().e("delete root file fail!!")
                    false
                } else {
                    AthenaTracker.deleteLocalData(false, 1)
                    true
                }
            } else {
                AdLogUtil.Log().e("root file is not exists, don't delete")
                false
            }
        } catch (e: java.lang.Exception) {
            AdLogUtil.Log().e(Log.getStackTraceString(e))
            return false
        }
    }

    /**
     * 得到需要查询的FeedCodes集合
     *
     * @param feedVOList feed列表哦数据
     * @return FeedCodes集合
     */
    fun getOfflineFeedCodes(feedVOList: List<FeedVO>?): ArrayList<String> {
        val offlineFeedCodes = ArrayList<String>()
        if (!CollectionUtils.isNotEmpty(feedVOList)) return offlineFeedCodes
        if (feedVOList != null) {
            for (feedVO in feedVOList) {
                if (TextUtils.isEmpty(feedVO.feedCode)) continue
                feedVO.feedCode?.let { offlineFeedCodes.add(it) }
            }
        }
        AdLogUtil.Log().i(
            CommonLogUtil.TAG,
            "getOfflineFeedCodes -> offlineFeedCodes.value = $offlineFeedCodes"
        )
        return offlineFeedCodes
    }

    /**
     * 获取feed列表里的图片集合
     *
     * @param feedVOList feed列表数据
     */
    fun getFeedImageLists(feedVOList: List<FeedVO?>?): List<DownImageInfo> {
        val images: MutableList<DownImageInfo> = ArrayList()
        if (feedVOList != null && CollectionUtils.isNotEmpty(feedVOList)) {
            for (feedVO in feedVOList) {
                if (feedVO == null) continue
                if (CollectionUtils.isNotEmpty(feedVO.coverImageUrls)) {
                    for (imageInfo in feedVO.coverImageUrls!!) {
                        if (!TextUtils.isEmpty(imageInfo)) {
                            images.add(
                                DownImageInfo(
                                    imageInfo,
                                    Constants.ImageType.FEED_LIST_IMAGE, feedVO.feedCode
                                )
                            )
                        }
                    }
                }
            }
        }
        return images
    }

    /**
     * 获取feed详情里的图片集合
     *
     * @param feedDetailVOList feed详情数据集合
     */
    fun getDetailImageLists(
        feedDetailVOList: List<FeedDetailVO?>?
    ): List<DownImageInfo> {
        val images: MutableList<DownImageInfo> = mutableListOf()
        if (CollectionUtils.isNotEmpty(feedDetailVOList)) {
            for (feedDetailVO in feedDetailVOList!!) {
                if (feedDetailVO != null && CollectionUtils.isNotEmpty(feedDetailVO.paragraphJson)) {
                    for (paragraphJsonVO in feedDetailVO.paragraphJson!!) {
                        if (!TextUtils.isEmpty(paragraphJsonVO.imgurl)) {
                            images.add(
                                DownImageInfo(
                                    paragraphJsonVO.imgurl,
                                    Constants.ImageType.FEED_DETAIL_IMAGE,
                                    feedDetailVO.feedCode
                                )
                            )
                        }
                    }
                }
            }
        }
        return images
    }

    /**
     * 根据feedCode获取缓存的feedDetail数据
     */
    fun getFeedDetailVOList(feedCode: String?): FeedDetailVO? {
        val feedDetailVOList =
            UnionWorker.getInstance(CoreUtil.getContext(), null).unionCache?.feedDetailList
        if (TextUtils.isEmpty(feedCode)) return null
        if (feedDetailVOList != null && CollectionUtils.isNotEmpty(feedDetailVOList)) {
            for (feedDetail in feedDetailVOList) {
                if (feedCode == feedDetail.feedCode) {
                    return feedDetail
                }
            }
        }
        return null
    }

    /**
     * 删除feed列表的图片资源
     */
    fun deleteFeedImage(): Boolean {
        try {
            AdLogUtil.Log().i(CommonLogUtil.TAG, "prepare delete feed images")
            val deleteFilePath =
                FileUtil.getAppFilePath(CoreUtil.getContext()) + File.separator +
                        Constants.DownPath.OFFLINE_FOLDER_NAME + File.separator + Constants.DownPath.FEED_LIST_IMAGE_NAME
            val deleteFile = File(deleteFilePath)
            return if (deleteFile.exists()) {
                if (!FileUtil.deleteFile(deleteFile)) {
                    AdLogUtil.Log().e("deleteFeedImage --> delete root file fail!!")
                    false
                } else {
                    true
                }
            } else {
                AdLogUtil.Log().e("deleteFeedImage --> root file is not exists, don't delete")
                false
            }
        } catch (e: java.lang.Exception) {
            AdLogUtil.Log().e(Log.getStackTraceString(e))
            return false
        }
    }

    /**
     * 删除feed详情的图片资源
     */
    fun deleteContentImage(): Boolean {
        try {
            AdLogUtil.Log().i(CommonLogUtil.TAG, "prepare delete content images")
            val deleteFilePath =
                FileUtil.getAppFilePath(CoreUtil.getContext()) + File.separator +
                        Constants.DownPath.OFFLINE_FOLDER_NAME + File.separator + Constants.DownPath.FEED_DETAIL_IMAGE_NAME
            val deleteFile = File(deleteFilePath)
            return if (deleteFile.exists()) {
                if (!FileUtil.deleteFile(deleteFile)) {
                    AdLogUtil.Log().e("deleteContentImage --> delete root file fail!!")
                    false
                } else {
                    true
                }
            } else {
                AdLogUtil.Log().e("deleteContentImage --> root file is not exists, don't delete")
                false
            }
        } catch (e: java.lang.Exception) {
            AdLogUtil.Log().e(Log.getStackTraceString(e))
            return false
        }
    }

    /**
     * 请求内容联盟sdk云控
     */
    fun initUnion(context: Context, unionSetting: UnionSetting) {
        try {
            val startTime = System.currentTimeMillis()
            getConfigStoreInstance(context).loadForService(object : ConfigResultListener {
                override fun onLoadDataFailure(code: Int, message: String?) {
                    initOther(context, unionSetting)
                }

                override fun onLoadDataSuccess(sdkConfigInfo: SdkConfigInfo?) {
                    if (sdkConfigInfo != null) {
                        AdLogUtil.Log().i(
                            CommonLogUtil.TAG,
                            "sdk config --> sdkConfigInfo " + sdkConfigInfo + " ,time =  " + (System.currentTimeMillis() - startTime)
                        )
                        if (sdkConfigInfo.openSdkCache == null) {
                            initOther(context, unionSetting)
                            return
                        }
                        if (sdkConfigInfo.openSdkCache) {
                            initOther(context, unionSetting)
                        } else {
                            val closeVersion = sdkConfigInfo.closeVersions
                            AdLogUtil.Log().i(
                                CommonLogUtil.TAG,
                                "sdk config --> closeVersion = " + closeVersion + " , current code = " + BuildConfig.SDK_CODE
                            )
                            if (!TextUtils.isEmpty(closeVersion) && !closeVersion!!.contains(
                                    BuildConfig.SDK_CODE
                                )
                            ) {
                                initOther(context, unionSetting)
                            }
                        }
                    } else {
                        AdLogUtil.Log().i(CommonLogUtil.TAG, "sdk config --> sdkConfigInfo is null")
                        initOther(context, unionSetting)
                    }
                }
            })
        } catch (e: Exception) {
            initUnion(context, unionSetting)
            AdLogUtil.Log().e(Log.getStackTraceString(e))
        }
    }

    /**
     * 处理离线数据中多张图时，只保留一张图片
     */
    fun handleSingleImageData(feedVOList: List<FeedVO>?): List<FeedVO>? {
        if (feedVOList != null && CollectionUtils.isNotEmpty(feedVOList)) {
            for (feedVO: FeedVO in feedVOList) {
                if (CollectionUtils.isNotEmpty(feedVO.coverImageUrls) && feedVO.coverImageUrls != null && feedVO.coverImageUrls!!.size > 1) {
                    val firstCoverImage = feedVO.coverImageUrls?.get(0) ?: ""
                    feedVO.coverImageUrls?.clear()
                    feedVO.coverImageUrls?.add(firstCoverImage)
                }
            }
        }
        return feedVOList
    }

    /**
     * 设置环境
     */
    fun setDebug(unionSetting: UnionSetting) {

        if (!unionSetting.isLogDebug) {
            AdLogUtil.Log().setLogSwitch(Log.isLoggable("BAOBAB", Log.DEBUG))
        } else {
            AdLogUtil.Log().setLogSwitch(unionSetting.isLogDebug)
        }
    }

    /**
     * 初始化内容联盟和athena
     */
    private fun initOther(context: Context, unionSetting: UnionSetting) {
        AdLogUtil.Log().i(CommonLogUtil.TAG, "init union sdk")
        UnionWorker.getInstance(context, unionSetting)
        AthenaTracker.trackInitSdk(1)
    }
}