package com.cloud.baobabsland.athena

import android.content.Context
import android.os.Bundle
import android.os.Handler
import android.text.TextUtils
import android.util.Log
import com.cloud.baobabsland.BuildConfig
import com.cloud.baobabsland.constant.Constants
import com.cloud.baobabsland.utils.*
import com.cloud.baobabsland.worker.UnionWorker
import com.google.gson.JsonObject
import com.transsion.athenacust.AthenaCust
import com.transsion.core.CoreUtil
import com.transsion.core.pool.TranssionPoolManager
import com.transsion.ga.AthenaAnalytics

object AthenaTracker {
    private var handler: Handler? = null
    private const val REQUEST_TIME = "request_time"

    fun initAthena(context: Context?, isDebug: Boolean) {
        try {
            AthenaAnalytics.init(
                context,
                Constants.CommonConstant.APP_ID,
                Constants.CommonConstant.TID,
                isDebug,
                false
            )
            AthenaAnalytics.setTest(isDebug)
            AthenaAnalytics.setDebug(isDebug)
            TranssionPoolManager.getInstance().addTask {
                //预加载，防止第一次初始化为空的情况
                val vaid = AthenaAnalytics.getAppVAID(context, true)
                if (!TextUtils.isEmpty(vaid)) {
                    PreferencesHelper.instance.putString(Constants.KeyName.KEY_VAID, vaid)
                }
            }

        } catch (e: Exception) {
            AdLogUtil.Log().e(Log.getStackTraceString(e))
        }

    }

    @Synchronized
    private fun initHandler() {
        if (handler == null) {
            val handlerThread = AthenaUtil.instance
            AthenaUtil.start()
            handler = Handler(handlerThread.looper)
            AdLogUtil.Log().d(CommonLogUtil.TAG, "AthenaTracker initHandlerThread")
        }
    }

    /**
     * 上报埋点新开一个子线程
     */
    private fun trackInNewThread(runnable: Runnable) {
        initHandler()
        handler?.post(runnable)
    }

    /**
     * 获取通用埋点数据
     */
    private fun getCommonBundle(): Bundle {
        val bundle = Bundle()
        var appId = ""
        val unionWorker = UnionWorker.getInstance(CoreUtil.getContext(), null)
        if (unionWorker.unionSetting != null) {
            appId = unionWorker.unionSetting?.appId.toString()
        }
        bundle.putString("appid", appId)
        bundle.putString("vaid", CommonUtils.getVAID())
        bundle.putString("gaid", CommonUtils.getGAID())
        bundle.putString("udid", CommonUtils.getUDID())
        bundle.putInt("udidType", PreferencesHelper.instance.getInt(Constants.KeyName.KEY_UDID_TYPE))
        bundle.putString("union_package_name", CoreUtil.getContext().packageName)
        bundle.putString("union_version", BuildConfig.SDK_VERSION)
        bundle.putString("country", CommonUtils.getCountry())
        bundle.putString("language", CommonUtils.getSystemLanguage())
        bundle.putLong("sdk_ts", System.currentTimeMillis())
        bundle.putInt("is_offline", if (CommonUtils.checkNetworkState()) 1 else 0)
        return bundle
    }

    /**
     * H5埋点上报
     * @param eventName 事件名称
     * @param jsonObject h5的埋点内容
     */
    @JvmStatic
    fun trackDataForHtml(eventName: String?, jsonObject: String?) {
        trackInNewThread {
            AdLogUtil.Log().i(
                "Athena",
                "--------trackDataForHtml() -> eventName = $eventName , jsonObject = $jsonObject"
            )
            val bundle = getCommonBundle()
            bundle.putString("data", jsonObject)
            AthenaCust(eventName, Constants.CommonConstant.TID).trackCommon(bundle, null).submit()
        }
    }

    /**
     * SDK埋点上报
     * @param eventName 时间名称
     * @param jsonObject 实验数据
     */
    private fun trackData(eventName: String, jsonObject: String) {
        trackInNewThread {
            AdLogUtil.Log().i(
                "Athena",
                "--------trackData() -> eventName = $eventName , jsonObject = $jsonObject"
            )
            val bundle = getCommonBundle()
            bundle.putString("data", jsonObject)
            AthenaCust(eventName, Constants.CommonConstant.TID).trackCommon(bundle, null).submit()
        }
    }

    /**
     * 初始化埋点
     * @param status 初始化的结果
     */
    fun trackInitSdk(status: Int) {
        val jsonObject = JsonObject()
        jsonObject.addProperty("init_status", status)
        trackData(Constants.CommonConstant.EVENT_SDK_INIT, jsonObject.toString())
    }

    /**
     * gslb初始化成功埋点
     * @param url 下发的url
     */
    @JvmStatic
    fun trackInitGslb(url: String?) {
        val jsonObject = JsonObject()
        jsonObject.addProperty("gslb_url", url)
        trackData(Constants.CommonConstant.EVENT_GSLB_INIT, jsonObject.toString())
    }

    /**
     * 云控请求判断埋点
     * @param requestType 判断时机类型 1：冷启动，2：网络切换
     */
    @JvmStatic
    fun trackDataByConfigReq(requestType: Int) {
        trackInNewThread {
            val jsonObject = JsonObject()
            jsonObject.addProperty("request_type", requestType)
            jsonObject.addProperty("union_times", System.currentTimeMillis())
            trackData(Constants.CommonConstant.EVENT_CONFIG_REQUEST, jsonObject.toString())
        }
    }

    /**
     * 云控响应判断埋点
     * @param openCache 缓存开关
     * @param interval 请求间隔
     * @param maxExpiredTime 过期时间
     * @param offlineFeedNum 离线缓存条数
     */
    @JvmStatic
    fun trackDataByConfigResponse(
        openCache: Boolean,
        interval: Int,
        maxExpiredTime: Int,
        offlineFeedNum: Int
    ) {
        trackInNewThread {
            val jsonObject = JsonObject()
            jsonObject.addProperty("is_open_cache", openCache)
            jsonObject.addProperty("union_interval", interval)
            jsonObject.addProperty("max_expired_time", maxExpiredTime)
            jsonObject.addProperty("config_num", offlineFeedNum)
            trackData(Constants.CommonConstant.EVENT_CONFIG_RESULT, jsonObject.toString())
        }
    }

    /**
     * 请求下载离线zip包
     * @param downloadUrl 下载地址
     */
    @JvmStatic
    fun trackDataByDownLoadZip(downloadUrl: String?) {
        trackInNewThread {
            val jsonObject = JsonObject()
            jsonObject.addProperty("download_data_type", 1)
            jsonObject.addProperty("download_url", downloadUrl)
            jsonObject.addProperty(REQUEST_TIME, System.currentTimeMillis())
            trackData(Constants.CommonConstant.EVENT_DOWNLOAD_ZIP, jsonObject.toString())
        }
    }

    /**
     * 下载离线zip包结果
     * @param checkMd5 md5检验结果
     */
    @JvmStatic
    fun trackDataByDownLoadZipRes(code: Int, message: String?, checkMd5: Boolean) {
        trackInNewThread {
            val jsonObject = JsonObject()
            jsonObject.addProperty("download_respones_code", code)
            jsonObject.addProperty("download_respones_message", message)
            jsonObject.addProperty("checkmd5", if (checkMd5) 1 else 0)
            jsonObject.addProperty(REQUEST_TIME, System.currentTimeMillis())
            trackData(Constants.CommonConstant.EVENT_DOWNLOAD_ZIP_RESULT, jsonObject.toString())
        }
    }

    /**
     * 下载物料的结果
     */
    @JvmStatic
    fun trackDataByDownloadPic(feedCode: String?, downloadUrl: String?, downloadResult: Int) {
        trackInNewThread {
            val jsonObject = JsonObject()
            jsonObject.addProperty("feedCode", feedCode)
            jsonObject.addProperty("download_url", downloadUrl)
            jsonObject.addProperty("download_result", downloadResult)
            jsonObject.addProperty(REQUEST_TIME, System.currentTimeMillis())
            trackData(Constants.CommonConstant.EVENT_DOWNLOAD_MATERIAL, jsonObject.toString())
        }
    }

    /**
     * 删除缓存数据埋点
     */
    fun deleteLocalData(openCache: Boolean, deleteResult: Int) {
        trackInNewThread {
            val jsonObject = JsonObject()
            jsonObject.addProperty("is_open_cache", openCache)
            jsonObject.addProperty("delete_result", deleteResult)
            jsonObject.addProperty(REQUEST_TIME, System.currentTimeMillis())
            trackData(Constants.CommonConstant.EVENT_DELETE_LOCAL_DATA, jsonObject.toString())
        }
    }

    /**
     * 媒体请求页面埋点
     * @param loadType 1：activity启动 2：fragment启动
     */
    @JvmStatic
    fun mediaLoadType(loadType: Int) {
        trackInNewThread {
            val jsonObject = JsonObject()
            jsonObject.addProperty("request_type", loadType)
            jsonObject.addProperty(REQUEST_TIME, System.currentTimeMillis())
            trackData(Constants.CommonConstant.EVENT_MEDIA_CALL_REQUEST, jsonObject.toString())
        }
    }

    /**
     * 列表数据请求埋点
     * @param channelId 导航栏tab页id
     * @param recExpId 策略实验id
     */
    @JvmStatic
    fun trackByOfflineFeedReq(channelId: Int, recExpId: String?) {
        trackInNewThread {
            val jsonObject = JsonObject()
            jsonObject.addProperty("channel_id", channelId.toString())
            jsonObject.addProperty("rec_exp_id", recExpId)
            trackData(Constants.CommonConstant.EVENT_OFFLINE_FEED_REQUEST, jsonObject.toString())
        }
    }

    /**
     * 详情数据请求埋点
     * @param feedCode 单个feedCode
     */
    @JvmStatic
    fun trackByOfflineFeedReq(feedCode: String?) {
        trackInNewThread {
            val jsonObject = JsonObject()
            jsonObject.addProperty("feedCodes", feedCode)
            trackData(Constants.CommonConstant.EVENT_OFFLINE_DETAIL_REQUEST, jsonObject.toString())
        }
    }

    /**
     * 缓存数据的接口返回结果
     * @param eventName 事件名称
     * @param responseNum 响应返回的数据条数
     * @param isSuccess  失败：0；成功：1
     * @param code 返回响应码
     * @param message 响应的message
     */
    @JvmStatic
    fun trackByOfflineDataRes(
        eventName: String,
        responseNum: Int,
        isSuccess: Int,
        code: Int,
        message: String?,
        recExpId: String?
    ) {
        trackInNewThread {
            val jsonObject = JsonObject()
            jsonObject.addProperty("response_data_num", responseNum)
            jsonObject.addProperty("response_state", isSuccess)
            jsonObject.addProperty("response_code", code)
            jsonObject.addProperty("response_message", message)
            jsonObject.addProperty("rec_exp_id", recExpId)

            trackData(eventName, jsonObject.toString())
        }
    }

    /**
     * 显示网络浮层的埋点
     * @param tipsType 1:列表 2.详情
     */
    @JvmStatic
    fun trackShowNetTip(tipsType: Int) {
        trackInNewThread {
            val jsonObject = JsonObject()
            jsonObject.addProperty("tips_type", tipsType)
            jsonObject.addProperty("ts", System.currentTimeMillis())
            trackData(Constants.CommonConstant.EVENT_SHOW_NO_NETWORK_TIPS, jsonObject.toString())
        }
    }

    /**
     * 关闭网络浮层的埋点
     * @param tipsType 1:列表 2.详情
     */
    @JvmStatic
    fun trackCloseNetTip(tipsType: Int) {
        trackInNewThread {
            val jsonObject = JsonObject()
            jsonObject.addProperty("tips_type", tipsType)
            jsonObject.addProperty("ts", System.currentTimeMillis())
            trackData(Constants.CommonConstant.EVENT_CLOSE_NO_NETWORK_TIPS, jsonObject.toString())
        }
    }

    /** H5页面加载完成埋点 。  */
    fun trackWebViewLoadFinish(startTime: Long) {
        trackInNewThread {
            val jsonObject = JsonObject()
            jsonObject.addProperty("load_time", (System.currentTimeMillis() - startTime))
            jsonObject.addProperty("ts", System.currentTimeMillis())
            trackData(Constants.CommonConstant.EVENT_WEB_VIEW_LOAD_FINISH, jsonObject.toString())
        }
    }

    /** H5页面加载url前耗时埋点 。*/
    fun trackLoadUrlBefore(startTime: Long) {
        trackInNewThread {
            val jsonObject = JsonObject()
            jsonObject.addProperty("load_time", (System.currentTimeMillis() - startTime))
            trackData(Constants.CommonConstant.EVENT_WEB_VIEW_LOAD_URL_BEFORE, jsonObject.toString())
        }
    }
}