package com.cloud.baobabsland.config

import android.content.Context
import android.text.TextUtils
import android.util.Log
import com.cloud.baobabsland.bean.PostBody
import com.cloud.baobabsland.bean.response.SdkConfigResponse
import com.cloud.baobabsland.config.utils.*
import com.cloud.baobabsland.constant.Constants
import com.cloud.baobabsland.constant.ErrorCode
import com.cloud.baobabsland.constant.ServerConfig
import com.cloud.baobabsland.http.ServerRequest
import com.cloud.baobabsland.http.listener.CommonResponseListener
import com.cloud.baobabsland.utils.AdLogUtil
import java.util.concurrent.atomic.AtomicBoolean

abstract class AbsConfigStore protected constructor(
    private val context: Context
) {

    private val requestFlag = AtomicBoolean(false)

    open fun getPackageName(): String? = null

    /**
     * 从服务端获取配置信息.
     */
    @JvmOverloads
    fun loadForService(configResultListener: ConfigResultListener? = null
    ) {
        if (requestFlag.compareAndSet(false, true)) {
            io().execute(
                RequestRunnable(
                    this,
                    configResultListener = configResultListener
                )
            )
        }
    }

    internal class RequestRunnable(
        private val store: AbsConfigStore,
        val configResultListener: ConfigResultListener? = null
    ) :
        Runnable {
        override fun run() {
            try {
                val packageName = if (store.getPackageName()
                        .isNullOrBlank()
                ) store.context.packageName else store.getPackageName()

                val serverRequest = ServerRequest()
                    .setListener(object : CommonResponseListener<SdkConfigResponse?>() {

                        override fun onRequestSuccess(
                            statusCode: Int,
                            response: SdkConfigResponse?
                        ) {
                            if (response != null) {
                                if (response.code == Constants.ResponseCode.BUSINESS_SUCCESS_CODE) {
                                    configResultListener?.onLoadDataSuccess(response.data)
                                } else {
                                    val message = if(TextUtils.isEmpty(response.message)) ERROR_1001_MESSAGE else response.message
                                    configResultListener?.onLoadDataFailure(ERROR_1001, message)
                                }
                            } else {
                                configResultListener?.onLoadDataFailure(ERROR_1002, ERROR_1002_MESSAGE)
                            }
                        }

                        override fun onRequestError(adError: ErrorCode) {
                            if (adError == null) return
                            store.requestFlag.set(false)
                            configResultListener?.onLoadDataFailure(adError.errorCode, adError.errorMessage)
                            AdLogUtil.Log().e("sdkConfig ->  error code = " + adError.errorCode + " , message = " + adError.errorMessage)
                        }

                    })
                    .setPostBody { PostBody.getConfigSdkBody(packageName) }
                    .setDebug(true)
                    .setUrl(ServerConfig.sdkConfigUrl)

                serverRequest?.netRequestPreExecute(null, null)
            } catch (e: Throwable) {
                AdLogUtil.netLog("AdServerRequest --> " + Log.getStackTraceString(e))

            } finally {
                store.requestFlag.set(false)
            }
        }
    }
}