package com.cloud.baobabsland.utils;

import android.annotation.SuppressLint;
import android.app.ActivityManager;
import android.app.Application;
import android.content.Context;
import android.net.ConnectivityManager;
import android.net.NetworkCapabilities;
import android.os.Build;
import android.os.LocaleList;
import android.os.Looper;
import android.text.TextUtils;
import android.util.Log;

import com.cloud.baobabsland.UnionSetting;
import com.cloud.baobabsland.bean.PostBody;
import com.cloud.baobabsland.bean.response.SaveUDIDResponse;
import com.cloud.baobabsland.constant.Constants;
import com.cloud.baobabsland.constant.ErrorCode;
import com.cloud.baobabsland.constant.ServerConfig;
import com.cloud.baobabsland.http.ServerRequest;
import com.cloud.baobabsland.http.listener.CommonResponseListener;
import com.cloud.baobabsland.worker.UnionWorker;
import com.transsion.core.CoreUtil;
import com.transsion.core.deviceinfo.DeviceInfo;
import com.transsion.core.pool.TranssionPoolManager;
import com.transsion.core.utils.NetUtil;
import com.transsion.ga.AthenaAnalytics;
import com.transsion.gslb.GslbSdk;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStreamReader;
import java.lang.reflect.Method;
import java.util.List;
import java.util.Locale;
import java.util.TimeZone;

public class CommonUtils {
    private static String zoneStr = "";

    /**
     * 字符串判空
     */
    public static boolean isEmpty(String info) {
        return info == null || info.length() == 0;
    }

    /**
     * 获取时区
     *
     * @return 时区
     */
    public static String getTimeZone() {
        if (!isEmpty(zoneStr)) return zoneStr;

        TimeZone timeZone = TimeZone.getDefault();
        String timeZoneId = timeZone.getID();
        if (isEmpty(timeZoneId)) {
            String name = timeZone.getDisplayName(false, TimeZone.SHORT);
            if (isEmpty(name)) {
                zoneStr = "";
            } else {
                zoneStr = name;
            }
        } else {
            zoneStr = timeZoneId;
        }

        return zoneStr;
    }

    /**
     * 获取当前进程的名称
     *
     * @param context 上下文
     * @return 进程名称
     */
    public static String getCurProcessName(Context context) {
        String processName = null;
        try {
            if (Build.VERSION.SDK_INT >= 28) {
                processName = Application.getProcessName();
            } else {
                int pid = android.os.Process.myPid();
                ActivityManager activityManager = (ActivityManager) context.getSystemService(Context.ACTIVITY_SERVICE);
                List<ActivityManager.RunningAppProcessInfo> appProcesses = activityManager.getRunningAppProcesses();
                if (appProcesses != null) {
                    for (ActivityManager.RunningAppProcessInfo appProcess : appProcesses) {
                        if (appProcess.pid == pid) {
                            processName = appProcess.processName;
                        }
                    }
                }
            }
        } catch (Exception e) {
            AdLogUtil.Log().e(Log.getStackTraceString(e));
        }
        if (TextUtils.isEmpty(processName)) {
            processName = context.getPackageName();
        }
        return processName;
    }

    /**
     * 把缓存写入文件
     */
    public static void bufferSave(String fileName, String msg) {
        if (!FileUtil.createOrExistsFile(new File(fileName))) {
            AdLogUtil.Log().e("createImageByPath", "create file <" + fileName + "> failed.");
        }

        TranssionPoolManager.getInstance().addTask(() -> {
            try (FileOutputStream fos = new FileOutputStream(fileName)) {

                //清空文件重新写入
                fos.write(msg.getBytes());
            } catch (Exception e) {
                AdLogUtil.Log().e(Log.getStackTraceString(e));
            }
        });
    }

    /**
     * 读取缓存
     */
    public static String bufferRead(File file) {
        if (file == null || !file.exists()) {
            return "";
        }

        try (FileInputStream fis = new FileInputStream(file);
             BufferedReader reader = new BufferedReader(new InputStreamReader(fis))) {
            StringBuilder out = new StringBuilder();
            String line;
            while ((line = reader.readLine()) != null) {
                out.append(line);
            }
            return out.toString();
        } catch (Exception e) {
            AdLogUtil.Log().e(Log.getStackTraceString(e));
        }

        return "";
    }

    private static Locale getLocale() {
        Locale locale;
        try {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                locale = LocaleList.getDefault().get(0);
            } else {
                locale = Locale.getDefault();
            }
        } catch (Exception e) {
            locale = Locale.getDefault();
            AdLogUtil.Log().e(Log.getStackTraceString(e));
        }

        return locale;
    }

    /**
     * @return 返回当前系统语言
     */
    public static String getSystemLanguage() {
        return getLocale().getLanguage();
    }

    /**
     * 获取国家
     */
    public static String getCountry() {
        String country = "";
        try {
            @SuppressLint("PrivateApi")
            Class<?> systemProperties = Class.forName("android.os.SystemProperties");
            Method get = systemProperties.getMethod("get", String.class, String.class);
            country = (String) get.invoke(systemProperties, "persist.sys.oobe_country", "unknown");
        } catch (Exception e) {
            AdLogUtil.Log().e(Log.getStackTraceString(e));
        }

        if (TextUtils.isEmpty(country) || "unknown".equals(country)) {
            country = getLocale().getCountry();
        }
        return country;
    }

    /**
     * 获取mcc
     *
     * @return mcc
     */
    public static String getMcc() {
        String deviceOperator = DeviceInfo.getSimOperator();
        if (!TextUtils.isEmpty(deviceOperator) && deviceOperator.length() >= 3) {
            return deviceOperator.substring(0, 3);
        }
        return "";
    }


    /**
     * 判断网络是否真的可用
     *
     * @return 网络是否可用
     */
    public static boolean checkNetworkState() {
        boolean isNetAvailable = NetUtil.checkNetworkState();
        //Android6.0（23）以上才有效果
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M && isNetAvailable) {
            try {
                ConnectivityManager mConnectivityManager =
                        (ConnectivityManager) CoreUtil.getContext().getApplicationContext().getSystemService(Context.CONNECTIVITY_SERVICE);
                NetworkCapabilities networkCapabilities = mConnectivityManager.getNetworkCapabilities(mConnectivityManager.getActiveNetwork());
                if (networkCapabilities != null) {
                    return networkCapabilities.hasCapability(NetworkCapabilities.NET_CAPABILITY_VALIDATED);
                }
            } catch (Exception e) {
                AdLogUtil.Log().e(Log.getStackTraceString(e));
                return true;
            }
        }
        return isNetAvailable;

    }

    /**
     * 手机机型
     */
    public static String getModel() {
        return Build.MODEL;
    }

    //GAID
    private static String mGAId = "";

    /**
     * 获取GAID
     *
     * @return gaid
     */
    public static String getGAID() {
        if (TextUtils.isEmpty(mGAId)) {
            mGAId = PreferencesHelper.getInstance().getString(Constants.KeyName.KEY_GAID, "");
            if (Looper.myLooper() == Looper.getMainLooper()) {
                TranssionPoolManager.getInstance().addTask(CommonUtils::queryGaid);
            } else {
                queryGaid();
            }
        }
        return mGAId;
    }

    private static void queryGaid() {
        try {
            AdvertisingIdClient.AdInfo adInfo = AdvertisingIdClient.getAdvertisingIdInfo(CoreUtil.getContext());
            if (null != adInfo) {
                if (!adInfo.getId().equals(mGAId)) {
                    mGAId = adInfo.getId();
                    PreferencesHelper.getInstance().putString(Constants.KeyName.KEY_GAID, mGAId);
                }
                AdLogUtil.Log().i(CommonLogUtil.TAG, "device gaid = " + mGAId);
            }
        } catch (Exception e) {
            AdLogUtil.Log().e(Log.getStackTraceString(e));
        }
    }

    /**
     * 判断gaid的有效性
     *
     * @return true：无效的gaid，false有效的gaid
     */
    private static boolean matchGaidIsZeroOrNull(String gaid) {
        if (null == gaid || "".equals(gaid)) {
            return true;
        }
        String removeDelimiterGaid = gaid.replace("-", "");
        char[] chars = removeDelimiterGaid.toCharArray();
        int zeroNum = 0;
        for (char c : chars) {
            // 计算字符串中，字符0的个数
            if (c == 48) {
                zeroNum++;
            }
        }
        return removeDelimiterGaid.length() == zeroNum;
    }

    /**
     * 获取Athena的vaid
     *
     * @return 返回vaid
     */
    public static String getVAID() {
        String vaid = PreferencesHelper.getInstance().getString(Constants.KeyName.KEY_VAID, "");
        try {
            if (TextUtils.isEmpty(vaid)) {
                vaid = AthenaAnalytics.getAppVAID(CoreUtil.getContext());
                if (!TextUtils.isEmpty(vaid)) {
                    PreferencesHelper.getInstance().putString(Constants.KeyName.KEY_VAID, vaid);
                }
            }
        } catch (Exception e) {
            return "";
        }

        return vaid;
    }

    /**
     * 获取内容联盟ID
     *
     * @return 返回udid
     */
    public static String getUDID() {
        String udid = PreferencesHelper.getInstance().getString(Constants.KeyName.KEY_UDID, "");
        int udidType = 2;
        if (TextUtils.isEmpty(udid)) {
            String gaid = CommonUtils.getGAID();
            if (matchGaidIsZeroOrNull(gaid)) {
                String vaid = getVAID();
                if (!TextUtils.isEmpty(vaid)) {
                    udid = "v" + vaid;
                    udidType = 2;
                } else {
                    udid = "";
                }

            } else {
                udid = "g" + gaid;
                udidType = 1;
            }
            if (!TextUtils.isEmpty(udid)) {
                PreferencesHelper.getInstance().putString(Constants.KeyName.KEY_UDID, udid);
                PreferencesHelper.getInstance().putInt(Constants.KeyName.KEY_UDID_TYPE, udidType);
                asyncSaveIfAbsent();
            }
        } else {
            return udid;
        }
        return udid;
    }

    /**
     * 异步保存不存在的udid
     */
    private static void asyncSaveIfAbsent() {
        boolean isDebug = false;
        UnionSetting unionSetting = UnionWorker.getInstance(CoreUtil.getContext(), null).getUnionSetting();
        if (unionSetting != null) {
            isDebug = unionSetting.isDebug();
        }
        try {
            ServerRequest serverRequest = new ServerRequest()
                    .setListener(new CommonResponseListener<SaveUDIDResponse>() {
                        @Override
                        protected void onRequestError(ErrorCode adError) {
                            AdLogUtil.Log().i(CommonLogUtil.TAG, "asyncSaveIfAbsent is fail");
                        }

                        @Override
                        protected void onRequestSuccess(int statusCode, SaveUDIDResponse response) {
                            AdLogUtil.Log().i(CommonLogUtil.TAG, "asyncSaveIfAbsent is success");
                        }
                    })
                    .setPostBody(PostBody::getSaveUDIDBody)
                    .setDebug(isDebug)
                    .setUrl(GslbSdk.getDomain(ServerConfig.getServerUrl() + Constants.HOST.SAVE_UDID_API, true));
            if (serverRequest != null) {
                serverRequest.netRequestPreExecute(null, null);
            }
        } catch (Exception e) {
            AdLogUtil.Log().e(Log.getStackTraceString(e));
        }


    }
}
