package com.cloud.baobabsland.utils;

import static android.os.Environment.MEDIA_MOUNTED;

import android.Manifest;
import android.content.Context;
import android.os.Environment;
import android.text.TextUtils;
import android.util.Log;

import com.transsion.core.CoreUtil;
import com.transsion.core.utils.PermissionUtil;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

/**
 * 文件工具类
 */

public class FileUtil {
    private static int sBufferSize = 524288;

    /**
     * 获得SDCard/Android/data/你的应用的包名/files/ 路径
     *
     * @return
     */
    public static File getAppDateFilePath() {
        Context context = CoreUtil.getContext();
        if (PermissionUtil.lacksPermissions(Manifest.permission.WRITE_EXTERNAL_STORAGE, Manifest.permission.READ_EXTERNAL_STORAGE)) {
            return null;
        }
        return context.getExternalFilesDir(null);
    }


    /**
     * 获得SDCard/android/data/你的应用包名/cache/路径
     *
     * @return
     */
    public static File getAppDateCachePath() {
        Context context = CoreUtil.getContext();
        return context.getExternalCacheDir();
    }

    /**
     * 该方法会判断当前sd卡是否存在，然后选择缓存地址
     *
     * @param context
     * @param uniqueName
     * @return
     */
    public static File getDiskCacheDir(Context context, String uniqueName) {
        String cachePath;
        if (Environment.MEDIA_MOUNTED.equals(Environment.getExternalStorageState()) || !Environment.isExternalStorageRemovable()) {
            cachePath = context.getExternalCacheDir().getPath();
        } else {
            cachePath = context.getCacheDir().getPath();
        }
        return new File(cachePath + File.separator + uniqueName);
    }

    public static String getStrFile(String filePath) {
        if (TextUtils.isEmpty(filePath)) {
            return "";
        }
        File file = new File(filePath);
        if (!file.exists() || !file.isFile()) {
            return null;
        }
        try(FileReader fileReader = new FileReader(file)) {
            char[] bt = new char[1024];
            StringBuilder sb = new StringBuilder();
            int len = 0;
            while ((len = fileReader.read(bt)) != -1) {
                sb.append(bt, 0, len);
            }
            if (sb.length() == 0) {
                return null;
            }
            return sb.toString();
        } catch (Exception e) {
            AdLogUtil.Log().e(Log.getStackTraceString(e));
            return null;
        }
    }

    /**
     * 删除文件夹下所有文件
     *
     * @param dir 文件
     */
    public static boolean deleteFile(File dir) {
        if (dir == null || !dir.exists() || !dir.isDirectory()) {
            return false;
        }
        boolean deleteResult = false;
        try {
            for (File file : dir.listFiles()) {
                if (file.isFile()) {
                    if (!file.delete()) { // 删除所有文件
                        AdLogUtil.Log().e("deleteFile() -> delete fail!!");
                    }
                } else if (file.isDirectory()) {
                    deleteFile(file); // 递规的方式删除文件夹
                }
            }
            deleteResult = dir.delete();// 删除目录本身
        } catch (Exception e) {
            AdLogUtil.Log().e(Log.getStackTraceString(e));
        }
        AdLogUtil.Log().i(CommonLogUtil.TAG, "delete file -> deleteResult = " + deleteResult);
        return deleteResult;
    }

    /**
     * 获取文件路径
     *
     * @return
     */
    public static String getAppFilePath(Context context) {
        String externalStorageState;
        File appCacheDir;
        try {
            externalStorageState = Environment.getExternalStorageState();
        } catch (NullPointerException | IncompatibleClassChangeError e) {
            externalStorageState = "";
        }
        if (MEDIA_MOUNTED.equals(externalStorageState)) {
            appCacheDir = context.getExternalFilesDir(null);
            if (appCacheDir != null && appCacheDir.exists()) {
                return appCacheDir.getAbsolutePath();
            }
        }
        appCacheDir = context.getFilesDir();
        if (appCacheDir != null && !appCacheDir.exists()) {
            return appCacheDir.getAbsolutePath();
        }
        return "";
    }

    /**
     * 判断文件是否存在，不存在则判断是否创建成功
     */
    public static boolean createOrExistsFile(final File file) {
        if (file == null) return false;
        if (file.exists()) return file.isFile();
        if (!createOrExistsDir(file.getParentFile())) return false;
        try {
            return file.createNewFile();
        } catch (IOException e) {
            AdLogUtil.Log().e(Log.getStackTraceString(e));
            return false;
        }
    }

    /**
     * 判断目录是否存在，不存在则判断是否创建成功
     * @param file
     * @return
     */
    public static boolean createOrExistsDir(final File file) {
        return file != null && (file.exists() ? file.isDirectory() : file.mkdirs());
    }

    /**
     * 将输入流写入文件
     * @param file 文件
     * @param downImagePath 下载地址
     * @return 操作是否成功
     */
    public static boolean writeFileFromIS(final File file,
                                          final String downImagePath) {
        if (TextUtils.isEmpty(downImagePath)) {
            return false;
        }
        File downImageFile = new File(downImagePath);
        if (!downImageFile.exists()) {
            AdLogUtil.Log().e("writeFileFromIS() -> downImageFile is not exists!! downImagePath = " + downImagePath);
            return false;
        }
        try(
                InputStream inputStream = new FileInputStream(downImagePath);
                FileOutputStream out = new FileOutputStream(file, false);
                OutputStream os = new BufferedOutputStream(out, sBufferSize)
        ) {
            byte[] data = new byte[sBufferSize];
            for (int len; (len = inputStream.read(data)) != -1; ) {
                os.write(data, 0, len);
            }
            return true;
        } catch (Exception e) {
            AdLogUtil.Log().e(Log.getStackTraceString(e));
            return false;
        }
    }


    /**
     * 得到文件名字
     * @return 文件名字
     */
    public static String getFileName(String urlPath) {
        if (TextUtils.isEmpty(urlPath)) return "";

        File urlFile = new File(urlPath);
        return urlFile.getName();
    }
}
