package com.cloud.baobabsland.webview.view

import android.animation.*
import android.animation.ValueAnimator.AnimatorUpdateListener
import android.content.Context
import android.graphics.*
import android.os.Handler
import android.util.AttributeSet
import android.util.TypedValue
import android.view.animation.DecelerateInterpolator
import android.view.animation.LinearInterpolator
import android.widget.FrameLayout
import com.cloud.baobabsland.R

/**
 * WebView进度条
 */
class WebViewProgress @JvmOverloads constructor(
    context: Context,
    attrs: AttributeSet? = null,
    defStyleAttr: Int = 0
) : FrameLayout(context, attrs, defStyleAttr) {
    /**
     * 当前匀速动画最大的时长
     */
    private var mCurrentMaxUniformDuration = MAX_UNIFORM_SPEED_DURATION

    /**
     * 当前加速后减速动画最大时长
     */
    private var mCurrentMaxDecelerateDuration = MAX_DECELERATE_SPEED_DURATION
    private val mTitleProgressDefaultGap = dip2px(24f)

    /**
     * 进度条的画笔
     */
    private var mProgressPaint: Paint? = null
    private var mProgressHintPaint: Paint? = null
    private var mLeftTextPaint: Paint? = null
    private var mRightTextPaint: Paint? = null
    private var mProgressRightPaint: Paint? = null
    private var mLeftTextWidth = 0f
    private var mLeftTextHeight = 0f
    private var mRightTextWidth = 0f
    private var mProgressTitle: String = ""
    private var mProgressGap = dip2px(PROGRESS_DEFAULT_GAP.toFloat())

    /**
     * 进度条动画
     */
    private var mAnimator: Animator? = null

    /**
     * 控件的宽度
     */
    private var mProgressWidth = 0

    /**
     * 控件的高度
     */
    private var mProgressHeight = 0

    /**
     * 标志当前进度条的状态
     */
    private var mProgressStatus = 0

    /**
     * 第一次过来进度show，后面就是setProgress
     */
    private var isShow = false
    private var mCurrentProgress = 0f
    private var dotStr = ""
    private var mHandler: Handler? = null

    private fun init(context: Context, attrs: AttributeSet?) {
        val typedArray = context.obtainStyledAttributes(attrs, R.styleable.WebViewProgress)
        mProgressHeight = typedArray.getDimensionPixelSize(
            R.styleable.WebViewProgress_progressHeight, dip2px(
                WEB_PROGRESS_DEFAULT_HEIGHT.toFloat()
            )
        )
        mProgressWidth = typedArray.getDimensionPixelSize(
            R.styleable.WebViewProgress_progressWidth, dip2px(
                PROGRESS_DEFAULT_WIDTH.toFloat()
            )
        )
        mProgressGap = typedArray.getDimensionPixelSize(
            R.styleable.WebViewProgress_progressGap, dip2px(
                PROGRESS_DEFAULT_GAP.toFloat()
            )
        )
        mProgressTitle = typedArray.getString(R.styleable.WebViewProgress_progressTitle) ?: ""
        val mProgressTitleColor = typedArray.getColor(
            R.styleable.WebViewProgress_progressTitleColor, Color.parseColor(PROGRESS_TEXT_COLOR)
        )
        val mColor = typedArray.getColor(
            R.styleable.WebViewProgress_progressColor, Color.parseColor(
                WEB_PROGRESS_COLOR
            )
        )
        val mProgressHintColor = typedArray.getColor(
            R.styleable.WebViewProgress_progressHintColor, Color.parseColor(
                WEB_DEFAULT_HINT_COLOR
            )
        )
        mProgressPaint = Paint()
        mProgressPaint?.isAntiAlias = true
        mProgressPaint?.color = mColor
        mProgressPaint?.isDither = true
        mProgressPaint?.strokeCap = Paint.Cap.SQUARE

        mProgressHintPaint = Paint()
        mProgressHintPaint?.isAntiAlias = true
        mProgressHintPaint?.color = mProgressHintColor
        mProgressHintPaint?.isDither = true
        mProgressHintPaint?.strokeCap = Paint.Cap.SQUARE

        mLeftTextPaint = Paint()
        mLeftTextPaint?.isAntiAlias = true
        mLeftTextPaint?.color = mProgressTitleColor
        mLeftTextPaint?.typeface = Typeface.DEFAULT_BOLD
        mLeftTextPaint?.textSize =
            dip2sp(PROGRESS_TEXT_SIZE.toFloat()).toFloat()
        val rect = Rect()
        mLeftTextPaint?.getTextBounds(mProgressTitle, 0, mProgressTitle.length, rect)
        mLeftTextWidth = rect.width().toFloat()
        mLeftTextHeight = rect.height().toFloat()

        mRightTextPaint = Paint()
        mRightTextPaint?.isAntiAlias = true
        mRightTextPaint?.color = mProgressTitleColor
        mRightTextPaint?.typeface = Typeface.DEFAULT_BOLD
        mRightTextPaint?.textSize =
            dip2sp(PROGRESS_TEXT_SIZE.toFloat()).toFloat()
        mRightTextWidth = mRightTextPaint?.measureText("...") ?: 0f

        mProgressRightPaint = Paint()
        mProgressRightPaint?.isAntiAlias = true
        mProgressRightPaint?.color =
            Color.parseColor(PROGRESS_COLOR)
        mProgressRightPaint?.textSize =
            dip2sp(PROGRESS_SIZE.toFloat()).toFloat()

        mHandler = Handler(context.mainLooper)
        mHandler?.post(mDotAnimRunnable)
        typedArray.recycle()
    }

    override fun onMeasure(widthMeasureSpec: Int, heightMeasureSpec: Int) {
        val wMode = MeasureSpec.getMode(widthMeasureSpec)
        var w = MeasureSpec.getSize(widthMeasureSpec)
        val hMode = MeasureSpec.getMode(heightMeasureSpec)
        var h = MeasureSpec.getSize(heightMeasureSpec)
        if (wMode == MeasureSpec.AT_MOST) {
            w = w.coerceAtMost(context.resources.displayMetrics.widthPixels)
        }
        if (hMode == MeasureSpec.AT_MOST) {
            h = mProgressHeight
        }
        setMeasuredDimension(w, h)
    }

    override fun dispatchDraw(canvas: Canvas) {
        if (mLeftTextPaint == null || mRightTextPaint == null || mProgressPaint == null
            || mProgressHintPaint == null || mProgressRightPaint == null
        ) {
            return
        }

        //画Loading
        val progressTitleX = (this.width - mLeftTextWidth - mRightTextWidth) / 2
        val progressTitleY = (this.height - mLeftTextHeight) / 2f
        canvas.drawText(mProgressTitle, progressTitleX, progressTitleY, mLeftTextPaint!!)

        //画...
        val progressRightTitleX = (this.width + mLeftTextWidth - mRightTextWidth / 2) / 2
        canvas.drawText(dotStr, progressRightTitleX, progressTitleY, mRightTextPaint!!)

        //画进度条
        val progressRightWidth = mProgressRightPaint!!.measureText("xx%")
        val progressLeft = (this.width - mProgressWidth - progressRightWidth) / 2
        val progressTop = this.height / 2f + mTitleProgressDefaultGap
        val progressRight = progressLeft + mCurrentProgress / 100 * mProgressWidth
        val progressBottom = this.height / 2f + mProgressHeight + mTitleProgressDefaultGap
        canvas.drawRect(
            progressLeft,
            progressTop,
            progressLeft + mProgressWidth,
            progressBottom,
            mProgressHintPaint!!
        )
        canvas.drawRect(progressLeft, progressTop, progressRight, progressBottom, mProgressPaint!!)

        //画进度条进度
        val progressRightY = this.height / 2f + mProgressHeight + mTitleProgressDefaultGap
        canvas.drawText(
            "${mCurrentProgress.toInt()}%",
            progressLeft + mProgressWidth + mProgressGap,
            progressRightY,
            mProgressRightPaint!!
        )
    }

    override fun onSizeChanged(w: Int, h: Int, oldW: Int, oldH: Int) {
        super.onSizeChanged(w, h, oldW, oldH)
        val screenWidth = context.resources.displayMetrics.widthPixels
        if (mProgressWidth >= screenWidth) {
            mCurrentMaxDecelerateDuration = MAX_DECELERATE_SPEED_DURATION
            mCurrentMaxUniformDuration = MAX_UNIFORM_SPEED_DURATION
        } else {
            //取比值
            val rate = mProgressWidth / screenWidth.toFloat()
            mCurrentMaxUniformDuration = (MAX_UNIFORM_SPEED_DURATION * rate).toInt()
            mCurrentMaxDecelerateDuration = (MAX_DECELERATE_SPEED_DURATION * rate).toInt()
        }
    }

    private fun setFinish() {
        isShow = false
        mProgressStatus = FINISH
    }

    private fun startAnim(isFinished: Boolean) {
        val v: Float = if (isFinished) 100f else 95.toFloat()
        if (mAnimator != null && mAnimator?.isStarted == true) {
            mAnimator?.cancel()
        }
        if (visibility == GONE) {
            visibility = VISIBLE
        }

        mCurrentProgress = if (mCurrentProgress == 0f) 0.00000001f else mCurrentProgress
        // 可能由于透明度造成突然出现的问题
        alpha = 1f
        if (!isFinished) {
            val valueAnimator = ValueAnimator.ofFloat(mCurrentProgress, v)
            val residue = 1f - mCurrentProgress / 100 - 0.05f
            valueAnimator.interpolator = LinearInterpolator()
            valueAnimator.duration = (residue * mCurrentMaxUniformDuration).toLong()
            valueAnimator.addUpdateListener(mAnimatorUpdateListener)
            valueAnimator.start()
            mAnimator = valueAnimator
        } else {
            var segment95Animator: ValueAnimator? = null
            if (mCurrentProgress < 95) {
                segment95Animator = ValueAnimator.ofFloat(mCurrentProgress, 95f)
                val residue = 1f - mCurrentProgress / 100f - 0.05f
                segment95Animator.interpolator = LinearInterpolator()
                segment95Animator.duration = (residue * mCurrentMaxDecelerateDuration).toLong()
                segment95Animator.interpolator = DecelerateInterpolator()
                segment95Animator.addUpdateListener(mAnimatorUpdateListener)
            }
            val mObjectAnimator = ObjectAnimator.ofFloat(this, "alpha", 1f, 0f)
            mObjectAnimator.duration = DO_END_ALPHA_DURATION.toLong()
            val mValueAnimatorEnd = ValueAnimator.ofFloat(95f, 100f)
            mValueAnimatorEnd.duration = DO_END_PROGRESS_DURATION.toLong()
            mValueAnimatorEnd.addUpdateListener(mAnimatorUpdateListener)
            var mAnimatorSet = AnimatorSet()
            mAnimatorSet.playTogether(mObjectAnimator, mValueAnimatorEnd)
            if (segment95Animator != null) {
                val mAnimatorSet1 = AnimatorSet()
                mAnimatorSet1.play(mAnimatorSet).after(segment95Animator)
                mAnimatorSet = mAnimatorSet1
            }
            mAnimatorSet.addListener(mAnimatorListenerAdapter)
            mAnimatorSet.start()
            mAnimator = mAnimatorSet
        }
        mProgressStatus = STARTED
    }

    private val mAnimatorUpdateListener = AnimatorUpdateListener { animation ->
        mCurrentProgress = animation.animatedValue as Float
        this@WebViewProgress.invalidate()
    }
    private val mAnimatorListenerAdapter: AnimatorListenerAdapter =
        object : AnimatorListenerAdapter() {
            override fun onAnimationEnd(animation: Animator) {
                doEnd()
            }
        }
    private val mDotAnimRunnable: Runnable = object : Runnable {
        override fun run() {
            mHandler!!.postDelayed(this, 500)
            dotStr =
                if (dotStr.isEmpty()) "." else if (dotStr.length == 1) ".." else if (dotStr.length == 2) "..." else ""
        }
    }

    override fun onDetachedFromWindow() {
        super.onDetachedFromWindow()
        //animator cause leak , if not cancel
        if (mAnimator != null && mAnimator!!.isStarted) {
            mAnimator!!.cancel()
            mAnimator = null
        }
        if (mHandler != null) {
            mHandler?.removeCallbacks(mDotAnimRunnable)
        }
    }

    private fun doEnd() {
        if (mProgressStatus == FINISH && mCurrentProgress == 100f) {
            visibility = GONE
            mCurrentProgress = 0f
            this.alpha = 1f
        }
        mProgressStatus = UN_START
    }

    fun setProgress(newProgress: Int) {
        setProgress(java.lang.Float.valueOf(newProgress.toFloat()))
    }

    private fun dip2px(dpValue: Float): Int {
        val scale = context.resources.displayMetrics.density
        return (dpValue * scale + 0.5f).toInt()
    }

    private fun dip2sp(dpVal: Float): Int {
        return TypedValue.applyDimension(
            TypedValue.COMPLEX_UNIT_DIP, dpVal,
            context.resources.displayMetrics
        ).toInt()
    }

    private fun setProgress(progress: Float) {
        // 解决 同时返回两个 100，产生两次进度条的问题；
        if (mProgressStatus == UN_START && progress == 100f) {
            visibility = GONE
            return
        }

        if (progress < 95) {
            return
        }
        if (mProgressStatus != FINISH) {
            startAnim(true)
        }
    }

    /**
     * 显示进度条
     */
    fun show() {
        isShow = true
        visibility = VISIBLE
        mCurrentProgress = 0f
        startAnim(false)
    }

    /**
     * 进度完成后消失
     */
    fun hide() {
        setWebViewProgress(100)
    }

    private fun setWebViewProgress(newProgress: Int) {
        if (newProgress in 0..94) {
            if (!isShow) {
                show()
            } else {
                setProgress(newProgress)
            }
        } else {
            setProgress(newProgress)
            setFinish()
        }
    }

    companion object {
        /**
         * 默认匀速动画最大的时长
         */
        private const val MAX_UNIFORM_SPEED_DURATION = 10 * 1000

        /**
         * 默认加速后减速动画最大时长
         */
        private const val MAX_DECELERATE_SPEED_DURATION = 450

        /**
         * 95f-100f时，透明度1f-0f时长
         */
        private const val DO_END_ALPHA_DURATION = 230

        /**
         * 95f - 100f动画时长
         */
        private const val DO_END_PROGRESS_DURATION = 200

        /**
         * 默认的高度(dp)
         */
        private const val WEB_PROGRESS_DEFAULT_HEIGHT = 3

        /**
         * 默认文字的大小
         */
        private const val PROGRESS_TEXT_SIZE = 18

        /**
         * 进度右边的字体大小
         */
        private const val PROGRESS_SIZE = 10

        /**
         * 进度条和进度之间的间隔
         */
        private const val PROGRESS_DEFAULT_GAP = 3

        /**
         * 进度条颜色默认
         */
        private const val WEB_PROGRESS_COLOR = "#2483D9"
        private const val WEB_DEFAULT_HINT_COLOR = "#E0E0E0"

        /**
         * title的颜色
         */
        private const val PROGRESS_TEXT_COLOR = "#191F2B"

        /**
         * 进度条右边字体大小
         */
        private const val PROGRESS_COLOR = "#A3A3A3"

        /**
         * 进度条默认的宽度
         */
        private const val PROGRESS_DEFAULT_WIDTH = 167
        private const val UN_START = 0
        private const val STARTED = 1
        private const val FINISH = 2
    }

    init {
        init(context, attrs)
    }
}