package com.cloud.baobabsland.worker

import android.annotation.SuppressLint
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.net.ConnectivityManager
import android.os.Handler
import android.os.HandlerThread
import android.os.Message
import android.os.Process
import android.text.TextUtils
import android.util.Log
import androidx.localbroadcastmanager.content.LocalBroadcastManager
import com.cloud.baobabsland.BuildConfig
import com.cloud.baobabsland.UnionSetting
import com.cloud.baobabsland.athena.AthenaTracker.trackByOfflineDataRes
import com.cloud.baobabsland.athena.AthenaTracker.trackByOfflineFeedReq
import com.cloud.baobabsland.athena.AthenaTracker.trackDataByConfigReq
import com.cloud.baobabsland.athena.AthenaTracker.trackDataByConfigResponse
import com.cloud.baobabsland.athena.AthenaTracker.trackInitGslb
import com.cloud.baobabsland.bean.DownImageInfo
import com.cloud.baobabsland.bean.PostBody
import com.cloud.baobabsland.bean.PostBody.getFeedListBody
import com.cloud.baobabsland.bean.response.*
import com.cloud.baobabsland.constant.Constants
import com.cloud.baobabsland.constant.Constants.CommonConstant
import com.cloud.baobabsland.constant.ErrorCode
import com.cloud.baobabsland.constant.ServerConfig
import com.cloud.baobabsland.constant.ServerConfig.h5Url
import com.cloud.baobabsland.constant.ServerConfig.ossUrl
import com.cloud.baobabsland.constant.ServerConfig.serverUrl
import com.cloud.baobabsland.download.ImageDownloadHelper
import com.cloud.baobabsland.download.OfflineAdManager
import com.cloud.baobabsland.http.ServerRequest
import com.cloud.baobabsland.http.listener.CommonResponseListener
import com.cloud.baobabsland.utils.*
import com.cloud.baobabsland.utils.PreferencesHelper.saveUpdateDay
import com.cloud.baobabsland.utils.gsonutil.GsonUtil
import com.cloud.baobabsland.worker.receiver.UnionBroadcastReceiver
import com.google.gson.Gson
import com.google.gson.reflect.TypeToken
import com.transsion.core.CoreUtil
import com.transsion.core.pool.TranssionPoolManager
import com.transsion.gslb.GslbSdk
import com.transsion.gslb.GslbSdk.InitListener
import com.transsion.http.HttpClient
import com.transsion.http.impl.StringCallback
import java.io.File
import java.util.*
import java.util.concurrent.CopyOnWriteArrayList
import java.util.concurrent.atomic.AtomicInteger
import kotlin.math.abs

class UnionWorker private constructor() : Handler.Callback {
    private var mContext: Context? = null

    /**
     * 获取媒体主的配置信息
     */
    var unionSetting: UnionSetting? = null
        private set

    @JvmField
    var unionCache: UnionCache?
    private val mHandler: Handler?
    private fun init(context: Context, setting: UnionSetting?) {
        mContext = context
        if (setting != null) {
            unionSetting = setting
            mHandler?.sendEmptyMessage(WHAT_INIT)
        }
    }

    /**
     * 初始化是否成功
     */
    val isInitSuccess: Boolean
        get() = mContext != null && unionSetting != null && !unionSetting!!.verifyIllegal()

    private object UnionWorkerInstance {
        @SuppressLint("StaticFieldLeak")
        val INSTANCE = UnionWorker()
    }

    override fun handleMessage(msg: Message): Boolean {
        try {
            if (unionCache == null) return false
            if (msg.what == WHAT_INIT) {
                AdLogUtil.Log().i(CommonLogUtil.TAG, "--> WHAT_INIT")
                if (isInitSuccess) {
                    registerReceiver()
                    initGslb()
                    //读取缓存的云控数据
                    unionCache?.readConfig()
                    unionCache?.readOfflineFeedList()
                    unionCache?.readOfflineFeedDetails()
                    unionCache?.readExperimentList()
                    unionCache?.registerReceiver()
                    checkEmpty(false)
                } else {
                    AdLogUtil.Log().i(CommonLogUtil.TAG, "--> WHAT_INIT is failure")
                }
            } else if (msg.what == WHAT_CHECK_EMPTY) {
                var isFromReceiver = false
                if (msg.obj is Boolean) {
                    isFromReceiver = msg.obj as Boolean
                }
                AdLogUtil.Log().i(
                    CommonLogUtil.TAG,
                    "--> WHAT_CHECK_EMPTY isFromReceiver = " + isFromReceiver + " , mLastUpdateTime = " + unionCache?.mLastUpdateTime
                )
                handleCloudConfigs(isFromReceiver)
            }
        } catch (e: Exception) {
            AdLogUtil.Log().e(Log.getStackTraceString(e))
        }
        return false
    }

    private fun handleCloudConfigs(isFromReceiver: Boolean) {
        if (CommonUtils.checkNetworkState()) {
            if (GslbSdk.isInitSuccess(serverUrl)) {
                if (unionCache?.isExpExpired == true) { //实验数据过期
                    unionCache?.queryCouldConfigs(unionSetting?.appId, isFromReceiver)
                } else {
                    AdLogUtil.Log().i(CommonLogUtil.TAG, "config data is not expired")
                }
            }
        } else {
            AdLogUtil.Log().e("error -> network error")
        }
    }

    private fun registerReceiver() {
        try {
            val filter = IntentFilter()
            filter.addAction(ConnectivityManager.CONNECTIVITY_ACTION)
            val receiver = UnionBroadcastReceiver()
            mContext?.registerReceiver(receiver, filter)
        } catch (e: Exception) {
            AdLogUtil.Log().e(Log.getStackTraceString(e))
        }
    }

    /**
     * 触发检查
     */
    fun checkEmpty(isFromReceiver: Boolean) {
        if (!isInitSuccess) return
        if (mHandler != null && !mHandler.hasMessages(WHAT_CHECK_EMPTY)) {
            val message = Message.obtain()
            message.obj = isFromReceiver
            message.what = WHAT_CHECK_EMPTY
            mHandler.sendMessageDelayed(message, 500)
        }
    }

    inner class UnionCache {
        private var mCacheConfigFilePath // 缓存Config文件
                : String? = null
        private var mCacheFeedListFilePath // 缓存FeedList文件
                : String? = null
        private var mCacheFeedDetailFilePath // 缓存FeedDetail文件
                : String? = null
        private var mCacheExperimentFilePath // 缓存实验文件
                : String? = null
        private var mInterval = 900 * 1000 // 两次请求间隔时间 默认900s
        private var maxExpiredTime = 7 * 24 * 60 * 60 * 1000 //缓存过期天数, 默认7天
        var mLastUpdateTime: Long = 0 // 上次请求时间

        /**
         * 获取缓存的config数据
         */
        var configDTO: ConfigDTO? = null
            private set

        /**
         * 获取缓存的feedList数据
         */
        var feedVOList: List<FeedVO>? = null
            private set

        /**
         * 缓存feedList的数据
         */
        var feedDetailList: MutableList<FeedDetailVO>? = null
            private set

        /**
         * 获取缓存的实验数据 。
         */
        var experimentList: List<AbTestHitExpVO?>? = null
            private set

        /**
         * 临时缓存数据
         */
        private var tempFeedDetailList: CopyOnWriteArrayList<FeedDetailVO>? = null

        /**
         * 实验数据是否过期
         */
        val isExpExpired: Boolean
            get() = mLastUpdateTime == 0L || abs(System.currentTimeMillis() - mLastUpdateTime) >= mInterval

        /**
         * 读取缓存的config数据
         */
        fun readConfig() {
            AdLogUtil.Log().i(CommonLogUtil.TAG, "--> readConfig()")
            if (TextUtils.isEmpty(mCacheConfigFilePath)) {
                mCacheConfigFilePath =
                    FileUtil.getAppFilePath(CoreUtil.getContext()) + File.separator + Constants.DownPath.ONLINE_FOLDER_NAME + File.separator + Companion.UNION_CONFIG_FILE_PATH + CommonUtils.getCurProcessName(
                        mContext
                    )
            }
            val buffer = CommonUtils.bufferRead(mCacheConfigFilePath?.let { File(it) })
            if (!TextUtils.isEmpty(buffer)) {
                try {
                    val decryptResult = AESUtils.decrypt(buffer)
                    configDTO = Gson().fromJson(decryptResult, ConfigDTO::class.java)
                    if (configDTO != null) {
                        mLastUpdateTime = configDTO?.lastUpdateTime ?: 0
                        if (configDTO?.basicJson != null) {
                            mInterval =
                                (configDTO?.basicJson?.requestIntervalSeconds
                                    ?: 900) * 1000
                            maxExpiredTime =
                                (configDTO?.basicJson?.cacheExpireDays
                                    ?: 7) * 24 * 60 * 60 * 1000
                        }
                    }
                    if (abs(mLastUpdateTime - System.currentTimeMillis()) >= maxExpiredTime) {
                        AdLogUtil.Log().i(CommonLogUtil.TAG, "readConfig() --> exceeded maxExpiredTime, local cache expires")
                        configDTO = null //数据缓存天数过期，需要把本地的缓存清空
                        mLastUpdateTime = 0
                        checkEmpty(false)
                    } else if (isExpExpired) {
                        checkEmpty(false)
                    }
                } catch (e: Exception) {
                    AdLogUtil.Log().e(Log.getStackTraceString(e))
                }
            } else {
                AdLogUtil.Log().i(CommonLogUtil.TAG, "readConfig data is null")
            }
        }

        /**
         * 读取缓存的feedList数据
         */
        fun readOfflineFeedList() {
            if (TextUtils.isEmpty(mCacheFeedListFilePath)) {
                mCacheFeedListFilePath =
                    (FileUtil.getAppFilePath(CoreUtil.getContext()) + File.separator + Constants.DownPath.OFFLINE_FOLDER_NAME + File.separator
                            + Companion.UNION_FEED_LIST_FILE_PATH + CommonUtils.getCurProcessName(
                        mContext
                    ))
            }
            feedVOList = WorkerHelper.readOfflineFeedList(mCacheFeedListFilePath)
        }

        /**
         * 读取缓存的feedDetail数据
         */
        fun readOfflineFeedDetails() {
            if (TextUtils.isEmpty(mCacheFeedDetailFilePath)) {
                mCacheFeedDetailFilePath =
                    (FileUtil.getAppFilePath(CoreUtil.getContext()) + File.separator + Constants.DownPath.OFFLINE_FOLDER_NAME + File.separator
                            + UNION_FEED_DETAIL_FILE_PATH + CommonUtils.getCurProcessName(
                        mContext
                    ))
            }
            feedDetailList = WorkerHelper.readOfflineFeedDetails(mCacheFeedDetailFilePath)
        }

        /**
         * 读取缓存的实验数据
         */
        fun readExperimentList() {
            if (TextUtils.isEmpty(mCacheExperimentFilePath)) {
                mCacheExperimentFilePath =
                    (FileUtil.getAppFilePath(CoreUtil.getContext()) + File.separator + Constants.DownPath.ONLINE_FOLDER_NAME + File.separator
                            + UNION_EXPERIMENT_FILE_PATH + CommonUtils.getCurProcessName(
                        mContext
                    ))
            }
            experimentList = WorkerHelper.readExperimentList(mCacheExperimentFilePath)
        }

        /**
         * config数据缓存
         */
        fun writeConfig() {
            if (TextUtils.isEmpty(mCacheConfigFilePath)) {
                mCacheConfigFilePath =
                    (FileUtil.getAppFilePath(CoreUtil.getContext()) + File.separator + Constants.DownPath.ONLINE_FOLDER_NAME + File.separator
                            + Companion.UNION_CONFIG_FILE_PATH + CommonUtils.getCurProcessName(
                        mContext
                    ))
            }
            WorkerHelper.writeConfig(mCacheConfigFilePath, configDTO)
        }

        /**
         * FeedList数据缓存
         */
        private fun writeFeedList() {
            if (TextUtils.isEmpty(mCacheFeedListFilePath)) {
                mCacheFeedListFilePath =
                    (FileUtil.getAppFilePath(CoreUtil.getContext()) + File.separator + Constants.DownPath.OFFLINE_FOLDER_NAME + File.separator
                            + Companion.UNION_FEED_LIST_FILE_PATH + CommonUtils.getCurProcessName(
                        mContext
                    ))
            }
            WorkerHelper.writeFeedList(mCacheFeedListFilePath, feedVOList)
        }

        /**
         * FeedDetail数据缓存
         */
        fun writeFeedDetailList() {
            if (TextUtils.isEmpty(mCacheFeedDetailFilePath)) {
                mCacheFeedDetailFilePath =
                    (FileUtil.getAppFilePath(CoreUtil.getContext()) + File.separator + Constants.DownPath.OFFLINE_FOLDER_NAME + File.separator
                            + Companion.UNION_FEED_DETAIL_FILE_PATH + CommonUtils.getCurProcessName(
                        mContext
                    ))
            }
            WorkerHelper.writeFeedDetailList(mCacheFeedDetailFilePath, feedDetailList)
        }

        /**
         * 实验数据缓存
         */
        fun writeExperimentList() {
            if (TextUtils.isEmpty(mCacheExperimentFilePath)) {
                mCacheExperimentFilePath =
                    (FileUtil.getAppFilePath(CoreUtil.getContext()) + File.separator + Constants.DownPath.ONLINE_FOLDER_NAME + File.separator
                            + Companion.UNION_EXPERIMENT_FILE_PATH + CommonUtils.getCurProcessName(
                        mContext
                    ))
            }
            WorkerHelper.writeExperimentList(mCacheExperimentFilePath, experimentList)
        }

        /**
         * 请求云控配置接口
         *
         * @param appId 媒体appId
         */
        fun queryCouldConfigs(appId: String?, isFromReceiver: Boolean) {
            AdLogUtil.Log().d(CommonLogUtil.TAG, "queryCouldConfigs appId = $appId")
            if (!CommonUtils.checkNetworkState()) {
                AdLogUtil.Log().e(CommonLogUtil.TAG, "requestCloudControl net is not available")
                return
            }
            try {
                if (TextUtils.isEmpty(CommonUtils.getVAID())) {
                    AdLogUtil.Log().e("queryCouldConfigs() -> vaid is null")
                }

                TranssionPoolManager.getInstance()
                    .addTask { getCloudConfigs(appId, isFromReceiver) }
            } catch (e: Exception) {
                AdLogUtil.Log().e(Log.getStackTraceString(e))
            }
        }

        /**
         * 云控请求
         */
        private fun getCloudConfigs(appId: String?, isFromReceiver: Boolean) {
            AdLogUtil.Log().i(
                CommonLogUtil.TAG,
                "request getCloudConfigs ---> isFromReceiver = $isFromReceiver"
            )
            trackDataByConfigReq(if (isFromReceiver) 2 else 1)
            getExperimentsByOneId()
            var configApi: String? = ServerConfig.sdkConfigUrl
            if (configDTO != null && configDTO?.basicJson != null && !TextUtils.isEmpty(
                    configDTO?.basicJson?.cloudConfigUrl
                )
            ) {
                configApi = configDTO?.basicJson?.cloudConfigUrl
            }
            val serverRequest = ServerRequest()
                .setListener(object : CommonResponseListener<ConfigResponse?>() {

                    override fun onRequestError(adError: ErrorCode) {
                        AdLogUtil.Log()
                            .e("requestConfig -> error code = ${adError.errorCode} , message = ${adError.errorMessage}")
                    }

                    override fun onRequestSuccess(statusCode: Int, response: ConfigResponse?) {
                        AdLogUtil.Log().d(
                            CommonLogUtil.TAG,
                            "requestConfig -> onRequestSuccess statusCode $statusCode"
                        )
                        if (response != null && response.code == Constants.ResponseCode.BUSINESS_SUCCESS_CODE) {
                            try {
                                mLastUpdateTime = System.currentTimeMillis()
                                val onlineConfigDTO = response.data
                                if (onlineConfigDTO != null) {
                                    var openCache = false //缓存的开关
                                    var offlineFeedNum = 0 //缓存的条数

                                    // 两次请求最小间隔900秒
                                    if (onlineConfigDTO.basicJson != null) {
                                        mInterval =
                                            (onlineConfigDTO.basicJson?.requestIntervalSeconds
                                                ?: 900) * 1000
                                        maxExpiredTime =
                                            (onlineConfigDTO.basicJson?.cacheExpireDays
                                                ?: 7) * 24 * 60 * 60 * 1000
                                        openCache =
                                            onlineConfigDTO.basicJson?.openCache ?: false
                                        offlineFeedNum =
                                            onlineConfigDTO.basicJson?.offlineFeedNum ?: 0
                                    } else {
                                        mInterval = 900 * 1000
                                        AdLogUtil.Log().e("interval parse error")
                                    }
                                    //埋点
                                    trackDataByConfigResponse(
                                        openCache,
                                        mInterval,
                                        maxExpiredTime,
                                        offlineFeedNum
                                    )
                                    onlineConfigDTO.lastUpdateTime = mLastUpdateTime
                                    try {
                                        onlineConfigDTO.vaid = CommonUtils.getVAID()

                                        if (unionSetting != null) {
                                            onlineConfigDTO.appId = unionSetting?.appId
                                        }
                                        onlineConfigDTO.pkgName = CoreUtil.getContext().packageName
                                        onlineConfigDTO.udid = CommonUtils.getUDID()
                                        onlineConfigDTO.udidType = PreferencesHelper.instance.getInt(Constants.KeyName.KEY_UDID_TYPE)
                                    } catch (e: Exception) {
                                        AdLogUtil.Log().e(Log.getStackTraceString(e))
                                    }
                                    onlineConfigDTO.phoneModel = CommonUtils.getModel()
                                    onlineConfigDTO.sdkVersion = BuildConfig.SDK_VERSION
                                    onlineConfigDTO.language = CommonUtils.getSystemLanguage()
                                    onlineConfigDTO.country = CommonUtils.getCountry()
                                    onlineConfigDTO.gaid = CommonUtils.getGAID()
                                    onlineConfigDTO.sdkCode = BuildConfig.SDK_CODE.toInt()

                                    handleConfigRes(onlineConfigDTO, appId, isFromReceiver)

                                    configDTO = onlineConfigDTO
                                    writeConfig()
                                } else {
                                    AdLogUtil.Log().i(
                                        CommonLogUtil.TAG,
                                        "requestConfig -> requestConfig get configDTO is null"
                                    )
                                }
                            } catch (e: Exception) {
                                AdLogUtil.Log().e(Log.getStackTraceString(e))
                            }
                        } else {
                            AdLogUtil.Log().i(
                                CommonLogUtil.TAG,
                                "requestConfig ->onRequestSuccess  response is null or response.code is not 0"
                            )
                        }
                    }
                })
                .setPostBody { PostBody.getConfigPostBody(appId) }
                .setDebug(unionSetting?.isDebug ?: false)
                .setUrl(configApi)
            AdLogUtil.Log().i(
                CommonLogUtil.TAG, "queryCloudConfig url = " + GslbSdk.getDomain(
                    serverUrl + configApi, true
                )
            )
            serverRequest?.netRequestPreExecute(null, null)
        }

        /** 处理返回的云控数据 */
        private fun handleConfigRes(
            onlineConfigDTO: ConfigDTO,
            appId: String?,
            isFromReceiver: Boolean
        ) {
            if (onlineConfigDTO.basicJson != null) {
                if (onlineConfigDTO.basicJson?.openCache == true) { //缓存开关--true
                    AdLogUtil.Log().i(
                        CommonLogUtil.TAG,
                        "cloudConfig openCache is true"
                    )
                    if (configDTO == null || configDTO?.basicJson != null
                        && (configDTO?.basicJson?.openCache == false || onlineConfigDTO.basicJson?.h5ZipVersion != configDTO?.basicJson?.h5ZipVersion)
                    ) { //没有缓存的情况或者zip版本不同
                        if (!TextUtils.isEmpty(onlineConfigDTO.basicJson?.h5FeedZipUrl)) {
                            AdLogUtil.Log().i(
                                CommonLogUtil.TAG,
                                "cloudConfig h5ZipVersion is different，and to downloadMaterials"
                            )
                            val zipUrl =
                                onlineConfigDTO.basicJson?.h5FeedZipUrl
                            ThreadUtils.runOnUiThread {  //下载zip包
                                OfflineAdManager.getInstance()
                                    .downloadMaterials(zipUrl)
                            }
                            queryOfflineFeeds(appId, isFromReceiver)
                        }
                    } else { //版本相同
                        AdLogUtil.Log().i(
                            CommonLogUtil.TAG,
                            "cloudConfig h5ZipVersion is same, isFromReceiver = $isFromReceiver"
                        )
                        handlerUpdateOffline(appId, isFromReceiver)
                    }
                } else { //缓存开关--false
                    val isDeleteSuccess = WorkerHelper.deleteOfflineData()
                    if (isDeleteSuccess) {
                        feedVOList = null
                        feedDetailList = null
                    }
                }
            }
        }

        /**
         * 请求离线feed列表接口
         *
         * @param appId 媒体appId
         */
        private fun queryOfflineFeeds(appId: String?, isFromReceiver: Boolean) {
            AdLogUtil.Log().d(CommonLogUtil.TAG, "queryCouldConfigs appId = $appId")
            if (!CommonUtils.checkNetworkState()) {
                AdLogUtil.Log().e(CommonLogUtil.TAG, "queryOfflineFeeds net is not available")
                return
            }
            TranssionPoolManager.getInstance().addTask { getOfflineFeeds(appId, isFromReceiver) }
        }

        /**
         * 离线feed列表请求
         */
        private fun getOfflineFeeds(appId: String?, isFromReceiver: Boolean) {
            trackByOfflineFeedReq(CommonConstant.CHANNEL_ID, "")
            val serverRequest = ServerRequest()
                .setListener(object : CommonResponseListener<FeedListResponse?>() {
                    override fun onRequestSuccess(
                        statusCode: Int,
                        response: FeedListResponse?
                    ) {
                        AdLogUtil.Log().d(
                            CommonLogUtil.TAG,
                            "getOfflineFeeds -> onRequestSuccess statusCode $statusCode"
                        )
                        if (response != null && response.code == Constants.ResponseCode.BUSINESS_SUCCESS_CODE) {
                            try {
                                val feedVOLists = response.data
                                feedVOList = WorkerHelper.handleSingleImageData(feedVOLists)
                                handleFeedRes(feedVOList, response, isFromReceiver)
                            } catch (e: Exception) {
                                AdLogUtil.Log().e(Log.getStackTraceString(e))
                            }
                        } else {
                            if (response == null) return
                            trackByOfflineDataRes(
                                CommonConstant.EVENT_OFFLINE_FEED_RESPONSE,
                                0,
                                0,
                                response.code ?: -1,
                                response.message,
                                ""
                            )
                        }
                    }

                    override fun onRequestError(adError: ErrorCode) {
                        trackByOfflineDataRes(
                            CommonConstant.EVENT_OFFLINE_FEED_RESPONSE,
                            0,
                            0,
                            adError.errorCode,
                            adError.errorMessage,
                            ""
                        )
                        AdLogUtil.Log()
                            .e("getOfflineFeeds ->  error code = " + adError.errorCode + " , message = " + adError.errorMessage)
                    }
                })
                .setPostBody { getFeedListBody(appId) }
                .setDebug(unionSetting?.isDebug ?: false)
                .setUrl(GslbSdk.getDomain(serverUrl + Constants.HOST.FEED_LIST_API, true))
            serverRequest?.netRequestPreExecute(null, null)
        }

        /**
         * 处理接口返回的feed列表数据
         */
        private fun handleFeedRes(
            feedVOList: List<FeedVO>?,
            response: FeedListResponse,
            isFromReceiver: Boolean
        ) {
            if (CollectionUtils.isNotEmpty(feedVOList)) {

                TranssionPoolManager.getInstance().addTask {
                    val deleteResult = WorkerHelper.deleteFeedImage()
                    if (deleteResult) {
                        AdLogUtil.Log().i(CommonLogUtil.TAG, "feed images delete success")
                    } else {
                        AdLogUtil.Log().i(CommonLogUtil.TAG, "feed images delete failure")
                    }

                    val deleteContentResult = WorkerHelper.deleteContentImage()
                    if (deleteContentResult) {
                        AdLogUtil.Log().i(CommonLogUtil.TAG, "content images delete success")
                    } else {
                        AdLogUtil.Log().i(CommonLogUtil.TAG, "content images delete failure")
                    }
                }

                trackByOfflineDataRes(
                    CommonConstant.EVENT_OFFLINE_FEED_RESPONSE,
                    feedVOList?.size ?: 0,
                    1,
                    response.code ?: -1,
                    response.message,
                    feedVOList?.get(0)?.recExpId
                )
                if (isFromReceiver) {
                    //记录广播来源更新缓存数据的日期，当天有次数限制
                    saveUpdateDay(
                        Constants.KeyName.KEY_CONFIG_DAILY,
                        Calendar.getInstance()[Calendar.DAY_OF_MONTH]
                    )
                }
                val feedImageLists = WorkerHelper.getFeedImageLists(feedVOList)
                ThreadUtils.runOnUiThreadDelayed(
                    { ImageDownloadHelper.downLoadImages(feedImageLists) },
                    500
                )
                //请求feed详情
                queryOfflineFeedDetails(WorkerHelper.getOfflineFeedCodes(feedVOList))
                writeFeedList()
            } else {
                trackByOfflineDataRes(
                    CommonConstant.EVENT_OFFLINE_FEED_RESPONSE,
                    0,
                    1,
                    response.code ?: -1,
                    response.message,
                    ""
                )
                AdLogUtil.Log().i(CommonLogUtil.TAG, "getOfflineFeeds ->  response list is empty")
            }
        }

        /**
         * 注册接收feed列表下载完成的广播通知
         */
        fun registerReceiver() {
            if (mContext != null) {
                val localBroadcastManager = LocalBroadcastManager.getInstance(mContext!!)
                val filter = IntentFilter(CommonConstant.RECEIVER_ACTION)
                val localReceiver = LocalReceiver()
                localBroadcastManager.registerReceiver(localReceiver, filter)
            }
        }

        private fun addOrderRequest(feedCodes: List<String?>?) {
            val feedCodesList: MutableList<String?> = ArrayList()
            if (CollectionUtils.isNotEmpty(feedCodes)) {
                feedCodesList.addAll(feedCodes!!)
            } else {
                return
            }
            var head: OrderRequest? = null
            var temp: OrderRequest? = null
            for (feedCode in feedCodesList) {
                if (head == null) {
                    head = OrderRequest(feedCode)
                    temp = head
                } else {
                    temp?.next = OrderRequest(feedCode)
                    temp = temp?.next
                }
            }
            head?.startDownload()
        }

        inner class OrderRequest(private val feedCode: String?) {
            var next: OrderRequest? = null
            private var imageCounter: AtomicInteger? = null
            fun startDownload() {
                if (TextUtils.isEmpty(feedCode)) {
                    downloadNext()
                    return
                }
                imageCounter = AtomicInteger(1)
                TranssionPoolManager.getInstance().addTask {
                    getOfflineFeedDetails(
                        feedCode, false
                    )
                }
            }

            fun calculateDownloadCount() {
                if (imageCounter != null && imageCounter!!.decrementAndGet() <= 0) {
                    downloadNext()
                }
            }

            private fun downloadNext() {
                next?.startDownload()
            }

            /**
             * 离线feed详情请求
             *
             * @param isRetry 是否是请求重试
             */
            private fun getOfflineFeedDetails(feedCode: String?, isRetry: Boolean) {
                trackByOfflineFeedReq(feedCode)
                if (!CollectionUtils.isNotEmpty(feedDetailList)) {
                    feedDetailList = mutableListOf()
                }
                if (!CollectionUtils.isNotEmpty(tempFeedDetailList)) {
                    tempFeedDetailList = CopyOnWriteArrayList()
                }
                var ossDomain: String? = ossUrl
                if (configDTO != null && configDTO?.basicJson != null && !TextUtils.isEmpty(
                        configDTO?.basicJson?.ossDomain
                    )
                ) {
                    ossDomain = configDTO?.basicJson?.ossDomain
                }
                val finalOssDomain = ossDomain
                HttpClient.get()
                    .log(unionSetting?.isDebug ?: false)
                    .connectTimeout(CommonConstant.TIMEOUT_IN_MILLIONS)
                    .readTimeout(CommonConstant.TIMEOUT_IN_MILLIONS)
                    .url(finalOssDomain + CommonConstant.FEED_DETAIL_PATH + feedCode + File.separator + feedCode + ".json")
                    .addHeader("Accept-Timezone", CommonUtils.getTimeZone())
                    .build()
                    .execute(object : StringCallback(true) {
                        override fun onSuccess(statusCode: Int, response: String) {
                            if (!TextUtils.isEmpty(response)) {
                                try {
                                    val feedDetailResponse =
                                        GsonUtil.fromJson(response, FeedDetailResponse::class.java)
                                    if (feedDetailResponse != null && feedDetailResponse.code == Constants.ResponseCode.BUSINESS_SUCCESS_CODE) {
                                        AdLogUtil.Log().i(
                                            CommonLogUtil.TAG,
                                            "-------------------getOfflineFeedDetails code = 0"
                                        )
                                        trackByOfflineDataRes(
                                            CommonConstant.EVENT_OFFLINE_DETAIL_RESPONSE,
                                            1,
                                            1,
                                            feedDetailResponse.code,
                                            feedDetailResponse.message,
                                            ""
                                        )
                                        val paragraphList =
                                            GsonUtil.fromJson<List<ParagraphJsonVO>>(
                                                feedDetailResponse.data,
                                                object : TypeToken<List<ParagraphJsonVO>>() {}.type
                                            )
                                        if (CollectionUtils.isNotEmpty(paragraphList)) {
                                            val feedDetailVO = FeedDetailVO()
                                            feedDetailVO.feedCode = feedCode
                                            feedDetailVO.paragraphJson = paragraphList
                                            feedDetailList?.add(feedDetailVO)
                                            writeFeedDetailList()
                                            tempFeedDetailList?.add(feedDetailVO)
                                        }
                                    } else {
                                        AdLogUtil.Log().i(
                                            CommonLogUtil.TAG,
                                            "-------------------getOfflineFeedDetails feedDetailResponse is null or code isn't 0"
                                        )
                                        downloadFeedDetailImage()
                                        if (!isRetry) {
                                            getOfflineFeedDetails(feedCode, true)
                                        }
                                        if (feedDetailResponse == null) return
                                        trackByOfflineDataRes(
                                            CommonConstant.EVENT_OFFLINE_DETAIL_RESPONSE,
                                            0,
                                            0,
                                            feedDetailResponse.code!!,
                                            feedDetailResponse.message,
                                            ""
                                        )
                                    }
                                } catch (e: Exception) {
                                    AdLogUtil.Log().e(Log.getStackTraceString(e))
                                } finally {
                                    calculateDownloadCount()
                                }
                            }
                        }

                        override fun onFailure(
                            statusCode: Int,
                            response: String,
                            throwable: Throwable
                        ) {
                            AdLogUtil.Log()
                                .e(" -------------------getOfflineFeedDetails  222 ->  error -> statusCode = $statusCode , response = $response , finalOssDomain = $finalOssDomain , feedCode = $feedCode")
                            downloadFeedDetailImage()
                            if (!isRetry) {
                                calculateDownloadCount()
                                getOfflineFeedDetails(feedCode, true)
                            }
                            trackByOfflineDataRes(
                                CommonConstant.EVENT_OFFLINE_DETAIL_RESPONSE,
                                0,
                                0,
                                statusCode,
                                response,
                                ""
                            )
                        }
                    }
                    )
            }
        }

        /**
         * 请求云控配置接口
         *
         * @param feedCodes 需要查询的feedCode集合
         */
        private fun queryOfflineFeedDetails(feedCodes: ArrayList<String>) {
            if (!CommonUtils.checkNetworkState() || feedCodes.isEmpty()) {
                AdLogUtil.Log().e(
                    CommonLogUtil.TAG,
                    "queryOfflineFeedDetails net is not available or feedCode size = 0"
                )
                return
            }
            addOrderRequest(feedCodes)
        }

        /**
         * 下载暂存的feed详情的图片
         */
        private fun downloadFeedDetailImage() {
            try {
                val copyTempFeedDetailList = tempFeedDetailList?.toMutableList()
                val detailImageLists: List<DownImageInfo?> =
                    WorkerHelper.getDetailImageLists(copyTempFeedDetailList)
                if (!CollectionUtils.isNotEmpty(detailImageLists)) {
                    return
                }
                ThreadUtils.runOnUiThread { ImageDownloadHelper.downLoadImages(detailImageLists) }
                tempFeedDetailList?.clear()
            } catch (e: Exception) {
                AdLogUtil.Log().e(Log.getStackTraceString(e))
            }

        }

        private inner class LocalReceiver : BroadcastReceiver() {
            override fun onReceive(context: Context, intent: Intent) {
                //请求feed列表结束，开始请求feed详情图片
                if (CommonConstant.RECEIVER_ACTION == intent.action) {
                    AdLogUtil.Log().i(CommonLogUtil.TAG, "--->  LocalReceiver")
                    downloadFeedDetailImage()
                }
            }
        }

        /**
         * 获取实验id
         */
        private fun getExperimentsByOneId() {
            val serverRequest = ServerRequest()
                .setListener(object : CommonResponseListener<ExperimentResponse?>() {
                    override fun onRequestSuccess(
                        statusCode: Int,
                        response: ExperimentResponse?
                    ) {
                        AdLogUtil.Log().d(
                            CommonLogUtil.TAG,
                            "getExperimentsByOneId -> onRequestSuccess statusCode $statusCode"
                        )
                        if (response != null && response.code == Constants.ResponseCode.BUSINESS_SUCCESS_CODE) {
                            try {
                                experimentList = response.data
                                if (CollectionUtils.isNotEmpty(experimentList)) {
                                    writeExperimentList()
                                } else {
                                    AdLogUtil.Log().i(
                                        CommonLogUtil.TAG,
                                        "getExperimentsByOneId ->  response list is empty"
                                    )
                                }
                            } catch (e: Exception) {
                                AdLogUtil.Log().e(Log.getStackTraceString(e))
                            }
                        } else {
                            AdLogUtil.Log().i(
                                CommonLogUtil.TAG,
                                "getExperimentsByOneId ->  code isn't success code"
                            )
                        }
                    }

                    override fun onRequestError(adError: ErrorCode) {
                        AdLogUtil.Log()
                            .e(" getExperimentsByOneId ->  error -> code = ${adError.errorCode} , message = ${adError.errorMessage}")
                    }
                })
                .setPostBody(PostBody::experimentBody)
                .setDebug(unionSetting?.isDebug ?: false)
                .setUrl(GslbSdk.getDomain(serverUrl + Constants.HOST.EXPERIMENT_API, true))
            serverRequest?.netRequestPreExecute(null, null)
        }

        /** 更新缓存数据的处理 */
        private fun handlerUpdateOffline(appId: String?, isFromReceiver: Boolean) {
            if (isFromReceiver) { //网络发生变化的处理
                queryOfflineFeeds(appId, isFromReceiver)
            } else {
                AdLogUtil.Log().i(
                    CommonLogUtil.TAG,
                    "cloudConfig isFromReceiver is false, version same don't update offline data"
                )
            }
        }
    }

    /**
     * 初始化Gslb
     */
    private fun initGslb() {
        AdLogUtil.Log().i(CommonLogUtil.TAG, "--> initGslb()")
        GslbSdk.init(mContext, arrayOf(serverUrl, h5Url), object : InitListener {
            override fun onInitSuccess(map: Map<String, String>) {
                checkEmpty(false) // 请求域名成功回调
                trackInitGslb(map.toString())
                AdLogUtil.Log().i(
                    CommonLogUtil.TAG, "initGslb() -> url = "
                            + serverUrl + " , after api url = " + GslbSdk.getDomain(
                        serverUrl,
                        true
                    ) + " ,h5 url = " + GslbSdk.getDomain(
                        h5Url, true
                    )
                )
            }

            override fun onInitFail() {
                AdLogUtil.Log().e("initGslb() -> onInitFail ")
            }
        })
    }

    companion object {
        private const val WHAT_INIT = 100
        private const val WHAT_CHECK_EMPTY = 101

        private const val UNION_CONFIG_FILE_PATH = "union_config_"
        private const val UNION_FEED_LIST_FILE_PATH = "union_feed_list_"
        private const val UNION_FEED_DETAIL_FILE_PATH = "union_feed_detail_"
        private const val UNION_EXPERIMENT_FILE_PATH = "union_experiment_"

        @JvmStatic
        fun getInstance(context: Context, setting: UnionSetting?): UnionWorker {
            val unionWorker = UnionWorkerInstance.INSTANCE
            unionWorker.init(context, setting)
            return unionWorker
        }
    }

    init {
        unionCache = UnionCache()
        val handlerThread = HandlerThread("union worker")
        handlerThread.priority = Process.THREAD_PRIORITY_BACKGROUND
        handlerThread.start()
        mHandler = Handler(handlerThread.looper, this)
    }
}