/*
 * Copyright 2025 Google Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the
 * License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.firebase.ui.auth.mfa

import com.firebase.ui.auth.configuration.MfaFactor
import com.firebase.ui.auth.configuration.string_provider.AuthUIStringProvider

/**
 * Represents the different steps in the Multi-Factor Authentication (MFA) enrollment flow.
 *
 * This enum defines the sequence of UI states that users progress through when enrolling
 * in MFA, from selecting a factor to completing the setup with recovery codes.
 *
 * @since 10.0.0
 */
enum class MfaEnrollmentStep {
    /**
     * The user is presented with a selection of available MFA factors to enroll in.
     * The available factors are determined by the [com.firebase.ui.auth.configuration.MfaConfiguration].
     */
    SelectFactor,

    /**
     * The user is configuring SMS-based MFA by entering their phone number.
     * This step prepares to send an SMS verification code to the provided number.
     */
    ConfigureSms,

    /**
     * The user is configuring TOTP (Time-based One-Time Password) MFA.
     * This step presents the TOTP secret (as both text and QR code) for the user
     * to scan into their authenticator app.
     */
    ConfigureTotp,

    /**
     * The user is verifying their chosen MFA factor by entering a verification code.
     * For SMS, this is the code received via text message.
     * For TOTP, this is the code generated by their authenticator app.
     */
    VerifyFactor,

    /**
     * The enrollment is complete and recovery codes are displayed to the user.
     * These backup codes can be used to sign in if the primary MFA method is unavailable.
     * This step only appears if recovery codes are enabled in the configuration.
     */
    ShowRecoveryCodes
}

/**
 * Returns the localized title text for this enrollment step.
 *
 * @param stringProvider The string provider for localized strings
 * @return The localized title for this step
 */
fun MfaEnrollmentStep.getTitle(stringProvider: AuthUIStringProvider): String = when (this) {
    MfaEnrollmentStep.SelectFactor -> stringProvider.mfaStepSelectFactorTitle
    MfaEnrollmentStep.ConfigureSms -> stringProvider.mfaStepConfigureSmsTitle
    MfaEnrollmentStep.ConfigureTotp -> stringProvider.mfaStepConfigureTotpTitle
    MfaEnrollmentStep.VerifyFactor -> stringProvider.mfaStepVerifyFactorTitle
    MfaEnrollmentStep.ShowRecoveryCodes -> stringProvider.mfaStepShowRecoveryCodesTitle
}

/**
 * Returns localized helper text providing instructions for this step.
 *
 * @param stringProvider The string provider for localized strings
 * @param selectedFactor The MFA factor being configured or verified. Used for [MfaEnrollmentStep.VerifyFactor]
 *                       to provide factor-specific instructions. Ignored for other steps.
 * @return Localized instructional text appropriate for this step
 */
fun MfaEnrollmentStep.getHelperText(
    stringProvider: AuthUIStringProvider,
    selectedFactor: MfaFactor? = null
): String = when (this) {
    MfaEnrollmentStep.SelectFactor -> stringProvider.mfaStepSelectFactorHelper
    MfaEnrollmentStep.ConfigureSms -> stringProvider.mfaStepConfigureSmsHelper
    MfaEnrollmentStep.ConfigureTotp -> stringProvider.mfaStepConfigureTotpHelper
    MfaEnrollmentStep.VerifyFactor -> when (selectedFactor) {
        MfaFactor.Sms -> stringProvider.mfaStepVerifyFactorSmsHelper
        MfaFactor.Totp -> stringProvider.mfaStepVerifyFactorTotpHelper
        null -> stringProvider.mfaStepVerifyFactorGenericHelper
    }
    MfaEnrollmentStep.ShowRecoveryCodes -> stringProvider.mfaStepShowRecoveryCodesHelper
}
