/*
 * Copyright 2025 Google Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the
 * License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.firebase.ui.auth.mfa

import android.content.Intent
import android.net.Uri
import com.google.firebase.auth.TotpSecret as FirebaseTotpSecret

/**
 * Wrapper class for Firebase TOTP secret that provides additional utility methods
 * for enrollment and integration with authenticator apps.
 *
 * This class encapsulates the Firebase [FirebaseTotpSecret] and provides methods to:
 * - Access the shared secret key
 * - Generate QR code URLs for easy scanning
 * - Open authenticator apps for automatic configuration
 * - Generate hashing algorithm and code generation parameters
 *
 * @property firebaseTotpSecret The underlying Firebase TOTP secret
 *
 * @since 10.0.0
 */
class TotpSecret internal constructor(
    private val firebaseTotpSecret: FirebaseTotpSecret
) {
    /**
     * The shared secret key that should be entered into an authenticator app.
     * This is a base32-encoded string that can be manually typed if QR scanning is not available.
     */
    val sharedSecretKey: String
        get() = firebaseTotpSecret.sharedSecretKey

    /**
     * Generates a Google Authenticator-compatible URI that can be encoded as a QR code
     * or used to automatically configure an authenticator app.
     *
     * The generated URI follows the format:
     * `otpauth://totp/{accountName}?secret={secret}&issuer={issuer}&algorithm={algorithm}&digits={digits}&period={period}`
     *
     * @param accountName The account identifier, typically the user's email address
     * @param issuer The name of your application or service
     * @return A URI string that can be converted to a QR code or used as a deep link
     *
     * @see openInOtpApp
     */
    fun generateQrCodeUrl(accountName: String, issuer: String): String {
        return firebaseTotpSecret.generateQrCodeUrl(accountName, issuer)
    }

    /**
     * Attempts to open the device's default authenticator app with the TOTP configuration.
     *
     * This method creates an Intent with the provided QR code URL and attempts to open
     * an authenticator app (such as Google Authenticator) that can handle the
     * `otpauth://` URI scheme. If successful, the app will be pre-configured with the
     * TOTP secret without requiring the user to manually scan a QR code.
     *
     * **Note:** This method may fail silently if no compatible authenticator app is installed
     * or if the app doesn't support automatic configuration via URI.
     *
     * @param qrCodeUrl The OTP auth URL generated by [generateQrCodeUrl]
     *
     * @see generateQrCodeUrl
     */
    fun openInOtpApp(qrCodeUrl: String) {
        firebaseTotpSecret.openInOtpApp(qrCodeUrl)
    }

    /**
     * Gets the underlying Firebase TOTP secret for use in enrollment operations.
     *
     * This method is primarily used internally by the enrollment handler to complete
     * the TOTP enrollment with Firebase Authentication.
     *
     * @return The underlying [FirebaseTotpSecret] instance
     */
    internal fun getFirebaseTotpSecret(): FirebaseTotpSecret {
        return firebaseTotpSecret
    }

    companion object {
        /**
         * Creates a [TotpSecret] instance from a Firebase TOTP secret.
         *
         * @param firebaseTotpSecret The Firebase TOTP secret to wrap
         * @return A new [TotpSecret] instance
         */
        internal fun from(firebaseTotpSecret: FirebaseTotpSecret): TotpSecret {
            return TotpSecret(firebaseTotpSecret)
        }
    }
}
